#!/usr/bin/python
# -*- coding: utf-8 -*-
"""
The arts_types module includes support for various ARTS-classes.

Due to the dynamic nature of Python, some are implemented in a more
generic way. For example, ArrayOf can be easily subclassed to be an array
of anything, and all gridded-fields are subclasses of GriddedField.

Classes of special interest may be:

- LatLonGriddedField3: Special case of GriddedField3 for lat/lon/pressure
  data
- AbsSpecies
- SingleScatteringData
- ScatteringMetaData

This module allows the generation, manipulation, and input/output in ARTS
XML format of these objects.
"""

from __future__ import division

import copy
import tempfile
import os
import string
import datetime
import contextlib
import gzip
import functools
import numbers

import numpy
import scipy
import scipy.ndimage.interpolation

from cStringIO import StringIO
from UserList import UserList

from . import artsXML
#from . import arts_scat
import arts_scat
from . import sli
from . import io
from . import physics
from . import general
from . import arts_math

from .general import PyARTSError, quotify
from .physics import k as BOLTZMANN_CONST
from .constants import (PARTICLE_TYPE_GENERAL, PARTICLE_TYPE_MACROS_ISO,
                        PARTICLE_TYPE_HORIZ_AL, PARTICLE_TYPE_SPHERICAL)

class BadFieldDimError(PyARTSError, ValueError): pass

class ArtsType(object):
    def save(self, f, compressed=None):
        """Writes data to file
        """

        c = self.to_xml()

        if isinstance(f, basestring):
            if compressed is None:
                compressed = f.lower().endswith(".gz")
            if compressed:
                writer = gzip.GzipFile(f, "w")
            else:
                writer = file(f, "w")
        else:
            writer = f
        with artsXML.XMLfile(writer, header=True) as out:
            out.write(c)

class ArrayOf(ArtsType, list):
    """Represents an array of something.

    The type is taken from the 'contains' attribute, to be defined by
    subclasses.
    """

    contains = None

    def to_xml(self):
        xml_obj = artsXML.XML_Obj(tag="Array",
                                  attributes = {"type": getattr(self.contains, "name", self.contains.__name__),
                                                "nelem": len(self)})
        for elem in self:
            xml_obj.write(elem.to_xml())
        return xml_obj.finalise().str

    def append(self, obj):
        if not isinstance(obj, self.contains):
            raise TypeError("Wrong type in array. Expected: %s. Got: %s" %
                            (self.contains, type(obj)))
        super(ArrayOf, self).append(obj)

    def __repr__(self):
        return ("<" + self.__class__.__name__ + "\n["
                + ",\n ".join(repr(elem) for elem in self) + "]" + ">")

    def __getattr__(self, attr):
        L = []
        for elem in self:
            L.append(getattr(elem, attr))
        return L

    @classmethod
    def load(cls, f):
        """Loads ArrayOfSomething

        - f: file to read it from

        Returns ArrayOfSomething
        """
        filedata = artsXML.load(f)
        obj = cls()

        tp = cls.contains
        tpname = cls.contains.__name__
        for artsXML_object in filedata:
            elem = tp._load_from_artsXML_object(artsXML_object[tpname])
            obj.append(elem)
        return obj


class GriddedField(ArtsType):
    """Gridded field of any dimension

    Arguments 1..n-1 contain the axes. The last argument contains the
    data, with must have n-1 dimensions, where n is the number of
    dimensions.
    """

    def __init__(self, *args):
        data = args[-1]
        axes = list(args[:-1])
        # convert all axes to ndarrays
        for (i, ax) in enumerate(axes):
            if not isinstance(ax, numpy.ndarray):
                axes[i] = numpy.array(ax)
        # check inputs. LBYL because errors may not cause exceptions soon
        # enough
        if not data.ndim == len(axes):
            raise BadFieldDimError(
                "Invalid axes dimensions: data-dimensionality must match "\
                "number of arguments-1. Dimensionality: %d. Number of " \
                "arguments-1: %d" % (data.ndim, len(axes)))
        for ax in axes:
            if ax.ndim != 1:
                raise BadFieldDimError(
                    "Invalid dimension for axis. Expected: 1. " \
                    "Found: %d." % ax.ndim)
        if not data.shape == tuple(ax.size for ax in axes):
            raise BadFieldDimError(
                "Invalid axes lengths. Expected: %s. " \
                "found: %s." % (str(data.shape),
                                str(tuple(ax.size for ax in axes))))
        self.axes = axes
        self.data = data

    def get_grid(self, i):
        """Return grid i
        """
        return self.axes[i]

    # need to have 'i' at the end, partial function applications doesn't
    # go well with keyword arguments that do not come at the eni
    def set_grid(self, val, i):
        """Set grid i to val
        """
        self.axes[i] = val

    @staticmethod
    def _get_prop(i):
        """Return property to get/set grid i
        """
        return property(functools.partial(GriddedField.get_grid, i=i),
                        functools.partial(GriddedField.set_grid, i=i))

    def interpolate(self, *other_axes):
        """Interpolate GF on new axes.
        """

        coords = []
        for my, other in zip(self.axes, other_axes):
            coords.append(numpy.interp(other, my, numpy.arange(my.size)))
        gridpoints = arts_math.combine_axes(*coords)
        #data_f = self.data.ravel()
        #newax_f = arts_math.combine_axes(*other_axes)
        #newdata_f = scipy.interpolate.griddata(ax_f, data_f, newax_f)
        newdata_f = scipy.ndimage.interpolation.map_coordinates(self.data, gridpoints.T, order=1)
        newdata = newdata_f.reshape([ax.size for ax in other_axes])
        return self.__class__(*(other_axes+(newdata,)))

    def writeXML(self, xmlfile):
        """Write to XMLfile-object
        """

        xmlfile.write(self.to_xml())

    def to_xml(self):
        """To XML, returns string (no header)
        """
        s = StringIO()
        xmlfile = artsXML.XMLfile(s, header=False)

        xmlfile.write("<GriddedField%d>\n" % self.data.ndim)
        for i, ax in enumerate(self.axes):
            if ax.dtype.kind in 'SU':
                # interpret as array of strings
                xmlfile.add(ax.tolist())
            else:
                xmlfile.add(ax)
        xmlfile.add(self.data)
        xmlfile.write("</GriddedField%d>\n" % self.data.ndim)
        xmlfile.close(really=False)
        return s.getvalue()

    @classmethod
    def load(cls,filename):
        """load a GriddedField object from an ARTS XML file"""
        M = artsXML.load(filename)
        return cls._load_from_artsXML_object(M)

    @classmethod
    def _load_from_artsXML_object(cls, M):
        data = M[M.keys()[-1]]
        axes = []
        for k in M.keys()[:-1]:
            if k.startswith("Array"): # assume ArrayOfString
                axes.append([s["String"] for s in M[k]])
            else:
                axes.append(M[k])
        return cls(*(axes + [data]))

    def __eq__(self, other):
        if isinstance(other, GriddedField) and  \
            (self.data.shape == other.data.shape) and \
            (self.data == other.data).all():
            # compare axes
            for i in xrange(len(self.axes)):
                if not (self.axes[i] == other.axes[i]).all():
                    return False
            return True
        else:
            return False

class GriddedField1(GriddedField):
    pass

class BaseGriddedField3(GriddedField):
    pass # nothing special

class LatLonGriddedField3(BaseGriddedField3):
    """A gridded field consists of a pressure grid vector, a latitude vector,
    a longitude vector and a Tensor3 for the data itself.
    """
    name = "GriddedField3" # needed for to_xml to work properly

    p_grid = GriddedField._get_prop(0)
    lat_grid = GriddedField._get_prop(1)
    lon_grid = GriddedField._get_prop(2)

    def __init__(self,p_grid,lat_grid,lon_grid,data):
        """GriddedField3 objects are initialised with a pressure grid vector,
        a latitude vector, a longitude vector and a Tensor3 for the data itself"""
        super(LatLonGriddedField3, self).__init__(p_grid, lat_grid, lon_grid, data)
        self.p_grid, self.lat_grid, self.lon_grid = self.axes
        self.data=data

    def expandTo3D(self,new_lat_grid,new_lon_grid):
        """converts the existing 1D field to 3D and returns a new field.
        The original field is not changed"""
        if not len(self.lat_grid)*len(self.lon_grid)==1:
            raise BadFieldDimError,"expandTo3D works only for 1D fields"
        data=numpy.zeros([len(self.p_grid),len(new_lat_grid),len(new_lon_grid)],
                   float)
        for ilat in range(len(new_lat_grid)):
            for ilon in range(len(new_lon_grid)):
                data[:,ilat,ilon]=numpy.squeeze(self.data)
        return GriddedField3(**{"p_grid":self.p_grid,
                                "lat_grid":new_lat_grid,
                                "lon_grid":new_lon_grid,
                                "data":data})        

    @classmethod
    def load(cls,filename):
        """load a GriddedField3 object from an ARTS XML file"""
        datastruct=artsXML.load(filename)
        p_grid=datastruct['Vector']
        lat_grid=datastruct['Vector 0']
        lon_grid=datastruct['Vector 1']
        data=datastruct['Tensor3']
        return cls(p_grid, lat_grid, lon_grid, data)

    def save(self,filename):
        """Save the GriddedField3 object to an ARTS XML file"""
        outfile=artsXML.XMLfile(filename)
        outfile.write('<GriddedField3>\n')
        outfile.add(self.p_grid)
        outfile.add(self.lat_grid)
        outfile.add(self.lon_grid)
        outfile.add(self.data)
        outfile.write('</GriddedField3>\n')
        outfile.close()

    def pad(self,plims=[1100e2,0.00001],latlims=[-90,90],lonlims=[-180,180]):
        """Adds extra gridpoints at new extremities in all dimensions. data values
        are copied from the existing end points"""
        p_grid=numpy.zeros(len(self.p_grid)+2,float)
        lat_grid=numpy.zeros(len(self.lat_grid)+2,float)
        lon_grid=numpy.zeros(len(self.lon_grid)+2,float)
        p_grid[0]=plims[0]
        p_grid[1:len(p_grid)-1]=self.p_grid
        p_grid[-1]=plims[1]
        lat_grid[0]=latlims[0]
        lat_grid[1:len(lat_grid)-1]=self.lat_grid
        lat_grid[-1]=latlims[1]
        lon_grid[0]=lonlims[0]
        lon_grid[1:len(lon_grid)-1]=self.lon_grid
        lon_grid[-1]=lonlims[1]
        new_shape=[len(p_grid),len(lat_grid),len(lon_grid)]
        data=numpy.zeros(new_shape,float)
        data[1:len(p_grid)-1,1:len(lat_grid)-1,1:len(lon_grid)-1]=self.data
        data[0,:,:]=data[1,:,:]
        data[-1,:,:]=data[-2,:,:]
        data[:,-1,:]=data[:,-2,:]
        data[:,0,:]=data[:,1,:]
        data[:,:,0]=data[:,:,1]
        data[:,:,-1]=data[:,:,-2]
        self.p_grid=p_grid
        self.lat_grid=lat_grid
        self.lon_grid=lon_grid
        self.data=data
        return self

    def __call__(self,p_grid,lat_grid,lon_grid):
        """interpolate the field onto new grids, returns only the field data"""
        np=len(p_grid)
        nlat=len(lat_grid)
        nlon=len(lon_grid)
        a=numpy.zeros((2,np,2,nlat,2,nlon),float)

        old_lnp_grid=log(self.p_grid)
        new_lnp_grid=log(p_grid)

        #calculate interpolation weights and indices
        #pressure
        p_index=[]
        for ip in range(np):
            lnp=new_lnp_grid[ip]
            p_index.append(arts_math.locate(old_lnp_grid,lnp))
            if p_index[-1] in (-1,len(self.p_grid)-1):
                errstr="Pressure "+str(p_grid[ip])+ " is outside original grid"
                raise ValueError,errstr
        pw_lo=1-(new_lnp_grid-take(old_lnp_grid,array(p_index)))/(take(old_lnp_grid,array(p_index)+1)-\
                                                                  take(old_lnp_grid,array(p_index)))
        pw=swapaxes(c_[nlon*[c_[nlat*[pw_lo]]]],0,2)
        assert(alltrue(pw_lo<=1)),str([p_grid,self.p_grid,p_index,pw_lo])
        assert(alltrue(pw_lo>=0)),str([p_grid,self.p_grid,p_index,pw_lo])

        #latitude        
        lat_index=[]
        for ilat in range(nlat):
            lat=lat_grid[ilat]
            lat_index.append(arts_math.locate(self.lat_grid,lat))
            if lat_index[-1] in (-1,len(self.lat_grid)-1):
                errstr="Latitude "+str(lat)+ " is outside original grid"
                raise ValueError,errstr
        latw_lo=1-(lat_grid-take(self.lat_grid,array(lat_index)))/(take(self.lat_grid,array(lat_index)+1)-\
                                                                   take(self.lat_grid,array(lat_index)))
        latw=swapaxes(c_[np*[c_[nlon*[latw_lo]]]],1,2)

        #longitude
        lon_index=[]
        for ilon in range(nlon):
            lon=lon_grid[ilon]
            lon_index.append(arts_math.locate(self.lon_grid,lon))
            if lon_index[-1] in (-1,len(self.lon_grid)-1):
                #we may have a grid with negative longitudes
                if any(self.lon_grid<0):
                    lon_index[-1]=arts_math.locate(self.lon_grid,lon-360)
                else:
                    errstr="Longitude %f is outside original grid (range %f to %f" % (lon,self.lon_grid[0],self.lon_grid[-1])
                    raise ValueError,errstr
        lonw_lo=1-(lon_grid-take(self.lon_grid,array(lon_index)))/(take(self.lon_grid,array(lon_index)+1)-\
                                                                   take(self.lon_grid,array(lon_index)))
        lonw=c_[np*[c_[nlat*[lonw_lo]]]]


        #now do the magic
        result=pw*latw*lonw*take(take(take(self.data,array(p_index),0),
                                      array(lat_index),1),array(lon_index),2)+\
              pw*latw*(1-lonw)*take(take(take(self.data,array(p_index),0),
                                         array(lat_index),1),array(lon_index)+1,2)+\
              pw*(1-latw)*lonw*take(take(take(self.data,array(p_index),0),
                                         array(lat_index)+1,1),array(lon_index),2)+\
              pw*(1-latw)*(1-lonw)*take(take(take(self.data,array(p_index),0),
                                             array(lat_index)+1,1),array(lon_index)+1,2)+\
              (1-pw)*latw*lonw*take(take(take(self.data,array(p_index)+1,0),
                                         array(lat_index),1),array(lon_index),2)+\
              (1-pw)*latw*(1-lonw)*take(take(take(self.data,array(p_index)+1,0),
                                             array(lat_index),1),array(lon_index)+1,2)+\
              (1-pw)*(1-latw)*lonw*take(take(take(self.data,array(p_index)+1,0),
                                             array(lat_index)+1,1),array(lon_index),2)+\
              (1-pw)*(1-latw)*(1-lonw)*take(take(take(self.data,array(p_index)+1,0),
                                                 array(lat_index)+1,1),array(lon_index)+1,2)

        return result

    def subset(self,p_range=None,lat_range=None,lon_range=None):
        """Takes the smallest subset of the scenario that includes the grid ranges
        specified by p_range=(pbottom,ptop), lat_range=(latmin,latmax),
        lon_range=(lon_min,lon_max)."""

        new_grid={}
        grid_range={'p':p_range,'lat':lat_range,'lon':lon_range}
        i_range={'p':(0,len(self.p_grid)),'lat':(0,len(self.lat_grid)),
                 'lon':(0,len(self.lon_grid))}
        if grid_range['p'] is not None:
            i_range['p']=(arts_math.locate(xa=self.p_grid,x=grid_range['p'][0]),
                          arts_math.locate(xa=self.p_grid,x=grid_range['p'][1])+1)
            new_grid['p']=self.p_grid[i_range['p'][0]:i_range['p'][1]]
        else:
            new_grid['p']=self.p_grid

        if grid_range["lat"] is not None:
            i_range['lat']=(arts_math.locate(xa=self.lat_grid,x=grid_range['lat'][0]),
                            arts_math.locate(xa=self.lat_grid,x=grid_range['lat'][1])+1)
            new_grid['lat']=self.lat_grid[i_range['lat'][0]:i_range['lat'][1]]
        else:
            new_grid['lat']=self.lat_grid

        ##Need to deal with cases where lon range spans 0 and the original grid is 0 - 360

        if prod(lon_range)<0 and self.lon_grid[-1]-self.lon_grid[0]>350:
            i_range['lon']=(arts_math.locate(xa=self.lon_grid,x=remainder(grid_range['lon'][0],360)),
                            arts_math.locate(xa=self.lon_grid,x=grid_range['lon'][1])+1)
            lon_indices=concatenate((arange(i_range['lon'][0],len(self.lon_grid)),
                                     arange(i_range['lon'][1])))
            new_grid['lon']=concatenate((self.lon_grid[i_range['lon'][0]:]-360,
                                         self.lon_grid[:i_range['lon'][1]]))

        else:
            i_range['lon']=(arts_math.locate(xa=self.lon_grid,x=grid_range['lon'][0]),
                            arts_math.locate(xa=self.lon_grid,x=grid_range['lon'][1])+1)
            lon_indices=arange(i_range['lon'][0],i_range['lon'][1])
            new_grid['lon']=self.lon_grid[i_range['lon'][0]:i_range['lon'][1]]

        #do grids

        new_data=take(self.data[i_range['p'][0]:i_range['p'][1],
                                i_range['lat'][0]:i_range['lat'][1],:],
                      lon_indices,2)
        return GriddedField3(p_grid=new_grid['p'],lat_grid=new_grid['lat'],lon_grid=new_grid['lon'],
                             data=new_data)

    def get_extent(self, coordinate):
        """Returns the extent (min, max) of non-zero data in the given coordinate.

        Coordinate can be 'p', 'lat', 'lon'.
        """

        raise NotImplementedError()

GriddedField3 = LatLonGriddedField3

class ArrayOfGriddedField3(ArrayOf):
    contains = GriddedField3

class ArrayOfGriddedField1(ArrayOf):
    contains = GriddedField1

class ArrayOfLatLonGriddedField3(ArrayOf):
    contains = LatLonGriddedField3

#class ArrayOfGriddedField3(ArtsType):
#    def __init__(self,list_of_gfields=[]):
#        self.data=copy.deepcopy(list_of_gfields)
#    def load(self,filename):
#        raw_data=artsXML.load(filename)
#        self.data=[]
#        for gfield in raw_data:
#            self.data.append(GriddedField3(**{"p_grid":gfield['GriddedField3']['Vector'],
#                                              "lat_grid":gfield['GriddedField3']['Vector 0'],
#                                              "lon_grid":gfield['GriddedField3']['Vector 1'],
#                                              "data":gfield['GriddedField3']['Tensor3']})
#                             )
#        return self
#    def save(self,filename):
#        outfile=artsXML.XMLfile(filename)
#        outfile.write('<Array type=\"GriddedField3\" nelem=\"'+str(len(self.data))+'\">\n')
#        for gfield in self.data:
#            outfile.write('<GriddedField3>\n')
#            outfile.add(gfield.p_grid)
#            outfile.add(gfield.lat_grid)
#            outfile.add(gfield.lon_grid)
#            outfile.add(gfield.data)
#            outfile.write('</GriddedField3>\n')
#        outfile.write('</Array>\n')
#        outfile.close()
#    def append(self,gfield):
#        self.data.append(gfield)
#    def __len__(self):
#        return len(self.data)
#    def __getitem__(self,index):
#        return self.data[index]
#
#class GasAbsLookup(ArtsType):
#    """This class enables the calculation of GasAbsLookup tables for arts 1.1.*
#    using the stable branch arts 1.0.*. An example of using this class will soon
#    be provided."""
#    def __init__(self,
#                 species=None,  #list of lists of species tags
#                 f_grid=None,   #vector
#                 p_grid=None,   #vector
#                 T_pert=None):  #vector
#        """The initialisation arguments are *species*, a list of lists of species tags as
#        appears in the arts 1.1 GasAbsLookupTable; and vectors *f_grid*, *p_grid*,
#        and *T_pert*, representing frequency, pressure and temperature perturbations"""
#        self.species=species
#        self.f_grid=f_grid
#        self.p_grid=p_grid
#        self.T_pert=T_pert
#        self.nonlinear_species=[]
#        self.nls_pert=array([])
#
#    def calc(self,
#             tags,         #list of strings e.g. ['H2O,H2O-MPM93','O2,O2-MPM93']
#             hitran_filename, #string
#             line_fmin, #float
#             line_fmax, #float
#             atm_basename): #string
#        """Calculated the lookup table using arts 1.0.
#        **Input:**
#        *tags*,             list of arts 1.0 tags e.g. ['H2O,H2O-MPM93','O2,O2-MPM93'];
#        *hitran_filename*,  the name of the hitran line file;
#        *line_fmin*,       float - the minimum frequency line to consider;
#        *line_fmax*,       float - the maximum frequency line to consider;
#        *atm_basename*,     string - the basename where the atmospheric profiles can be found
#                                  in arts1.0 ascii format"""
#        if ARTS1_EXEC==None:
#            raise PyARTSError("ARTS 1 executable path not defined. Either set the environment variable ARTS1_PATH, or manually set arts1.ARTS1_EXEC")
#        basename=tempfile.mktemp()
#        arts1_save(self.p_grid,basename+'.p_grid.aa')
#        arts1_save(self.f_grid,basename+'.f_grid.aa')
#
#        arts1args={'tags':tags,'hitran_filename':hitran_filename,'line_fmin':line_fmin,
#                   'line_fmax':line_fmax,'atm_basename':atm_basename,'T_offset':0.0,
#                   'p_file':basename+'.p_grid.aa','f_file':basename+'.f_grid.aa'}
#        arts1_file_from_command_list(abs_file(**arts1args),basename+'.arts')
#        os.system(ARTS1_EXEC+' '+basename+'.arts')
#        self.t_abs=squeeze(arts1_load(basename+'.t_abs.aa'))
#        self.vmrs=arts1_load(basename+'.vmrs.aa')
#        self.xsec=numpy.zeros([len(self.T_pert),len(self.species),len(self.f_grid),
#                         len(self.p_grid)],float)
#        for i in range(len(self.T_pert)):
#            arts1args.update({'T_offset':self.T_pert[i]})
#            arts1_file_from_command_list(abs_file(**arts1args),basename+'.arts')
#            os.system(ARTS1_EXEC+' '+basename+'.arts')
#            abs_per_tg=arts1_load(basename+'.abs_per_tg.aa')
#            n = self.p_grid/BOLTZMANN_CONST/(self.t_abs+self.T_pert[i])
#            xsec_per_tg = abs_per_tg
#            #loop species
#            for j in range(len(abs_per_tg)):
#                #loop pressure
#                for k in range(len(self.p_grid)):
#                    xsec_per_tg[j][:,k]=abs_per_tg[j][:,k]/n[k]/self.vmrs[j,k]
#            for j in range(len(self.species)):
#                self.xsec[i,j,:,:]=xsec_per_tg[j]
#
#    def save(self,filename):
#        """Saves the lookup table in arts-1.1 XML format"""
#        f=artsXML.XMLfile(filename)
#        f.write('<GasAbsLookup>\n')
#        f.write('<Array type="ArrayOfSpeciesTag" nelem='+quotify(str(len(self.species)))+'>\n')
#        for i in range(len(self.species)):
#            f.write('<Array type="SpeciesTag" nelem='+quotify(str(len(self.species[i])))+'>\n')
#            for j in range(len(self.species[i])):
#                f.addText(tag='SpeciesTag',text=quotify(self.species[i][j]))
#            f.write('</Array>\n')
#        f.write('</Array>\n')
#        f.write('<Array type="Index" nelem='+quotify(str(len(self.nonlinear_species)))+'>\n')
#        for i in range(len(self.nonlinear_species)):
#            f.write(str(self.nonlinear_species[i])+'\n')
#        f.write('</Array>\n')
#        f.add(self.f_grid)
#        f.add(self.p_grid)
#        f.add(self.vmrs)
#        f.add(squeeze(self.t_abs))
#        f.add(self.T_pert)
#        f.add(self.nls_pert)
#        f.add(self.xsec)
#        f.write('</GasAbsLookup>\n')
#        f.close()
#
#    def load(self,filename):
#        """loads gas abs lookup table from an ARTS 1.1. XML file"""
#        data=artsXML.load(filename,use_names=False)#['GasAbsLookup']
#        self.species=[]
#        for a1 in data['Array']:
#            l=[]
#            for a2 in a1['Array']:
#                l.append(a2['SpeciesTag'])
#            self.species.append(l)
#        self.f_grid=data['Vector']
#        self.p_grid=data['Vector 0']
#        self.vmrs=data['Matrix']
#        self.t_abs=data['Vector 1']
#        self.T_pert=data['Vector 2']
#        #ignore nls_pert for now (need to fix artsXML.load to properly deal with empty Tensors)
#        self.xsec=data['Tensor4']
#        return self
#
#    def __call__(self,f_index,pressure,temperature,vmrs):
#        """return a vector containing the contribution of each species to the
#        scalar gas absorption coefficient. Before this can be called the method
#        set_slidata must be called (only once). This is analagous to GasAbsLookup::Extract
#        in ARTS"""
#        n=pressure/(temperature * BOLTZMANN_CONST )
#        sga=[]
#        if not len(vmrs)==len(self.species):
#            raise ValueError, "the length of vmrs must match the length of the species data member"
#        for i in range(len(self.species)):
#            sga.append(n*vmrs[i]*self.slidata[f_index][i].interp(pressure,temperature))
#        return array(sga)
#
#
#    def set_slidata(self):
#        """Sets the sli_data member, which is used by the __call__ function to interpolate
#        the lookup table."""
#        x1=self.p_grid
#        x2={}
#        for i in range(len(x1)):
#            x2[x1[i]]=self.t_abs[i]+self.T_pert
#
#        self.slidata=[]
#        for fi in range(len(self.f_grid)):
#            self.slidata.append([])
#            for si in range(len(self.species)):
#                y={}
#                for i in range(len(x1)):
#                    y[x1[i]]=self.xsec[:,si,fi,i]
#                this_slidata=sli.SLIData2()
#                this_slidata.x1=x1
#                this_slidata.x2=x2
#                this_slidata.y=y
#                self.slidata[fi].append(this_slidata)
#
class AbsSpecies(ArtsType):
    """Container for abs_species.

    Accepts "order" keyword-argument for the order in which the species
    should be added (important for interaction with e.g.
    AtmFieldsFromCompact). All other arguments are either:

    - positional arguments, species to consider (no info on tag)
    - keyword arguments. The key is the species, the value either None (no
      info on tag) or a list of tag-attachments, where None means the
      species itself.

    Initialised with mapping table, e.g.
    >>> p = AbsSpecies(H2O=[None, "MPM93"], CO2=None, order=["H2O", "MPM93"])
    >>> print repr(p.tostr())
    '["H2O,H2O-MPM93", "CO2"]'
    >>> print p.species()
    ["H2O", "CO2"]
    """

    def __init__(self, *args, **d):
        self.order = d.pop("order", None)
        if len(args)==1 and isinstance(args[0], (list, tuple)):
            spec = dict.fromkeys(args[0])
        else:
            spec = dict.fromkeys(args)
        spec.update(d)
        self.species_dict = spec
        self.all_species = spec.keys()
        self.species_str = self.tostr()
        if self.order and set(self.order) != set(spec.keys()):
            raise ValueError("order (%s) does not contain same species " \
                "as keys (%s)" % (str(order), str(spec.keys())))

    def species(self):
        """Returns list of species
        """
        return self.all_species

    def tostr(self):
        """String-representation for SpeciesSet.
        """
        keys = self.order or self.all_species
        L = []
        for k in keys:
            v = self.species_dict[k]
            if v is None:
                L.append(quotify(self.item2str(k, v)))
            else:
                L.append(quotify(",".join(self.item2str(k, p) for p in v)))
        return "[" + ",".join(L) + "]"

    def update(self, other):
        """Updates species with other.
        
        All species in other must exist in self.
        """

        for k, v in other.species_dict.items():
            if not k in self.species():
                raise ValueError("Species not found: %s. I have: %s." % \
                    (k, str(self.species())))
            self.species_dict[k] = v
        self.species_str = self.tostr()

    def copy(self):
        return self.__class__(**self.species_dict)
    __copy__ = copy

    def __str__(self):
        return self.species_str

    @staticmethod
    def item2str(species, item):
        """String representation for item

        Item can be None, then it's the species.
        Otherwise, return species-item.
        """

        if item is None:
            return species
        else:
            return "%s-%s" % (species, item)

class SingleScatteringData(ArtsType):
    """The class representing the arts SingleScatteringData class.
    
    The data members of this object are identical to the class of the same name in
    ARTS; it includes all the single scattering properties required for
    polarized radiative transfer calculations: the extinction matrix, the
    phase matrix, and the absorption coefficient vector.  The angular,
    frequency, and temperature grids for which these are defined are also
    included.  Another data member - *ptype*, describes the orientational
    symmetry of the particle ensemble, which determines the format of the
    single scattering properties.  The data structure of the ARTS
    SingleScatteringData class is described in the ARTS User
    Guide.

    The methods in the SingleScatteringData class enable the
    calculation of the single scattering properties, and the output of the
    SingleScatteringData structure in the ARTS XML format (see example file).
    The low-level calculations are performed in arts_scat.

    Constructor input
    ~~~~~~~~~~~~~~~~~
    
    ptype : integer
        As for ARTS; see Arts User Guide

    f_grid : 1-D array
        array for frequency grid [Hz]

    T_grid : 1-D array
        array for temperature grid [K]

    za_grid : 1-D array
        array for zenith-angle grid [degree]

    aa_grid : 1-D array
        array for azimuth-angle grid [degree]

    equiv_radius : number
        equivalent volume radius [micrometer]

    NP : integer or None
        code for shape: -1 for spheroid, -2 for cylinder,
        positive for chebyshev, None for arbitrary shape
        (will not use tmatrix for calculations)

    phase : string
        ice, liquid

    aspect_ratio : number
        Aspect ratio [no unit]

    Some inputs have default values, see SingleScatteringData.defaults.
    """

    defaults={"ptype": PARTICLE_TYPE_MACROS_ISO, # as defined in optproperties.h
              "T_grid": numpy.array([250]),
              "za_grid": numpy.arange(0, 181, 10),
              "aa_grid": numpy.arange(0, 181, 10),
              "equiv_radius":200, # equivalent volume radius
              "NP":-1, # -1 for spheroid, -2 for cylinder, positive for chebyshev
              # set to None for non-tmatrix (this will make calculations
              # impossible, making this object just a data container)
              'phase':'ice',
              "aspect_ratio":1.000001}

    mrr = None
    mri = None

    def __init__(self,params={}, **kwargs):
        """See class documentation for constructor info.
        """

        # enable keyword arguments
        if kwargs and not params:
            params = kwargs

        params = general.dict_combine_with_default(params,
                                              self.__class__.defaults)
        # check parameters
        # make sure grids are numpy arrays
        for grid in ['f','T','za','aa']:
            params[grid+'_grid'] = numpy.array(params[grid+'_grid'])
        if params['aspect_ratio'] == 1:
            raise PyARTSError("'aspect_ratio' can not be set to exactly 1 due to numerical difficulties in the T-matrix code. use 1.000001 or 0.999999 instead.")

        if "description" in params:
            self.description = params["description"]
        else:
            self.description="This arts particle file was generated by the arts_scat Python\n"+\
                "module, which uses the T-matrix code of Mishchenko to calculate single \n"+\
                "scattering properties. The parameters used to create this file are shown\n"+\
                "below\n"+str(params)
        for k, v in params.items():
            setattr(self, k, v)

    def calc(self,precision=0.001):
        """Calculates the extinction matrix, phase matrix, and absorption
        vector data required for an arts single scattering data file.
        """
        # some name shortening
        self.description += "\n\nThe precision parameter for these" \
                            " calculations is %s\n" % precision
        ext_mat_data, pha_mat_data, abs_vec_data = \
            arts_scat.calc_SSP(
                f_grid = self.f_grid,
                T_grid = self.T_grid,
                za_grid = self.za_grid,
                aa_grid = self.aa_grid,
                r_v = self.equiv_radius,
                aspect_ratio = self.aspect_ratio,
                NP = self.NP,
                precision = precision,
                phase = self.phase,
                ptype = self.ptype,
                mrr = getattr(self, "mrr", None),
                mri = getattr(self, "mri", None))
                           
        self.ext_mat_data = ext_mat_data
        self.pha_mat_data = pha_mat_data
        self.abs_vec_data = abs_vec_data

    def to_xml(self):
        xml_obj=artsXML.XML_Obj('SingleScatteringData')
        xml_obj.write(artsXML.number_to_xml("Index",self.ptype))
        xml_obj.write(artsXML.text_to_xml("String","\""+self.description+"\""))
        xml_obj.write(artsXML.tensor_to_xml( numpy.array(self.f_grid)))
        xml_obj.write(artsXML.tensor_to_xml( numpy.array(self.T_grid)))
        xml_obj.write(artsXML.tensor_to_xml( numpy.array(self.za_grid)))
        xml_obj.write(artsXML.tensor_to_xml( numpy.array(self.aa_grid)))
        xml_obj.write(artsXML.tensor_to_xml(self.pha_mat_data))
        xml_obj.write(artsXML.tensor_to_xml(self.ext_mat_data))
        xml_obj.write(artsXML.tensor_to_xml(self.abs_vec_data))
        return xml_obj.finalise().str

    def filename_gen(self):
        """Creates a horrible looking filename from existing parameters
        """
##        #check that scat directory exists
##        if not os.path.exists(general.DATA_PATH+'/scat'):
##            os.mkdir(general.DATA_PATH+'/scat')
        filename=general.DATA_PATH+'/scat/p'+str(self.ptype)+'f'+\
                str(int(self.f_grid[0]*1e-9))+\
                '-'+ str(int(self.f_grid[-1]*1e-9))+\
                'T'+str(int(self.T_grid[0]))+\
                '-'+str(int(self.T_grid[-1]))+'r'+\
                str(self.equiv_radius)+'NP'+\
                str(self.NP)+'ar'+\
                str(self.aspect_ratio).replace('.','_')[:5]+\
                self.phase+'.xml'
        return filename

    def generate(self,precision=0.001):
        """performs *calc()* and *save* with a filename
        generated from particle parameters"""
        self.calc(precision)
        self.save(self.filename_gen())
        return self

    
    def __repr__(self):
        S = StringIO()
        S.write("<SingleScatteringData ")
        S.write("ptype=%d " % self.ptype)
        S.write("phase=%s " % self.phase)
        S.write("equiv_radius=%4e" % self.equiv_radius)
        for nm in ("f_grid", "T_grid", "za_grid", "aa_grid"):
            g = getattr(self, nm)
            S.write(" ")
            if g.size > 1:
                S.write("%s=%4e..%4e" % (nm, g.min(), g.max()))
            elif g.size == 1:
                S.write("%s=%4e" % (nm, float(g.squeeze())))
            else:
                S.write("%s=[]" % nm)
        S.write(">")

        return S.getvalue()


    def integrate(self):
        """Calculate integrated Z over the sphere.

        Only work for azimuthally symmetrically oriented particles.
        """
        if self.ptype == PARTICLE_TYPE_MACROS_ISO:
            return arts_math.integrate_phasemat(
                self.za_grid,
                self.pha_mat_data[..., 0].squeeze())
        else:
            raise general.PyARTSError("Integration implemented only for"
                "ptype = %d. Found ptype = %s" %
                    (PARTICLE_TYPE_MACROS_ISO, self.ptype))

    def normalise(self):
        """Normalises Z to E-A.
        """

        Z_int = self.integrate()
        E_min_A = self.ext_mat_data.squeeze() - self.abs_vec_data.squeeze()
        # use where to prevent divide by zero
        factor = E_min_A/numpy.where(Z_int==0, 1, Z_int)
        factor.shape = (factor.size, 1, 1, 1, 1, 1)
        self.pha_mat_data[..., 0] *= factor
        #self.pha_mat_data[..., 0] = self.pha_mat_data[..., 0] * factor

    def normalised(self):
        """Returns normalised copy
        """

        c = copy.deepcopy(self)
        c.normalise()
        return c

    def __getitem__(self, v):
        """Get subset of single-scattering-data

        Must take four elements (f, T, za, aa).
        Only implemented for randomly oriented particles.
        """
        
        if self.ptype != PARTICLE_TYPE_MACROS_ISO:
            raise general.PyARTSError("Slicing implemented only for"
                "ptype = %d. Found ptype = %d" %
                    (PARTICLE_TYPE_MACROS_ISO, self.ptype))
        v2 = list(v)
        for i, el in enumerate(v):
            # to preserve the rank of the data, [n] -> [n:n+1]
            if isinstance(el, numbers.Integral):
                v2[i] = slice(v[i], v[i]+1, 1)
        f, T, za, aa = v2
        # make a shallow copy (view of the same data)
        c = copy.copy(self)
        c.f_grid = c.f_grid[f]
        c.T_grid = c.T_grid[T]
        c.za_grid = c.za_grid[za]
        c.aa_grid = c.aa_grid[aa]
        c.ext_mat_data = c.ext_mat_data[f, T, :, :, :]
        c.pha_mat_data = c.pha_mat_data[f, T, za, aa, :, :, :]
        c.abs_vec_data = c.abs_vec_data[f, T, :, :, :]
        c.checksize()
        return c

    def checksize(self):
        """Verifies size is consistent.

        raises PyARTSError if not. Otherwise, do nothing.
        """
        if not ((self.f_grid.size or 1, self.T_grid.size or 1) ==
                 self.ext_mat_data.shape[:2] ==
                 self.pha_mat_data.shape[:2] ==
                 self.abs_vec_data.shape[:2] and
                (self.za_grid.size or 1, self.aa_grid.size or 1) ==
                 self.pha_mat_data.shape[2:4]):
            raise general.PyARTSError(
                "Inconsistent sizes in SingleScatteringData.\n"
                "f_grid: %s, T_grid: %s, za_grid: %s, aa_grid: %s, "
                "ext_mat: %s, pha_mat: %s, abs_vec: %s" %
                    (self.f_grid.size or 1, self.T_grid.size or 1,
                     self.za_grid.size or 1, self.aa_grid.size or 1,
                     self.ext_mat_data.shape, self.pha_mat_data.shape,
                     self.abs_vec_data.shape))

    @classmethod
    def _load_from_artsXML_object(cls, artsXML_object):
        """Loads a SingleScatteringData object from an artsXML_object.
        """

        description = artsXML_object['String'][1:-1]

        params={
                "ptype":artsXML_object['Index'],
                "f_grid":artsXML_object['Vector'],
                "T_grid":artsXML_object['Vector 0'],
                "za_grid":artsXML_object['Vector 1'],
                "aa_grid":artsXML_object['Vector 2']
            }

        obj = cls(description=description, **params)
        obj.pha_mat_data = artsXML_object['Tensor7']
        obj.ext_mat_data = artsXML_object['Tensor5']
        obj.abs_vec_data = artsXML_object['Tensor5 0']
        return obj


    @classmethod
    def load(cls, filename):
        """Loads a SingleScatteringData object from an existing file.
        
        Note that this can only import data members that are actually in the file
        - so the scattering properties may not be consistent with the
        *params* data member.
        """
        scat_data = artsXML.load(filename)
        return cls._load_from_artsXML_object(scat_data)

    @classmethod
    def fromData(cls, tp, size, shapestr, frequencies, angles, data_elem):
        """Read from one row of data

        IN
            tp          type, can be "Hong", "Yang".
            size        size for the particle [m]
            shapestr    string representing the shape
            frequencies array with frequency grid
            angles      array with zenith-angle grid
            data_elem   one row as returned by .io.readHong/.io.readYang
        OUT
            SingleScatteringData object

        """
        desc = "Scattering data from %s dataset. " \
               "SSD-object generated %s. " \
               "Shape: %s. " \
               u"Particle size [micrometer]: %s" % \
                (tp, datetime.datetime.now(), shapestr, size)
        ssd = cls(
            f_grid = frequencies,
            za_grid = angles,
            aa_grid = numpy.zeros(shape=(0,)),
            equiv_radius = data_elem[0]["V"]**(1/3),
            NP = None,
            description = desc)

        # ext_mat: Tensor5
        ext_mat = numpy.zeros((frequencies.size, 1, 1, 1, 1), dtype=numpy.float32)
        ext_mat[:, 0, 0, 0, 0] = data_elem["K"]
        ssd.ext_mat_data = ext_mat
        # abs_vec: Tensor5
        abs_vec = numpy.zeros((frequencies.size, 1, 1, 1, 1), dtype=numpy.float32)
        abs_vec[:, 0, 0, 0, 0] = data_elem["A"]
        ssd.abs_vec_data = abs_vec
        # pha_mat: construct Tensor7 as per ARTS User guide, section 8.2.2
        pha_mat = numpy.zeros((frequencies.size, 1, angles.size, 1, 1, 1, 6), dtype=numpy.float32)
        if tp in set(["Yang", "Yang_2005"]):
            pha_mat[:, 0, :, 0, 0, 0, 0] = data_elem["P11"] 
        elif tp == "Hong":
            # we have no need for P21, P43
            pha_mat[:, 0, :, 0, 0, 0, :] = data_elem["Z"].swapaxes(1, 2)[..., [0, 1, 3, 4, 5, 7]]
        ssd.pha_mat_data = pha_mat
        return ssd

class ScatteringMetaData(ArtsType):
    """Represents a ScatteringMetaData object.

    See online ARTS documentation for object details.
    """

    def __init__(self, type, shape, density, d_max, V,
                 A_projec, asratio, description=None):
        self.type = type
        self.shape = shape
        self.density = density
        self.d_max = d_max
        self.V = V
        self.A_projec = A_projec
        self.asratio = asratio
        self.description = description or "ScatteringMetaData\n" + repr(self)

    def to_xml(self):
        """Get artsXML string-representation
        """
        x = artsXML.XML_Obj("ScatteringMetaData")
        x.write(artsXML.text_to_xml("String", general.quotify(self.description)))
        x.write(artsXML.text_to_xml("String", general.quotify(self.type)))
        x.write(artsXML.text_to_xml("String", general.quotify(self.shape)))
        x.write(artsXML.number_to_xml("Numeric", self.density))
        x.write(artsXML.number_to_xml("Numeric", self.d_max))
        x.write(artsXML.number_to_xml("Numeric", self.V))
        x.write(artsXML.number_to_xml("Numeric", self.A_projec))
        x.write(artsXML.number_to_xml("Numeric", self.asratio))
        return x.finalise().str

    def __repr__(self):
        S = StringIO()
        S.write("<ScatteringMetaData ")
        S.write("type=%s " % self.type)
        S.write("shape=%s " % self.shape)
        S.write("density=%4e " % self.density)
        S.write("d_max=%4e " % self.d_max)
        S.write("V=%4e " % self.V)
        S.write("A_projec=%4e " % self.A_projec)
        S.write("asratio=%4e>" % self.asratio)
        return S.getvalue()


    @classmethod
    def load(cls, f):
        raise NotImplementedError

    @classmethod
    def fromData(cls, tp, size, shapestr, frequencies, angles, data_elem):
        """fromData (Hong, Yang, etc.)
        IN
            tp          type, can be "Hong", "Yang".
            size        size for the particle [m]
            shapestr    string representing the shape
            frequencies array with frequency grid
            angles      array with zenith-angle grid
            data_elem   one row as returned by .io.readHong/.io.readYang
        OUT
            ScatteringMetaData object

       """
        return cls(
            type = "Ice",
            shape = shapestr,
            density = 920, # kg/m^3
            d_max = data_elem[0]["d_max"],
            V = data_elem[0]["V"],
            A_projec = data_elem[0]["A_projected"],
            asratio = -1)




class ArrayOfSingleScatteringData(ArrayOf):
    """Represents an ArrayOfSingleScatteringData.
    """

    contains = SingleScatteringData

    dispatcher_SSD = {
        "Hong": functools.partial(SingleScatteringData.fromData, "Hong"),
        "Yang": functools.partial(SingleScatteringData.fromData, "Yang"),
        "Yang_2005": functools.partial(SingleScatteringData.fromData, "Yang_2005"),
    }

    dispatcher_SMD = {
        "Hong": functools.partial(ScatteringMetaData.fromData, "Hong"),
        "Yang": functools.partial(ScatteringMetaData.fromData, "Yang"),
        "Yang_2005": functools.partial(ScatteringMetaData.fromData, "Yang_2005"),
    }

    def normalise(self):
        """Call .normalise() on each SSD
        """

        for elem in self:
            elem.normalise()

    def normalised(self):
        """Return normalised copy of self
        """

        c = copy.deepcopy(self)
        for elem in c:
            elem.normalise()
        return c

    def integrate(self):
        """Return list with all integrated values.
        """

        L = []
        for elem in self:
            L.append(elem.integrate())
        return L

    @property
    def sub(self):
        """Get subslice of each element.
        """
        class Subber(object):
            def __getitem__(self2, *args):
                L = self.__class__()
                for elem in self:
                    L.append(elem.__getitem__(*args))
                return L
        return Subber()

    @classmethod
    def fromData(cls, name, shapestr, sizes, frequencies, angles, data):
        """Load from data.

        Do not call directly, call fromYang, fromHong, etc.
        """

        obj = cls()
        obj.smd = ArrayOfScatteringMetaData()
        for (i, size) in enumerate(sizes):
            # add SSD/SMD, one by one
            obj.append(cls.dispatcher_SSD[name](size, shapestr, frequencies, angles, data[:, i]))
            obj.smd.append(cls.dispatcher_SMD[name](size, shapestr, frequencies, angles, data[:, i]))
        return obj

    # FIXME: put this in a better place
    yang_phase = "/storage4/home/mendrok/data/cloud/yang/P11/${shape}_P11_infr.dat"
    yang_ang = "/storage4/home/mendrok/data/cloud/yang/P11/angle.dat"
    yang_other = "/storage4/home/mendrok/data/cloud/yang/P11/optical/${shape}_total.dat"

    @classmethod
    def fromYang(cls, shapestr):
        """Loads ArrayOfSingleScatteringData from Yang-data.

        Also sets ArrayOfScatteringMetaData object.
        """

        f_phase = string.Template(cls.yang_phase).substitute(shape=shapestr)
        f_other = string.Template(cls.yang_other).substitute(shape=shapestr)
        
        (sizes, wavelengths, angles), data = io.readYang(f_phase, f_other, cls.yang_ang)
        frequencies = physics.wavelength2frequency(wavelengths)
        obj = cls.fromData("Yang", shapestr, sizes, frequencies, angles, data)
        return obj

    # FIXME: put this in a better place
    hong = "/storage3/data/scattering_databases/hong/single/${shape}.single_mm_21"
    @classmethod
    def fromHong(cls, shapestr):
        """Loads ArrayOfSingleScatteringData from Hong-data
        """

        f = string.Template(cls.hong).substitute(shape=shapestr)

        (sizes, wavelengths, angles), data = io.readHong(f)
        frequencies = physics.wavelength2frequency(wavelengths)
        obj = cls.fromData("Hong", shapestr, sizes, frequencies, angles, data)
        return obj

    # FIXME: put this in a better place
    yang_2005 = "/storage3/data/scattering_databases/yang/Yang_2005_IR/${shape}.out"
    @classmethod
    def fromYang_2005(cls, shapestr):
        f = string.Template(cls.yang_2005).substitute(shape=shapestr)

        (sizes, wavelengths, angles), data = io.readYang2005(f)
        frequencies = physics.wavelength2frequency(wavelengths)
        obj = cls.fromData("Yang_2005", shapestr, sizes, frequencies, angles, data)
        return obj
        
#        obj.smd = ArrayOfScatteringMetaData()
#        # one SingleScatteringData object for each size
#        for (i, size) in enumerate(sizes):
#            desc = "Scattering data from Yang dataset. " \
#                   "SSD-object generated %s. " \
#                   "Shape: %s. " \
#                   "Size: %s" % \
#                    (datetime.datetime.now(), shapestr, size)
#            ssd = SingleScatteringData(
#                f_grid = frequencies,
#                za_grid = angles,
#                aa_grid = numpy.zeros(shape=(0,)),
#                equiv_radius = data[0, i]["V"]**(1/3),
#                NP = None,
#                description = desc)
#
#            # ext_mat: Tensor5
#            ext_mat = numpy.zeros((frequencies.size, 1, 1, 1, 1), dtype=numpy.float32)
#            ext_mat[:, 0, 0, 0, 0] = data[:, i]["E"]
#            ssd.ext_mat_data = ext_mat
#            # abs_vec: Tensor5
#            abs_vec = numpy.zeros((frequencies.size, 1, 1, 1, 1), dtype=numpy.float32)
#            abs_vec[:, 0, 0, 0, 0] = data[:, i]["A"]
#            ssd.abs_vec_data = abs_vec
#            # pha_mat: construct Tensor7 as per ARTS User guide, section 8.2.2
#            pha_mat = numpy.zeros((frequencies.size, 1, angles.size, 1, 1, 1, 6), dtype=numpy.float32)
#            pha_mat[:, 0, :, 0, 0, 0, 0] = data[:, i]["P11"] 
#            ssd.pha_mat_data = pha_mat
#
#            obj.append(ssd)
#            obj.smd.append(ScatteringMetaData(
#                description = desc,
#                type = "Ice",
#                shape = shapestr,
#                density = 920, # kg/m^3
#                d_max = data[0, i]["d_max"],
#                V = data[0, i]["V"],
#                A_projec = data[0, i]["A_projected"],
#                asratio = -1))
#
#        return obj
#




class ArrayOfScatteringMetaData(ArrayOf):
    contains = ScatteringMetaData
