"""Represent 3D clouds and cloud microphysics.

This module includes functions and classes representing 3D clouds and cloud
microphysics.  This module has all you need for the generation of scattering
data files (via `the arts_scat module`_) and particle number density fields, which
enable the representation of 3D cloud fields in ARTS simulations

Example of use: a 3D ice and liquid cloud field.

1. load iwc and lwc fields from TRMM data

>>> from PyARTS import *
>>> iwc_field=arts_types.LatLonGriddedField3.load('../016/iwc_field.xml')
>>> lwc_field=arts_types.LatLonGriddedField3.load('../016/lwc_field.xml')

2. load temperature field I prepared earlier

>>> t_field=arts_types.LatLonGriddedField3.load('t_field.xml')

3. Define hydrometeors

>>> ice_column=clouds.Crystal(NP=-2,aspect_ratio=0.5,ptype=30,npoints=10)
>>> water_droplet=clouds.Droplet(c1=6,c2=1,rc=20)

4. Create cloud field

>>> a_cloud=clouds.Cloud(t_field=t_field,iwc_field=iwc_field,lwc_field=lwc_field)

5. add hydrometeors

>>> a_cloud.addHydrometeor(ice_column,habit_fraction=1.0)
>>> a_cloud.addHydrometeor(water_droplet,habit_fraction=1.0)

6. generate (or find existing) single scattering data

>>> a_cloud.scat_file_gen(f_grid=[200e9,201e9],num_proc=2)

7. generate pnd fields

>>> a_cloud.pnd_field_gen('pnd_field.xml')

8. save cloud object for later

>>> quickpickle(a_cloud,'Cloud3D.pickle')


"""


#from arts_types import *
#from scipy import *
#from scipy.special import gamma,genlaguerre,laguerre
#from scipy import special

import math

import numpy
import scipy
import scipy.special

from . import artsXML
from . import arts_math
from . import arts_scat
from . import arts_types
from .general import dict_combine_with_default,DATA_PATH, PyARTSError
from .arts_types import LatLonGriddedField3, ArrayOfLatLonGriddedField3

#Constants################
rho_ice=0.917#gcm^-3

#SIZE DISTRIBUTIONS#######

def gamma_dist(r,CWC,r_eff,g,phase='ice'):
    """Gamma distribution.
    Returns an array of particle number densities, for an array of particle radii, *r*,
    according to the Gamma size distribution,

    :m:`\[n(r)=N_0 \\frac{(\\frac{r}{\\beta})^{\gamma-1}\exp(-\\frac{r}{\\beta})}{\\beta\Gamma(\gamma)}\]`

    ,where :m:`$\gamma$` is the shape parameter, and :m:`$\\beta=\\frac{r_{eff}}{\gamma+2}$`.

    Arguments
    ~~~~~~~~~

    r: 1-D array
        array of particle radii [micrometer]
    IWC: number
        cloud water content [gm\ :sup:`-3`]
    r_eff: number
        effective radius [?]
    g: number
        shape parameter (:m:`$\gamma$`)
    """
    if phase=='ice':
        rho=rho_ice*1e6#convert from g/cm^3 to g/m^3
    elif phase=='liquid':
        rho=1e6
    else:
        raise ValueError,"invalid phase argument' must be 'ice' or 'liquid'"
    beta=r_eff/(g+2.0)
    x=r/beta
    n=3*CWC/(4*numpy.pi*rho*beta**4*scipy.special.gamma(g+3))*x**(g-1)*numpy.exp(-x)*1e18
    return n

def mh97(IWC,r,TK):
    """MH97 McFarquhar and Heymsfield 1997's particle size distribution

    Usage
    ~~~~~

       n,integrated_n=mh97(IWC,r,TK).

    Arguments
    ~~~~~~~~~

        IWC: number
            Ice water content [gm\ :sup:`-3`]
        r: 1-D array
            Array of radii [micrometer]
        TK: number
            Temperature [K]
        n: number
            number density in l\ :sup:`-1`\ mm\ :sup:`-1`
            (or :m:`$\\rm{m}^{-3}\mu \\rm{m}^{-1}$`);
        integrated_n:
            integrated number density in each size bin in m\ :sup:`-3`.

    Equation
    ~~~~~~~~

       The MH97 size distribution is given by the sum of a gamma distribution for small
       particles and a log-normal distribution for large particles.
       The small particle gamma component is

       :m:`\[ N(D_m) = \\frac{6IWC_{<100}\\alpha^5_{<100}D_m}{\pi\\rho_{ice}\Gamma(5)}
       \exp\left(-\\alpha_{<100}D_m\\right)\]`,

       and the large particle lognormal component is

       :m:`\[ N(D_m) = \\frac{6 IWC_{>100}}{\pi^{1.5}\\rho_{ice}\sqrt{2}\exp\left(3\mu{>100}+4.5\sigma^2_{>100}\\right)D_m\sigma_{>100}D_0^3}\exp\left[-\\frac{1}{2}\left(\\frac{\log\\frac{D_m}{D_0}-\mu{>100}}{\sigma_{>100}}\\right)^2\\right]\]`

       For details see [McFarquharHeymsfield97]_.

    """
    if IWC==0:
        return numpy.zeros(len(r),float),numpy.zeros(len(r),float)

    D=r*2
    T=TK-273.15
    #Calculate bin boundarys
    Di=numpy.zeros(len(D)+1,float)
    Di[0]=1
    Di[1:len(D)]=0.5*(D[:len(D)-1]+D[1:len(D)])
    Di[len(D)]=1.5*D[len(D)-1]-0.5*D[len(D)-2]

    IWC_0=1.0; #gm^-3
    IWC_lt=min(IWC,0.252*(IWC/IWC_0)**0.837)
    IWC_gt=IWC-IWC_lt
    alpha=-4.99e-3-0.049*numpy.log10(IWC_lt/IWC_0)#micron^-1

    D_0=1.0#micron


    #Some abbreviations
    a=IWC_lt*alpha**5/4/numpy.pi/rho_ice*1e-6#micron^-5
    b=alpha#micron^-1

    #Gamma distributionn component
    gamma_n=a*D*numpy.exp(-b*D)#micron^-4
    gamma_int=a*numpy.exp(-b*Di)*(-1/b**2-Di/b)#micron^-3


    #Log normal component

    if IWC_gt>0:
        sigma=0.47+2.1e-3*T+(0.018-2.1e-4*T)*numpy.log10(IWC_gt/IWC_0)
        mu=5.2+0.0013*T+(0.026-1.2e-3*T)*numpy.log10(IWC_gt/IWC_0)
        c=6*IWC_gt/(numpy.sqrt(2*numpy.pi**3)*rho_ice*D_0**3*sigma*\
                    numpy.exp(3*mu+4.5*sigma**2))*1e-6
        g=sigma
        f=mu
        lognorm_n=c/D*numpy.exp(-0.5*((numpy.log(D/D_0)-f)/g)**2)
        lognorm_int=c*g*numpy.sqrt(numpy.pi/2)*scipy.special.erf((numpy.log(Di/D_0)-f)/numpy.sqrt(2)/g)
    else:
        lognorm_n=numpy.zeros(len(r),float)
        lognorm_int=numpy.zeros(len(r)+1,float)

    #Combine
    n=gamma_n+lognorm_n
    integrated_n=gamma_int[1:]+lognorm_int[1:]-\
                gamma_int[:len(Di)-1]-lognorm_int[:len(Di)-1]
    n=n*2e18;   # convert to l^-1mm^-1 m^-3micron-1(radius)
    integrated_n=integrated_n*1e18 # convert to m^-3

    IWC_check=(integrated_n*4/3*numpy.pi*(D*1e-6/2)**3*rho_ice*1e6).sum()

    #This seems to be what JPL do to conserve IWC.  The alternative would be to
    #take bin boundaries and choose the appropriate D that conserves IWC.
    integrated_n=integrated_n*IWC/IWC_check
    return n,integrated_n

def mh97_vect(IWC,r,TK):
    """vectorised version of mh97 returning n only"""
    #I am going to try and reduce the number of variables to try and save memory
    final_shape=list(IWC.shape)
    nr=len(r)
    final_shape.append(nr)
    IWC=IWC.ravel()
    nIWC=len(IWC)
    n=numpy.zeros((nIWC,nr),float)

    TK=TK.ravel()-273.15 ##TK is now in celcius
    d=r*2#
    #Gamma distribution component
    A=0.252*(IWC)**0.837
    IWC_lt=numpy.where(IWC<A,IWC,A)
    ok = IWC_lt>0
    A = numpy.zeros(IWC_lt.shape)
    A[ok] = -4.99e-3-0.049*numpy.log10(IWC_lt[ok])
    #A=numpy.where(IWC_lt>0,-4.99e-3-0.049*numpy.log10(IWC_lt),0)
    for ir in range(nr):
        n[:,ir]=IWC_lt*A**5/4/numpy.pi/rho_ice*1e-6*d[ir]*numpy.exp(-A*d[ir])#micron^-4
    assert not (numpy.isnan(n)).any()
    assert (n>=0).all()
    #Log normal component
    IWC=IWC-IWC_lt
    del IWC_lt
    non_zero=(IWC>0).nonzero()
    TK=TK.take(non_zero)
    IWC_=IWC.take(non_zero)
    A_=0.47+2.1e-3*TK+(0.018-2.1e-4*TK)*numpy.log10(IWC_)
    mu_=5.2+0.0013*TK+(0.026-1.2e-3*TK)*numpy.log10(IWC_)

    del(TK)
    #put values back into correctly sized vectors
    IWC=numpy.zeros(nIWC,float)
    A=numpy.ones(nIWC,float)
    mu=numpy.zeros(nIWC,float)
    numpy.put(IWC,non_zero,IWC_)
    numpy.put(A,non_zero,A_)
    numpy.put(mu,non_zero,mu_)

    del mu_,IWC_,A_,non_zero
    for ir in range(nr):
        n[:,ir]+=6*IWC/(numpy.sqrt(2*numpy.pi**3)*rho_ice*A*numpy.exp(3*mu+4.5*A**2))*1e-6/d[ir]*\
         numpy.exp(-0.5*((numpy.log(d[ir])-mu)/A)**2)
    assert not numpy.isnan(n).any()
    assert (n>=0).all()
    n*=2e18
    n.shape=final_shape
    return n


def mh97_int(IWC,r,TK):
    """calls mh97 but only returns the integrated number density over each
    size-bin"""
    n,integrated_n=mh97(IWC,r,TK)
    return integrated_n

def nioku(LWC,r,rc,c1,c2):
    """The modified gamma size distribution for water droplets as given in the MLS cloudy sky
    ATBD. The units are  l\ :sup:`-1`\ mm\ :sup:`-1` (or :m:`$\\rm{m}^{-3}\mu m^{-1}$`)

    :m:`\[n(r)=Ar^{c_1}\exp\left(-Br^{c_2}\\right)\]`

    , where

    :m:`\[A=\\frac{3LWCc_2B^\\frac{c1+4.0}{c2}\\times10^{12}}{4\pi\Gamma\left(\\frac{c1+4.0}{c2}\\right)}\]`

    ,and :m:`$B=\\frac{c_1}{c_2 r_c^{c_2}}$`

    **Arguments**:
       *LWC*,
          ice water content in gm\ :sup:`-3`;
       *r*,
          Array of radii in microns;
       *rc,c1,c2*,
          size distribution paramaters. Some suggested values for these parameters are:
          Stratus (rc=10 micron, c1=6, c2=1), Cumulus Congestus (rc=20 micron, c1=5, c2= 0.5).

    """


    B=float(c1)/(c2*rc**c2)
    A=3*LWC*c2*B**((c1+4.0)/c2)*1e12/(4*numpy.pi*scipy.special.gamma((c1+4.0)/c2))
    n=A*r**c1*numpy.exp(-B*r**c2)
    return n

def nioku_int(LWC,r,rc,c1,c2):
    """Gives the integrated droplet number density for each size bin ,with size
    bin radii defined by the input vector r"""
    #define function to be integrated
    def func(r1):
        return nioku(LWC,r1,rc,c1,c2)
    intn=numpy.zeros(r.shape,float)
    #do the integrations, summing estimated LWC along the way
    sum=0
    intn[0]=arts_math.qromb(func,0.01,(r[0]+r[1])/2,EPS=0.05,JMAX=10,K=3)[0]
    sum+=intn[0]*4*pi/3*1e-12*r[0]**3
    for i in range(1,len(r)-1):
        intn[i]=arts_math.qromb(func,(r[i]+r[i-1])/2,(r[i]+r[i+1])/2,
                                EPS=0.05,JMAX=10,K=3)[0]
        sum+=intn[i]*4*pi/3*1e-12*r[i]**3
    intn[-1]=arts_math.qromb(func,(r[-1]+r[-2])/2,r[-1]+(r[-1]-r[-2])/2,
                             EPS=0.05,JMAX=10,K=3)[0]
    sum+=intn[-1]*4*pi/3*1e-12*r[-1]**3
    #rescale to conserve LWC
    #print LWC/sum
    intn*=LWC/sum
    return intn


###################Cloud structures################################

class Cloud(object):
    """High-level class representing a cloud.
    
    A high level class for the generation of ARTS cloud field data.
    A Cloud object is initialised with up to three arts_type.LatLonGriddedField3
    objects representing 3D temperature, ice water content, and liquid
    water content fields.  The temperature field is compulsory but either
    IWC or LWC may be omitted.
    Droplet or Crystal objects can then be added to the Cloud Object using
    the addHydrometeor method.  The user is encouraged to create their own
    Hydrometeor classes (all that is required is that they have scat_calc
    and pnd_calc methods with the same input/output arguments).
    The scat_file_gen and pnd_field_gen methods create the single
    scattering data files and particle number density files required to
    represent the cloud field in ARTS simulations.

    Note that, as of ARTS-1-14-139, ARTS can internally calculate particle
    number density fields. The single-scattering-data calculations,
    however, are still required to do externally.

    Constructor input
    ~~~~~~~~~~~~~~~~~

    t_field: LatLonGriddedField3
        Temperature field [K]

    iwc_field: LatLonGriddedField3
        Ice water content field [g/m^3]
        Maybe omitted, but then lwc_field must be present.

    lwc_field: LatLonGriddedField3
        Liquid water content field [g/m^3]
        Maybe omitted, but then iwc_field must be present.

    Attributes
    ~~~~~~~~~~

    Note: not all attributes are documented here, only those that are of
    relevance for the user.

    t_field: LatLonGriddedField3
        As passed on to the constructor

    iwc_field: LatLonGriddedField3
        As passed on to the constructor

    lwc_field: LatLonGriddedfield3
        As passed on to the constructor

    p_grid: 1-D array
        Pressure grid obtained from iwc_field if present, otherwise
        lwc_field [Pa].

    lat_grid: 1-D array
        Latitude grid, obtained as for p_grid [degrees]

    lon_grid: 1-D array
        Longitude grid, obtained as for lon_grid [degrees]

    scat_files: list of strings
        List of scattering data files generated; filled by
        .scat_file_gen().

    pnd_fields: list of string
        List of pnd field data files generated; filled by
        .pnd_field_gen().

    pnd_data: ArrayOfLatLonGriddedField3
        Contains all the pnd data, as calculated by .pnd_field_gen()

    """
    def __init__(self,t_field,iwc_field=None,lwc_field=None):
        """See class docstring for constructor information.
        """
        self.iwc_field=iwc_field
        self.lwc_field=lwc_field
        self.t_field=t_field
        if iwc_field is not None:
            self.p_grid=iwc_field.p_grid
            self.lat_grid=iwc_field.lat_grid
            self.lon_grid=iwc_field.lon_grid
        elif lwc_field is not None:
            self.p_grid=lwc_field.p_grid
            self.lat_grid=lwc_field.lat_grid
            self.lon_grid=lwc_field.lon_grid
        else:
            raise PyARTSError('Must supply either IWC or LWC field (or both)')
        self.hydrometeors=[]
        self.habit_fractions=[]
        self.scat_files=[]
        self.pnd_fields=[]

    def addHydrometeor(self,hydrometeor,habit_fraction=1.0):
        """Adds hydrometeor to cloud.
        
        Adds a hydrometeor (e.g. a Droplet or Crystal object) to the cloud
        object.  The habit_fraction argument allows the implementation of
        multi-habit ice clouds. The habit_fractions for all of the added
        Crystal objects should add up to 1.0. Otherwise the specified iwc_field
        will not be reproduced.

        Parameters
        ~~~~~~~~~~

        hydrometeor :
            Anything that has .scat_calc and .pnd_calc, like the
            MonoCrystal, Crystal and Droplet objects do.
        habit_fraction : float between 0 and 1
            Fraction of total cloud that this particle type has.

        """
        self.hydrometeors.append(hydrometeor)
        self.habit_fractions.append(habit_fraction)
        return self

    def scat_file_gen(self,f_grid,za_grid=numpy.arange(0,181,10),
                      aa_grid=numpy.arange(0,181,10),num_proc=1):
        """Calculates single scattering data and generate files.
        
        Calculates all of the single scattering data files required to
        represent the cloud field in an ARTS simulation.  The file names are
        stored in the scat_files data member.  The input arguments are
        f_grid,T_grid,za_grid, and aa_grid: numpy arrays determining the
        corresponding data in the arts_types.SingleScatteringData objects. The
        optional argument num_proc determines the number of processes used to
        complete the task.

        Parameters
        ~~~~~~~~~~

        f_grid : 1D-array
            Frequency grid for scattering calculation [Hz].
        za_grid: 1D-array, optional
            Zenith angle grid for scattering calculation [degree].
            Defaults to every 10 degrees.
        aa_grid: 1D-array, optional
            Azimuth angle grid for scattering calculation [degree].
            Defaults to every 10 degrees.
        num_proc: int, optional
            Number of processors to use in calculation. Defaults to 1.

        """
        for hydrometeor in self.hydrometeors:
            self.scat_files.extend(hydrometeor.scat_calc(f_grid, za_grid,
                                                         aa_grid, num_proc))
        return self

    def particle_masses(self):
        """Calculates the mass of each particle type.

        Returns

        pm : 1D-array
            Particle masses [FIXME: unit?]
        """
        pm=self.hydrometeors[0].particle_masses
        for i in range(1,len(self.hydrometeors)):
            pm=numpy.c_[pm,self.hydrometeors[i].particle_masses]
        return pm


    def pnd_field_gen(self,filename):
        """Calculates pnd_field_raw and stores it to ''filename''.
        
        Calculates the pnd data required to represent the cloud field in
        an ARTS simulation.  This produces an arts_types.ArrayOfLatLonGriddedField3
        object, which has the same number of elements as the scat_files data
        member.  This is stored in the pnd_data member and output to *filename*
        in ARTS XML format.

        Parameters
        ~~~~~~~~~~

        filename: stream or string
            Either a stream (something with a .write method) or a string
            describing a filename to write the data to.
        """
        for (i, hydrometeor) in enumerate(self.hydrometeors):
            pnd_fields=hydrometeor.pnd_calc(self.lwc_field,self.iwc_field,
                                            self.t_field)
            for pnd_field in pnd_fields:
                #multiply data by habit fraction
                pnd_field.data*=self.habit_fractions[i]
            self.pnd_fields.extend(pnd_fields)
        self.pnd_data=ArrayOfLatLonGriddedField3(self.pnd_fields)
        self.pnd_data.save(filename)
        self.pnd_file=filename
        return self

    #@property
    #def cloudbox(self):
        #raise NotImplementedError("Calculation of cloudbox not implemented yet")
    def __repr__(self):
        return "<Cloud, hydrometeors=%d>" % len(self.hydrometeors)

def boxcloud(ztopkm,zbottomkm,lat1,lat2,lon1,lon2,cb_size,zfile,tfile,IWC=None,LWC=None):
    """Return a box shaped Cloud object.
    ztopkm,zbottomkm,lat1,lat2,lon1,lon2, IWC, LWC are Numeric values,
    cb_size is a dictionary with keys 'np','nlat', and 'nlon'
    """
    zfield=LatLonGriddedField3.load(zfile)
    tfield=LatLonGriddedField3.load(tfile)
    iwcfield=None
    lwcfield=None

    ztopm=ztopkm*1e3
    zbottomm=zbottomkm*1e3

    p1=numpy.exp(arts_math.interp(zfield.data.squeeze(),numpy.log(zfield.p_grid),zbottomm))[0]
    p2=numpy.exp(arts_math.interp(zfield.data.squeeze(),numpy.log(zfield.p_grid),ztopm))[0]

    #create zero valued gridpoints just outside the cloudbox
    p_grid=numpy.zeros(cb_size['np']+2,float)
    lat_grid=numpy.zeros(cb_size['nlat']+2,float)
    lon_grid=numpy.zeros(cb_size['nlon']+2,float)

    p_grid[0]=p1+1;p_grid[-1]=p2-1
    p_grid[1:-1]=arts_math.nlogspace(p1,p2,cb_size['np'])

    lat_grid[0]=lat1-0.1;lat_grid[-1]=lat2+0.1
    lat_grid[1:-1]=arts_math.nlinspace(lat1,lat2,cb_size['nlat'])

    lon_grid[0]=lon1-0.1;lon_grid[-1]=lon2+0.1
    lon_grid[1:-1]=arts_math.nlinspace(lon1,lon2,cb_size['nlon'])

    if IWC is not None:
        data=numpy.zeros([cb_size['np']+2,cb_size['nlat']+2,cb_size['nlon']+2],float)
        data[1:-1,1:-1,1:-1]=IWC*numpy.ones([cb_size['np'],cb_size['nlat'],cb_size['nlon']],float)
        iwcfield=LatLonGriddedField3(p_grid,lat_grid,lon_grid,data)
        iwcfield.pad()
    if LWC is not None:
        data=numpy.zeros([cb_size['np']+2,cb_size['nlat']+2,cb_size['nlon']+2],float)
        data[1:-1,1:-1,1:-1]=LWC*numpy.ones([cb_size['np'],cb_size['nlat'],cb_size['nlon']],float)
        lwcfield=LatLonGriddedField3(p_grid,lat_grid,lon_grid,data)
        lwcfield.pad()

    old_t_grid=tfield.data.squeeze()
    new_t_grid=arts_math.interp(numpy.log(tfield.p_grid),old_t_grid,numpy.log(p_grid))
    tdata=numpy.zeros([cb_size['np']+2,cb_size['nlat']+2,cb_size['nlon']+2],float)
    for i in range(len(lat_grid)):
        for j in range(len(lat_grid)):
            tdata[:,i,j]=new_t_grid

    tfield=LatLonGriddedField3(p_grid,lat_grid,lon_grid,tdata)

    tfield.pad()
    the_cloud=Cloud(iwc_field=iwcfield,lwc_field=lwcfield,t_field=tfield)
    the_cloud.cloudbox={'p1':p1,'p2':p2,'lat1':lat1,'lat2':lat2,'lon1':lon1,
                        'lon2':lon2}
    return the_cloud


def boxcloud_1D(ztopkm,zbottomkm,cb_size,zfile,tfile,IWC):
    """Return a box shaped Cloud object 
    """
    zfield=LatLonGriddedField3.load(zfile)
    tfield=LatLonGriddedField3.load(tfile)
    ztopm=ztopkm*1e3
    zbottomm=zbottomkm*1e3

    p1=numpy.exp(arts_math.interp(zfield.data.squeeze(),numpy.log(zfield.p_grid),zbottomm))
    p2=numpy.exp(arts_math.interp(zfield.data.squeeze(),numpy.log(zfield.p_grid),ztopm))

    #create zero valued gridpoints just outside the cloudbox
    p_grid=numpy.zeros(cb_size['np']+2,float)
    lat_grid=numpy.zeros([1],float)
    lon_grid=numpy.zeros([1],float)
    data=numpy.zeros([cb_size['np']+2,1,1],float)

    p_grid[0]=p1+1;p_grid[-1]=p2-1
    p_grid[1:-1]=arts_math.nlogspace(p1,p2,cb_size['np'])

    data[1:-1,0,0]=IWC*onumpy.nes([cb_size['np']],float)

    old_t_grid=tfield.data.squeeze()
    new_t_grid=arts_math.interp(numpy.log(tfield.p_grid),old_t_grid,numpy.log(p_grid))
    tdata=numpy.zeros([cb_size['np']+2,1,1],float)
    tdata[:,0,0]=new_t_grid

    plims=[1100e2,0.00001]
    p_grid_new=numpy.zeros([len(p_grid)+2],float)
    p_grid_new[0]=plims[0]
    p_grid_new[1:len(p_grid_new)-1]=p_grid
    p_grid_new[-1]=plims[1]

    tdata_new = numpy.zeros([len(p_grid_new),1,1], float)
    tdata_new[1:len(p_grid_new)-1,0,0]=data.squeeze()
    data_new =numpy.zeros([len(p_grid_new),1,1], float)
    data_new[1:len(p_grid_new)-1,0,0]=data.squeeze()

    tfield=LatLonGriddedField3(p_grid_new,lat_grid,lon_grid,tdata_new)

    iwcfield=LatLonGriddedField3(p_grid_new,lat_grid,lon_grid,data_new)

    the_cloud=Cloud(iwc_field=iwcfield,t_field=tfield)
    the_cloud.cloudbox={'p1':p1,'p2':p2,'lat1':0.,'lat2':0.,'lon1':0.,
                        'lon2':0.}
    return the_cloud


##############Microphysics########################

class Droplet(object):
    """Represents a floating particle consisting of liquid water.
    
    Produces scattering data and pnd fields for liquid water
    clouds.  The size distribution is the modified gamma distribution
    of Nioku, as used in the EOSMLS cloudy-sky forward model.  A
    Droplet object is initialised with the distribution parameters c1,
    c2, and rc. Some suggested values for these parameters are:

    - Stratus (rc=10 micron, c1=6, c2=1)
    - Cumulus Congestus (rc=20 micron, c1=5, c2= 0.5).

    Scattering properties are integrated over the size distribution
    using an *npoints* Laguerre Gauss quadrature, to give a one
    ``arts_types.SingleScatteringData object``.  *T_grid* is an optional
    initialisation argument which overides the default temperature
    grid ([260,280,300,320]) for scattering data file generation.  The
    pnd field is then simply scaled by the lwc_field.  The methods
    scat_calc and pnd_calc are called by the parent Cloud object.

    Parameters
    ~~~~~~~~~~

    c1: number
        Nioku distribution parameter (?)

    c2: number
        Nioku distribution parameter (?)

    rc: number
        Particle radius [micrometer]

    npoints: number
        ?

    T_grid: 1-D array
        temperature grid for scattering calculations [K]

    """
    def __init__(self,c1=6,c2=1,rc=10, npoints=5, T_grid=[260,280,300,320]):
        """Please see the class docstring for constructor information.
        """
        self.c1=c1
        self.c2=c2
        self.rc=rc
        self.T_grid=T_grid

        #First get Laguerre Gauss weights and abscissa
        alpha=(c1+c2+1)/c2
        A=scipy.special.genlaguerre(n=npoints,alpha=alpha)
        x=numpy.real(A.weights[:,0])
        w=numpy.real(A.weights[:,1])
        #turn the abscissa into radii
        B=float(c1)/(c2*rc**c2)
        self.r=(x/B)**(1/c2)
        A=3*c2*B**((c1+4.0)/c2)*1e12/(4*numpy.pi*x**alpha*scipy.special.gamma((c1+4.0)/c2))
        self.pnd_vec=A*w*self.r**(c1+1-c2)/c2/B
        self.particle_masses=4*numpy.pi*self.r**3/3*1e-12 #mass of each particle in g.

    def scat_calc(self,f_grid, za_grid=numpy.arange(0,181,10),
                  aa_grid=numpy.arange(0,181,10),num_proc=1):
        """Produces a single scattering data object that can be simply
        scaled by LWC.  The xx_grid variables have the same meaning as in
        arts_types.SingleScatteringData objects. num_proc determines the number
        of processes used to complete the task. Returns a list (length=1) of
        scattering data files.
        
        Input:
            f_grid      array for frequency grid (Hz)
            za_grid     array for zenith-angle grid (degrees)
            aa_grid     array for azimuth-angle grid (degrees)
            num_proc    number of processes to use in single-scattering calculation

        Output:
            list of strings. Each string contains a path to an
            arts-xml-file describing a SingleScatteringData object.
        """

        self.scat_file=DATA_PATH+'/scat/Droplet'+str(self.c1)+'_'+str(self.c2)+'_'+str(self.rc)+\
            'f'+str(f_grid[0])+'-'+str(f_grid[-1])+'T'+\
            str(self.T_grid[0])+'-'+str(self.T_grid[-1])+'.xml'
        #generate single scattering data files
        scat_params={'f_grid':[f_grid],'T_grid':[self.T_grid],'NP':[-1],
                     'aspect_ratio':[1.000001],'ptype':[20],
                     'phase':'liquid','equiv_radius':self.r,'za_grid':za_grid,
                     'aa_grid':aa_grid}
        scat_files=arts_scat.batch_generate(scat_params,num_proc)
        #Combine these to form a single scattering data object

        scat_data_list=[]
        for fname in scat_files:
            scat_data_list.append(arts_types.SingleScatteringData.load(fname))
        scat_data=arts_scat.combine(scat_data_list,self.pnd_vec)
        scat_data.save(self.scat_file)
        return [self.scat_file]

    def pnd_calc(self,LWC_field,IWC_field,T_field):
        """Calculate pnd-field.
        
        Calculate the pnd field associated with the scattering data
        calculated by the scat_calc method above.  Returns a list (length=1)
        of pnd fields.

        Input:
            LWC_field   LatLonGriddedField3 containing liquid water content
                        mass densities [g/m^3]
            IWC_field   LatLonGriddedField3 containing ice water content
                        mass densities [g/m^3]
                        (ignored)
            T_field     LatLonGriddedField3 containing temperature [K]
                        (ignored)
        Output:
            list with exactly one LatLonGriddedField3 for the liquid particle
            number density [/m^3]
            FIXME: equal to LWC_field??
        """
        # FIXME: how can this make sense?
        return [LWC_field]

    @property
    def aossd(self):
        return arts_types.ArrayOfSingleScatteringData(
            [arts_types.SingleScatteringData.load(self.scat_file)])

    @property
    def aosmd(self):
        return arts_types.ArrayOfScatteringMetaData(
            [arts_types.ScatteringMetaData(
                type="Water",
                shape="spherical",
                density=1e3, 
                d_max=2*self.rc*1e-6, 
                V=4.0/3.0*math.pi*(self.rc*1e-6)**3,
                A_projec=2*self.rc*1e-6*math.pi, 
                asratio=1,
                description="ScatteringMetaData generated with PyARTS")
            ])


class Crystal(object):
    """Produces scattering data and pnd fields for ice clouds.

    The size distribution is the McFarquhar- Heymsfield 1997 distribution
    (see `clouds.mh97`_, as
    used in the EOSMLS cloudy-sky forward model.  A Crystal object is initialised
    with the particle parameters ptype, aspect_ratio, NP, which have the same
    meaning as in arts_types.SingleScatteringData objects. equivalent particle
    radii and pnd values are determined from the abscissas and weights for an
    *npoints* Laguerre Gauss
    quadrature, to give *npoints* arts_types.SingleScatteringData objects, and
    a pnd field corresponding to these particles.  The methods scat_calc and
    pnd_calc are called by the parent Cloud object.

    Parameters
    ~~~~~~~~~~

    ptype: integer
        As for arts_types.SingleScatteringData

    aspect_ratio: float
        As for arts_types.SingleScatteringData

    NP: integer
        As for arts_types.SingleScatteringData

    npoints: ? (FIXME GH 2011-03-25: what is this?)

    T_grid: 1D-array
        As for arts_types.SingleScatteringData
    """

    def __init__(self,ptype,aspect_ratio,NP, npoints=10, T_grid=[215,272.15]):
        """A Crystal object is initialised with the particle parameters ptype,
        aspect_ratio, NP, which have the same meaning as in
        arts_types.SingleScatteringData objects. equivalent particle radii and
        pnd values are determined from the abscissas and weights for an
        *npoints* Laguerre Gauss quadrature.

        See also the class docstring.
        """
        self.ptype=ptype
        self.aspect_ratio=aspect_ratio
        self.NP=NP
        self.a=0.02#this can be tuned
        #First get Laguerre Gauss weights and abscissa
        A=scipy.special.laguerre(n=npoints)
        x=numpy.real(A.weights[:,0])
        self.w=numpy.real(A.weights[:,1])

        self.r=x/2/self.a
        self.T_grid=T_grid
        self.particle_masses=4*numpy.pi*self.r**3/3*1e-12*rho_ice#mass of each particle in g.

    def scat_calc(self,f_grid,  za_grid=numpy.arange(0,181,10),
                  aa_grid=numpy.arange(0,181,10), num_proc=1):
        """Calculates single scattering data.

        Produces npoints scattering data objects, and returns a list
        (length=npoints) of scattering data files.
        
        Parameters
        ~~~~~~~~~~

        f_grid: 1D-array
            frequency grid for scattering calculations [Hz?]

        za_grid: 1D-array
            zenith angle grid [degree]

        aa_grid: 1D-array
            azimuth angle grid [degree]

        num_proc: int
            Number of processors to use in the calculation

        Returns
        ~~~~~~~

        scat_files: list of strings
            Each string contains the path to a
            arts-xml-file. The contents of those files describe the
            single-scattering-properties. The number of strings is
            determined by npoints (as passed to the constructor).
        """
        #generate single scattering data files
        scat_params={'f_grid':[f_grid],'T_grid':[self.T_grid],'NP':[self.NP],
                     'aspect_ratio':[self.aspect_ratio],'ptype':[self.ptype],
                     'phase':'ice','equiv_radius':self.r,'za_grid':za_grid,
                     'aa_grid':aa_grid}
        scat_files=arts_scat.batch_generate(scat_params,num_proc)
        return scat_files

    def pnd_calc(self,LWC_field,IWC_field,T_field):
        """Calculates particle number densities.
        
        Returns a list (length=npoints) of pnd fields (GriddedField3), given LWC,
        IWC and temperature fields (all GriddedField3)

        Parameters
        ~~~~~~~~~~

        LWC_field: LatLonGriddedField3
            Liquid water content field [g/m^3]

        IWC_field: LatLonGriddedField3
            Ice water content field [g/m^3]

        T_field: LatLonGriddedField3
            Temperature field [K]

        Returns
        ~~~~~~~

        pnd_list
            list of LatLonGriddedField3 containing particle number density
            data [g/m^3]. The number of pnd-fields is defined by npoints
            (see class documentation).

        """
        p_grid=IWC_field.p_grid
        lat_grid=IWC_field.lat_grid
        lon_grid=IWC_field.lon_grid
        pnd_list=[]
        n=mh97_vect(IWC_field.data,self.r,T_field.data)

        for ir in range(len(self.r)):
            r=self.r[ir]
            pnd=n[:,:,:,ir]*numpy.exp(2*self.a*r)/2/self.a*self.w[ir]
            pnd_list.append(arts_types.LatLonGriddedField3(p_grid,lat_grid,lon_grid,pnd))
        return pnd_list


class MonoCrystal(object):
    """A monodisperse ice particle.
    
    Produces scattering data and pnd fields for ice clouds with particles of
    only one size.

    Parameters
    ~~~~~~~~~~

    ptype: integer
        As for arts_types.SingleScatteringData

    aspect_ratio: float
        As for arts_types.SingleScatteringData

    NP: integer
        As for arts_types.SingleScatteringData

    equiv_radius: float
        As for arts_types.SingleScatteringData

    T_grid: 1D-array
        As for arts_types.SingleScatteringData
    """
    def __init__(self,ptype,aspect_ratio,NP, equiv_radius, T_grid=[215,272.15]):
        """A MonoCrystal object is initialised with the particle parameters ptype,
        aspect_ratio, NP, and equiv_radius which have the same meaning as in
        arts_types.SingleScatteringData objects.

        See also the class docstring.
        """
        self.ptype=ptype
        self.aspect_ratio=aspect_ratio
        self.NP=NP
        self.equiv_radius=equiv_radius
        self.T_grid=T_grid
        self.particle_masses=4*numpy.pi*equiv_radius**3/3*1e-12*rho_ice#mass of each particle in g.

    def scat_calc(self,f_grid,  za_grid=numpy.arange(0,181,10),
                  aa_grid=numpy.arange(0,181,10), num_proc=1):
        """Calculate single scattering properties.
        
        Produces a scattering data object, and returns a list
        (length=1) of scattering data files.
        """
        #generate single scattering data files
        scat_params={'f_grid':f_grid,'T_grid':self.T_grid,'NP':self.NP,
                     'aspect_ratio':self.aspect_ratio,'ptype':self.ptype,
                     'phase':'ice','equiv_radius':self.equiv_radius,'za_grid':za_grid,
                     'aa_grid':aa_grid}
        scat_data=arts_types.SingleScatteringData(scat_params)
        scat_data.generate()
        return [scat_data.filename]    

    def pnd_calc(self,LWC_field,IWC_field,T_field):
        """Returns a list (length=1) of pnd fields.
        """
        p_grid=IWC_field.p_grid
        lat_grid=IWC_field.lat_grid
        lon_grid=IWC_field.lon_grid
        pnd_list=[]
        data=numpy.zeros([len(p_grid),len(lat_grid),len(lon_grid)],float)
        V=4/3*numpy.pi*self.equiv_radius**3#microns^3

        for ip in range(len(p_grid)):
            for ilat in range(len(lat_grid)):
                for ilon in range(len(lon_grid)):
                    data[ip,ilat,ilon]=IWC_field.data[ip,ilat,ilon]/rho_ice/V*1e12#m^-3
            pnd_list.append(GriddedField3(p_grid,lat_grid,lon_grid,data))
        return [GriddedField3(p_grid,lat_grid,lon_grid,data)]


class Gamma(object):
    """Scattering data and pnd fields for ice or liquid.

    Produces scattering data and pnd fields for a gamma distribution of either
    ice or liquid water clouds.  The size distribution function is given by
    :m:`$n(r)=N_0 \\frac{(\\frac{r}{\\beta})^{\gamma-1}\exp(-\\frac{r}{\\beta})}{\\beta\Gamma(\gamma)}$`
    ,where :m:`$\gamma$` is the shape parameter, and :m:`$\\beta=\\frac{r_{eff}}{\gamma+2}$`

    A Gamma object is initialised with the distribution parameters
    r_eff (effective radius), and
    shape parameter g(:m:`$\gamma$`).  *T_grid* is required, as
    is the phase argument ('ice','liquid').  These two arguments need
    to be consistent.
    Scattering properties are
    integrated over the size distribution using an *npoints* Laguerre Gauss
    quadrature, to give a one `arts_types.SingleScatteringData`_ object.  
    The pnd field is then simply scaled by the iwc_field (or) lwc_field.  The methods
    scat_calc and pnd_calc are called by the parent Cloud object.
    """

    def __init__(self,r_eff, g, phase='ice', T_grid=[215,272.15], npoints=5,
                 ptype=20,NP=-1,aspect_ratio=1.000001):
        """Do docstring yet!"""
        self.T_grid=T_grid
        self.phase=phase
        self.r_eff=r_eff
        self.g=g
        self.ptype=ptype
        self.NP=NP
        self.aspect_ratio=aspect_ratio
        #First get Laguerre Gauss weights and abscissa
        alpha=g+1.0
        A=scipy.special.genlaguerre(n=npoints,alpha=alpha)
        x=numpy.real(A.weights[:,0])
        w=numpy.real(A.weights[:,1])
        #turn the abscissa into radii
        beta=r_eff/(g+2.0)
        self.r=beta*x

        if self.phase=='ice':
            rho=rho_ice*1e6#convert from g/cm^3 to g/m^3
        elif self.phase=='liquid':
            rho=1e6
        else:
            raise ValueError,"invalid phase argument' must be 'ice' or 'liquid'"


        self.pnd_vec=3*w/4/numpy.pi/rho/x**2/beta**3*1e18/scipy.special.gamma(g+3)
        #self.particle_masses=4*pi*self.r**3/3*1e-18*rho
        self.particle_masses=numpy.array([1.0])

    def scat_calc(self,f_grid, za_grid=numpy.arange(0,181,10),
                  aa_grid=numpy.arange(0,181,10),num_proc=1):
        """The xx_grid variables have the same meaning as in
        arts_types.SingleScatteringData objects. num_proc determines the number
        of processes used to complete the task. Returns a list (length=1) of
        scattering data files"""
        self.scat_file=DATA_PATH+'/scat/GammaReff'+str(self.r_eff)+'g'+str(self.g)+\
            'NP'+str(self.NP)+'p'+str(self.ptype)+'ar'+\
            str(self.aspect_ratio)[:3]+\
            'f'+str(f_grid[0])[:3]+'-'+str(f_grid[-1])[:3]+'T'+\
            str(self.T_grid[0])+'-'+str(self.T_grid[-1])+self.phase+'.xml'
        #generate single scattering data files
        scat_params={'f_grid':[f_grid],'T_grid':[self.T_grid],'NP':[self.NP],
                     'aspect_ratio':[self.aspect_ratio],'ptype':[self.ptype],
                     'phase':self.phase,'equiv_radius':self.r,'za_grid':za_grid,
                     'aa_grid':aa_grid}
        scat_files=arts_scat.batch_generate(scat_params,num_proc)
        #Combine these to form a single scattering data object

        scat_data_list=[]
        for fname in scat_files:
            scat_data_list.append(arts_types.SingleScatteringData.load(fname))
        scat_data=arts_scat.combine(scat_data_list,self.pnd_vec)
        scat_data.save(self.scat_file)
        return [self.scat_file]

    def pnd_calc(self,LWC_field,IWC_field,T_field):
        """calculate the pnd field associated with the scattering data
        calculated by the scat_calc method above.  Returns a list (length=1)
        of pnd fields.  For the Gamma class, this just returns the IWC of LWC
        field"""
        if self.phase=='ice':
            return [IWC_field]
        else:
            return [LWC_field]




################################################################################
#some other functions

def scat_data_raw_calc(list_of_hydrometeors,f_grid,num_proc=1):
    """Creates an arts useable scat_data_raw"""
    scatfiles=[]
    for hydrometeor in list_of_hydrometeors:
        scatfiles.extend(hydrometeor.scat_calc(f_grid,num_proc=num_proc))
    scat_data_list=[]
    for f in scatfiles:
        scat_data_list.append(arts_types.SingleScatteringData.load(f))
    return arts_types.ArrayOfSingleScatteringData(scat_data_list)

