%------------------------------------------------------------------------------
% NAME:     qpcls
%
%           Performs an inversion by constrained least squares.
%
%           However, the only method handled so far is OEM.
%
%           The function can also be used to calculate the contribution
%           function matrix and the weighting function matrix (the second 
%           format below). The number of input argument shall then be 1.
%
%           The fields of X and G are preliminary and can be changed. A
%           short description of X and G:
%
%           X is array of structures. There is one structure for each
%           retrieval identity (species profile etc.). The structure fields
%           depend on the retrieval quantity. For species, the retrieved
%           profile is both given in VMR (field x) and in fraction to the
%           a priori profile (field xfrac).
%
%           G gives some information about the sucess and progress of the 
%           inversion. It is a structure with the fields:
%              converged   1 if convergence was reached, else 0.
%              failure     Description why iteration was stoped (only if
%                          converged=0).
%              cost        Final total cost (chi sqaure).
%              cost_x      Final value of cost corresponding to x
%              cost_y      Final value of cost corresponding to y
%                          All cost values are normalised to the length of y.
%              niter       Number of iterations performed.
%              start_ga    Start value of the ML regularisation parameter.
%              stop_ga     Stop value of the ML regularisation parameter.
%
%           S is a structure giving error estimates, the measurement 
%           response etc. The errors are estimated by the used Se and Sx.
%           That is, only error sources flagged as level 2 are included.
%           For a more detailed error estimation, use qpcls_invchar.
%           The structure has the fields:
%              si_tot      Standard deviation for total error.
%              si_obs      Standard deviation for observation error.
%              si_smo      Standard deviation for smoothing error.
%              mresp       Measurement response.
%	       A           Averaging kernel matrix
%          unit_avk    The units for the avk matrix. Depending on the
%                      actual setup this may be 1./xa, or vmr. The unit
%                      1./xa corresponds to the retrieval of the
%                      normailized state vector (x/xa).
%	       S_obs       Covariance matrix for the observation error.
%	       S_smo       Covariance matrix for the smoothing error.
%
% FORMAT:   [X,G,y_corrected,y_fitted,bl,Xiter,S, ...  
%                             Dy,Kx,kx_names,kx_index,kx_aux] = qpcls( Q, y )
%             or
%           [Dy,Kx,kx_names,kx_index,kx_aux] = qpcls( Q ) 
%
% OUT:      X             See above.
%           G             See above.
%           y_corrected   The measurement spectrum (Y) corrected according
%                         to retrieved calibration and baseline parameters.
%           y_fitted      Fitted (to Y_CORRECTED) spectrum.
%           bl            Fitted baseline.
%           Xiter         The state vector for each teration. No unit
%                         conversions have been performed and e.g. species
%                         profiles are in fraction of a priori profile.
%           S             Error estimates and measurement response. See above.
% IN:       Q             Qpack structure.
%           y             Measurement spectrum.
%------------------------------------------------------------------------------

% HISTORY: 2001.01.04  Started by by Patrick Eriksson.
%          2003.11.07  Updated by Sho Tsujimaru.

function [X,G,y_corrected,y_fitted,bl,Xiter,S, ...
                               Dy,Kx,kx_names,kx_index,kx_aux] = qpcls( Q, y )

% Remember to adjust the checks of nargin and nargout below if argument
% lists are changed.

if ~qpcls_methods(Q.RETRIEVAL_METHOD)
  error( sprintf('You have selected an unknown retrieval method (%s)', ...
                                                         Q.RETRIEVAL_METHOD));
end

out(1,1);


numstable = 0;


%=== CALCULATION OPTIONS
%
%= Set 0 as first guess for the calculation flags.
nonlin       = 0;
return_Dy    = 0;  
do_post      = 0;
absspecies   = 0;
unitvmr = 0;
unitfrac = 1;
logspecies   = repmat( 0, sstring_length(Q.RETRIEVAL_TAGS), 1 );
absspecies   = 0;
unitvmr = 0;
unitfrac = 1;
do_xiter     = 0;
do_errorchar = 0;
recalc_abs   = 0;
%
%= Set the number of iterations between WFs recalculation to 1.
recalc_wfs_niter = 1;
%


if (nargin<2) | isempty(y)
  %
  return_Dy = 1;  
  %
  out(2,'ONLY CALCULATING Dy');
  %
else
  %
  nonlin = Q.CLS_NONLIN_ON;
  %
  if nargout > 1
    do_post = 1;
  end
  %
  if nonlin 
    %
    if strcmpi(Q.CLS_WFS_UNIT, 'vmr') 
      unitvmr = 1;
      unitfrac = 0;
    end
    
    % POS_ON has precedence !!!
    if Q.CLS_ABS_SPECIES_ON & isempty(find(Q.CLS_SPECIES_POS_ON))
      absspecies = 1;
      Q.CLS_WFS_UNIT = 'vmr';
      unitvmr = 1;
      unitfrac = 0;
    end

    if ~( length(Q.CLS_SPECIES_POS_ON) == 1  |  ...
          length(Q.CLS_SPECIES_POS_ON) == sstring_length(Q.RETRIEVAL_TAGS) )
      error( [ 'The field CLS_SPECIES_POS_ON must have length 1 or match ', ...
               'the number of retrieval tag groups.' ] );
    end
    %
    if sstring_length(Q.RETRIEVAL_TAGS)
      if length(Q.CLS_SPECIES_POS_ON) == 1
        logspecies = repmat( Q.CLS_SPECIES_POS_ON, ...
                             sstring_length(Q.RETRIEVAL_TAGS), 1 );
      else
        logspecies = Q.CLS_SPECIES_POS_ON;
      end
    end
    %
    if Q.CLS_RECALC_ABS_ON | Q.TEMPERATURE_DO == 3
      recalc_abs = 1;
    end
    %
    if Q.CLS_RECALC_WFS_NITER > 1
      recalc_wfs_niter = Q.CLS_RECALC_WFS_NITER;
    end
    %
  end
  %
  out(1,'RETRIEVAL BY CONSTRAINED LEAST SQUARES');
end
%
if nargout > 5
  do_xiter = 1;
end
%
if nargout > 6
  do_errorchar = 1;
end

 

%=== SOME PRINTING
%
out(2,0);
%
if nonlin
  if Q.CLS_GA_START_VALUE == 0
    out(2,sprintf('Method         : %s, Gauss-Newton', ... 
                                                  upper(Q.RETRIEVAL_METHOD) ));
  else
    out(2,sprintf('Method         : %s, Marquardt-Levenberg', ...
                                                   upper(Q.RETRIEVAL_METHOD)));
  end
  out(2,sprintf('Start gamma    : %.1f', Q.CLS_GA_START_VALUE ));
  out(2,sprintf('Factor OK      : %.1f', Q.CLS_GA_FAC_WHEN_OK ));
  out(2,sprintf('Factor not OK  : %.1f', Q.CLS_GA_FAC_WHEN_NOT_OK ));
  out(2,sprintf('Stop value     : %.3f', Q.CLS_STOP ));
  out(2,sprintf('Max iterations : %.0f', Q.CLS_MAX_ITER ));
  %
  if recalc_abs
    out(2,'   Absorption will be calculated for each iteration.');
  else
    out(2,'   Absorption will be scaled by species abundance.');
  end
  %
  if recalc_wfs_niter > 1
    out(2,sprintf(...
            '   There will %d iterations between recalculations of WFs.',...
                                                            recalc_wfs_niter));
  end
  %
  if any (logspecies)
    out(2,['   Positive constraint for species profiles is turned ' ...
           'on.']);
  elseif absspecies
    out(2,['   Absolute vmr profiles is turned will be ' ...
           'retrieved.']);
  
  else
    out(2,'   No constraint for species profiles.');
  end
  out(2,sprintf('   The weighting functions are calulated in %s', ...
                Q.CLS_WFS_UNIT')) ;
else
  out(2,sprintf('Method : Linear %s', upper(Q.RETRIEVAL_METHOD) ));
end
%
if ~return_Dy
  if do_post
    out(2,'   Spectrum, cost etc. will be calculated after retrieval.');
  else
    out(2,'   No post-retrieval calculations will be performed.');
  end
end



%=== LOAD H MATRICES
%
load( [Q.OUT,'.h'], '-mat' );
load( [Q.OUT,'.hd'], '-mat' );
%
if ~return_Dy & size(H,1)>1 & length(y) ~= size(H,1)
  error('Wrong length of input measurement vector.');
end


%=== CREATE TEMPORARY DIRECTORY AND RUN ARTS FOR FIRST ITERATION
%
tmpdir    = temporary_directory( Q.TMP_AREA );
%
do_nonlin = nonlin | do_post;
%
[ fy, Kx, kx_names, kx_index, kx_aux, kx_aux_abs, ...
                                        Kb, kb_names, cfile, basename] = ...
                        qp_iter1( Q, H, Hd, tmpdir, 3, do_nonlin, recalc_abs);



%
wfs_done = 1;  % Flag to indicate that WFs are (re)calculated
%
%
%---
%--- remove channels which do not fulfill the selection criteria ...
%
n_remove_index=0; % by default, no channels are removed.
if isfield(Q,'VALID_CHANNELS')
  if strcmp(Q.VALID_CHANNELS,'Tb') & ~isempty(Q.VALID_CHANNELS_RANGE)
    if length(Q.VALID_CHANNELS_RANGE) == 2 
      remove_index=find((y<Q.VALID_CHANNELS_RANGE(1)) | (y>Q.VALID_CHANNELS_RANGE(2)));
      n_remove_index = size(remove_index,1);
      if n_remove_index > 0  
        y(remove_index) = 0.;       % measured channels
        fy(remove_index) = 0.;      % forward model channels
        Kx(remove_index,:) = 0.;    % weighting functions
      end
    end
  end
end
%---
%---
%
if isempty( fy )
  [X,G,y_corrected,y_fitted,bl,Xiter,S] = arts_failure;
  return
else
  yfilename = [ basename, '.y', qp_hdf(Q) ];
end
%
if ~return_Dy & length(y) ~= length(fy)
  error('Wrong length of input measurement vector.');
end
%
%= Some sizes
xa     = kx_aux(:,2);
nx     = length( xa );
ny     = size( Kx, 1 );

%=== LOAD COVARIANCE MATRICES
%
load( [Q.OUT,'.dxinv'], '-mat' );
load( [Q.OUT,'.seinv'], '-mat' );
%
if strcmp( lower( Q.RETRIEVAL_METHOD ), 'oem' )
  load( [Q.OUT,'.sxinv'], '-mat' );
  RM = Sxinv;
  clear Sxinv
end
%
if numstable
  load( [Q.OUT,'.sx'], '-mat' );
  load( [Q.OUT,'.se'], '-mat' );
  Sxinvsqrt = chol(Sx)' \ speye(size(Sx));
  Seinvsqrt = chol(Se)' \ speye(size(Se));
  clear Se Sx 
end
%
if size(Kx,2) ~= size(RM,1)
  error(['Sizes of Kx and regularization matrix (Sx) do not match', ...
					      ' (recalculation needed?).'] );
end 


%=== RETURN HERE IF ONLY Dy AND Kx SHALL BE CALCULATED
%
if  return_Dy
  %
  KtSeinv     = Kx'*Seinv;
  X           = ( (RM + KtSeinv*Kx ) \ speye(size(RM)) ) * KtSeinv;
  G           = Kx; 
  y_corrected = kx_names;
  y_fitted    = kx_index;
  bl          = kx_aux;
  %
  out(1,-1);
  if out(1)
    fprintf('\n');                  % NOTE NOTE NOTE
  end                               %
  return                            %>>>>> Exit >>>>>
end                                 %


%=== SOME SPECIAL STUFF FOR NON-LINEAR
%
if nonlin

  gamma  = Q.CLS_GA_START_VALUE;

  %= Build cfile for calculation of Kx
  template = which( 'cls_iter2_kx.tmplt' );
  %
  QE            = [];
  QE.RECALC_ABS = recalc_abs;
  QE            = qp_hdf( Q, QE );
  %
  qtool( Q, tmpdir, template, QE );
  cfile_kx  = [ cfile, '-FOR_KX' ];
  copyfile( cfile, cfile_kx );

  %= Index for species with or without positive constrain
  %
  ilinsp = [];
  ilogsp = [];
  for i = 1 : size( kx_names, 1 )
    group = qp_kinfo( kx_names{i} );
    if strcmp( group, 'Species' ) 
      if logspecies(i)
        ilogsp = [ ilogsp; (kx_index(i,1):kx_index(i,2))' ];
      else
        ilinsp = [ ilinsp; (kx_index(i,1):kx_index(i,2))' ];
      end
    end
  end

  %= Take log of species if positive constraint
  %
  if any(logspecies)
    xa(ilogsp) = log( xa(ilogsp) );
  end

  % If the Kx matrix is calculated in vmr units and the
  % retrieved quantity is x/x_a calculate Kx in units of x_a
  %= Index for species profiles
  %
  if unitvmr & ~absspecies & ~any(logspecies)
    Kx(:,ilinsp) = Kx(:,ilinsp) .* (ones(ny,1)* xa(ilinsp)');
    % Set the a priori to ones and keep the original a priori in
    % x_apriori
    x_apriori = xa;
    xa(ilinsp) = ones(size(xa(ilinsp)));
  end


  

%=== SOME SPECIAL STUFF FOR LINEAR
%
else
  gamma = 0;
end


%=== SOME SPECIAL STUFF FOR NON-LINEAR AND/OR POST CALCULATIONS
%
if nonlin | do_post

  %= Build cfile for calculation of FY 
  template = which( 'cls_iter2_y.tmplt' );
  %
  QE            = [];
  QE.RECALC_ABS = recalc_abs;
  QE            = qp_hdf( Q, QE );
  %
  qtool( Q, tmpdir, template, QE );
  cfile_y  = [ cfile, '-FOR_Y' ];
  copyfile( cfile, cfile_y );

  %= Start cost
  %
  cost_x = 0;
  a      = y - fy;
  cost_y = (a' * Seinv * a) / (ny-n_remove_index);
  cost   = cost_x + cost_y;

  %= Printing
  out(2,0);
  out(1,sprintf(...
          ' Iter              Total    Profile    Spectrum   Conver-   WFs'));
  out(1,sprintf(...
          '  nr    Gamma      cost      cost       cost       gence   calc.'));
  out(1,0);
  out(1,sprintf('%4.0f      -      %.2e      0.0     %.2e       -      -',...
                                                         0, cost_y, cost_y ));

  %= Load needed variables
  %
  %= Absorption
  if ~recalc_abs
    Abs  = read_artsvar( basename, 'abs');
    Absp = read_artsvar( basename, 'abs_per_tg');
  else
    Abs  = [];
    Absp = [];
  end
  %
  %= Zenith angles
  if Q.POINTING_DO == 3
    za_pencil = read_artsvar( basename, 'za_pencil');
  else
    za_pencil = [];
  end
end

%=== SOME SPECIAL STUFF FOR NON-LINEAR 
%

% Copy original database as backup.

if Q.PSF_DO == 3
 fname   = fullfile( Q.SPECTRO_DIR,  'backup_linefile');
 copyfile( fullfile( Q.SPECTRO_DIR, Q.LINEFILE ), fname );
end



%=== READ SOME ARTS VARIABLES THAT ALWAYS ARE NEEDED
%
p_abs  = read_artsvar( basename, 'p_abs' );
f_mono = read_artsvar( basename, 'f_mono' );
vmrs   = read_artsvar( basename, 'vmrs' );


%=== INIT ITERATION VARIABLES 
%
iteration   = 0;
if nonlin
  max_iter  = Q.CLS_MAX_ITER;
else
  max_iter  = 1;
end
%
x           = xa;
y_corrected = y;
%
if do_xiter
  Xiter     = [];
end
%
G           = [];
G.converged = 0;
min_ga      = Inf;
if nonlin
  G.failure   = 'max iterations reached';    % default assumption
  G.start_ga  = gamma;
else
  G.failure   = '- (linear inversion)';
  G.start_ga  = NaN;
  G.stop_ga   = NaN;
end


%##############################################################################
%### Start of retrieval loop
%##############################################################################
%
while ~G.converged & ( iteration < max_iter )


  iteration = iteration + 1;
  G.niter   = iteration;


  %=== Calculate new Kx ? 
  %
  if iteration > 1  &  ~rem( iteration-1, recalc_wfs_niter )
    %
    copyfile( cfile_kx, cfile );
    qp_arts( Q, cfile ); 
    %
    Kx = qp_assemble_K( Q, basename, H, Hd, y, bl, 3, x, iteration, QE);
    %
    wfs_done = 1;
    %    
    %    
    %---
    %--- remove channels which do not fulfill the selection criteria ... 
    %
    if isfield(Q,'VALID_CHANNELS')
      if strcmp(Q.VALID_CHANNELS,'Tb') & ~isempty(Q.VALID_CHANNELS_RANGE)
        if length(Q.VALID_CHANNELS_RANGE) == 2 
          if n_remove_index > 0 
            y(remove_index) = 0.;       % measured channels
            fy(remove_index) = 0.;      % forward model channels
            Kx(remove_index,:) = 0.;    % weighting functions
          end
        end
      end
    end
    %---
    %---
    %
    % The WFs shall corresponds to fractions of the a priori state, but the
    % WFs are calculated for present retrieved state and a rescaling is 
    % required if logspecies=0.
    %   The Kx to apply for the the log-retrieval of species is Kx*x (see 
    % Eq. 45 of Paper A in my thesis), but this is achieved automatically as 
    % Kx is calculated in fractions of x (and not xa).
    %
    if ~isempty(ilinsp) & ~absspecies & unitfrac
      Kx(:,ilinsp) = Kx(:,ilinsp) ./ (ones(ny,1)*x(ilinsp)');
    elseif ~isempty(ilinsp) & ~any(logspecies) & unitvmr & ~absspecies
      % If the Kx matrix is calculated in vmr units and the
      % retrieved quantity is x/x_a calculate Kx in units of x_a
      Kx(:,ilinsp) = Kx(:,ilinsp) .* (ones(ny,1)* x_apriori(ilinsp)');
    end
  end

  
  %=== Make inversion
  %
  % Gaussian elimination (\) is used as this is faster than calculating the
  % inverse explicitely (by inv()).
  %
  KtSeinv = Kx'*Seinv;
  AAAA    = RM + KtSeinv*Kx;
  %
  ok_iter = 0;
  %
  while ~ok_iter

    %= If linear, set ok flag for iteration to true
    if ~nonlin
      ok_iter     = 1;
    end

    lastwarn( '' );    % Clear warning stack
    
  if ~numstable
    %= A vector needed in different places
    if iteration == 1
      wdy = KtSeinv * ( y_corrected - fy );
    else
      wdy = KtSeinv * ( y_corrected - fy ) - RM * ( x - xa );
    end

    %= New x
    %
    if gamma == 0
      x_new = x + AAAA \ wdy;
    else
      x_new = x + (AAAA + gamma*Dxinv) \ wdy; 
    end

  else
    Dx        = (gamma*Dxinv)\speye(size(Dxinv));
    Dxinvsqrt = chol(Dx)' \ speye(size(Dxinv));
    x_new     = x + [Seinvsqrt*Kx;Sxinvsqrt;Dxinvsqrt] \ ...
          [Seinvsqrt*(y_corrected - fy);-Sxinvsqrt*(x-xa);zeros(size(RM,1),1)];
  end

    %= Exit if calculations above have caused any warnings, probably caused
    %  by singular or badly scaled matrix inversion.
    if ~Q.QPACK_WARNING_IGNORE & ~isempty( lastwarn )
      out(1,'Singular or badly scaled matrix inversion. Exiting ...');
      [X,G,y_corrected,y_fitted,bl,Xiter,S] = qpack_failure;
      out(1,-1);
      return
    end

  
    %= Check if any temperatures outside defined limits  
    %
    itemp = find( strncmp( 'Temperature:', kx_names, 12 ) );
    %
    if ~isempty( itemp )
      iit              = kx_index(itemp,1) : kx_index(itemp,2);
      xxt = x_new(iit);
      iiti             = find( x_new(iit) < Q.TEMPERATURE_RANGE(1) );
      x_new(iit(iiti)) = Q.TEMPERATURE_RANGE(1);
      iiti             = find( x_new(iit) > Q.TEMPERATURE_RANGE(2) );
      x_new(iit(iiti)) = Q.TEMPERATURE_RANGE(2);
    end
    
  
    %= Calculate spectrum corresponding to X_NEW and calculate cost function
    if nonlin | do_post
 
      %= Map X onto absorption etc.
      qp_x2arts( Q, x_new, basename, Kx, kx_names, kx_index, kx_aux, ...
           recalc_abs, logspecies, p_abs, f_mono, vmrs, Abs, Absp, za_pencil );

      %= New fy (delete first the old file to detect ARTS crasches)
      delete( yfilename );
      copyfile( cfile_y, cfile );
      qp_arts( Q, cfile ); 
      if exist( yfilename, 'file' )
        fy_new = read_artsvar( basename, 'y' );
        fy_new = H*fy_new;
      else
        [X,G,y_corrected,y_fitted,bl,Xiter,S] = arts_failure;
        return
      end

      %= Correct measurement spectrum according to retrieved calibration and 
      %= baseline variables 
      [y_corrected,bl] = qp_x2y( Q, x_new, Kx, kx_names, kx_index, y );

      %---
      %--- remove channels which do not fulfill the selection criteria ... 
      %
      if isfield(Q,'VALID_CHANNELS')
        if strcmp(Q.VALID_CHANNELS,'Tb') & ~isempty(Q.VALID_CHANNELS_RANGE)
          if length(Q.VALID_CHANNELS_RANGE) == 2 
            if n_remove_index > 0 
              y_corrected(remove_index) = 0.; % measured (baseline corrected) channels
              fy_new(remove_index) = 0.;      % reproduced channels
            end
          end
        end
      end
      %---
      %---

      %= New cost
      a        = x_new - xa;
      cost_x   = (a' * RM * a) / (ny-n_remove_index);  
      a        = y_corrected - fy_new;
      cost_y   = (a' * Seinv * a) / (ny-n_remove_index); 
      cost_new = cost_x + cost_y;

    end

    %= If non-linear, check if cost has decreased
    if nonlin

      if cost_new < cost 
        ok_iter = 1;

      else
        out(1,sprintf('FAILED %.1e   %.2e', gamma, cost_new ));

        if gamma > Q.CLS_GA_UPPER_LIMIT
          ok_iter   = 1;     
          iteration = 9999;
          G.failure = 'runaway gamma';
        elseif gamma > 0
          gamma = gamma * Q.CLS_GA_FAC_WHEN_NOT_OK;
        else
          gamma = Q.CLS_GA_FAC_WHEN_NOT_OK;
        end
      end
    end

  end % while ~ok_iter

  if nonlin | do_post 

    %= Calculate convergence number
    %
    % (Equation 5.31 in the Rodgers book was used here but it didn't work
    % well for some harder inversion cases)
    %
    conv = ( ( x_new - x )' * AAAA * ( x_new - x ) ) / nx;
    %
    if wfs_done
      wfs_symbol = 'y';
      wfs_done = 0;
    else
      wfs_symbol = 'n';
    end
    %
 
    out(1,sprintf('%4.0f   %.1e   %.2e   %.2e   %.2e   %.2e   %s',...
    	       iteration, gamma, cost_new, cost_x, cost_y, conv, wfs_symbol ));
    
    %= Fill G if ready with non-linear inversion
    if nonlin & ( conv < Q.CLS_STOP )
      G.converged = 1;
      G.failure   = [];
    end 

    %= Update spectrum and cost
    fy   = fy_new;
    cost = cost_new;

    %= Update spectrum and cost, if not ready
    if ~G.converged & nonlin
      gamma   = gamma / Q.CLS_GA_FAC_WHEN_OK;
    end
  end

  %= Move X_NEW to X
  x = x_new;

  if do_xiter
    Xiter = [Xiter, x];
  end

  if gamma < min_ga
    min_ga = gamma;
  end
end

%##############################################################################
%### End of retrieval loop
%##############################################################################


%=== Recover original database from backup.
%
if Q.PSF_DO == 3
 fname   = fullfile( Q.SPECTRO_DIR,  'backup_linefile' );
 copyfile( fname, fullfile( Q.SPECTRO_DIR, Q.LINEFILE ) );
 delete(fname);
end


%=== Clear some variables to save some memory
%
clear RM Seinv H Hd
%
if ~do_errorchar
  clear AAAA KtSeinv
end



%=== IF LOG-SPECIES, CONVERT BACK
%
if any( logspecies  )
  x(ilogsp) = exp( x(ilogsp) );
end


%=== FILL X STRUCTURE
%
[X,unitfac] = ...
     qp_x2X( x, Q, tmpdir, basename, kx_names, kx_index, kx_aux, p_abs, vmrs );


%=== FIX OTHER RETURN VARIABLES
%
if do_post
  %
  y_fitted = fy;
  %
  G.cost    = cost;
  G.cost_x  = cost_x;
  G.cost_y  = cost_y;
  G.min_ga  = min_ga;
  G.stop_ga = gamma;
end



%=== DELETE THE TEMPORARY DIRECTORY
%
delete_tmp_dir( Q.TMP_AREA, tmpdir );



%=== Make error characterisation
%
if do_errorchar
  
  % Contribution function matrix
  if ~numstable
    Dy = ( AAAA \ speye(size(AAAA)) ) * KtSeinv;
  else
    Dy = pinv([Seinvsqrt*Kx;full(Sxinvsqrt)]) * [Seinvsqrt;sparse(nx,ny)];
  end
  clear AAAA KtSeinv

  % Observation error
  load( [Q.OUT,'.se'], '-mat' );
  S.S_obs = (unitfac*unitfac') .* (Dy * Se * Dy');
  clear Se
  
  % Rescaling needed if log-species used
  if any( logspecies )
    S.S_obs(ilogsp,ilogsp) = (x(ilogsp)*x(ilogsp)') .* S.S_obs(ilogsp,ilogsp);
  end

  % Averaging kernel matrix
  A   = Dy*Kx;
  S.A = A;
  S.unit_avk = unitfac;

  % Measurement response
  S.mresp = zeros(nx,1);
  for i = 1 : size( kx_index, 1 )
    ind = kx_index(i,1) : kx_index(i,2);
    S.mresp(ind) = sum( A(ind,ind)' )';
  end

  % Smoothing error
  load( [Q.OUT,'.sx'], '-mat' );
  A       = A - eye(nx,nx);
  S.S_smo = A * Sx * A';
  clear Sx A 

  % Standard deviations
  S.si_obs = sqrt( diag( S.S_obs ));
  S.si_smo = sqrt( diag( S.S_smo ));
  S.si_tot = sqrt( S.si_obs.^2 + S.si_smo.^2 );

  % Store kx_index in S
  S.kx_index = kx_index;

end


%=== END MESSAGES
out(1,-1);
if out(1)
  fprintf('\n');
end





%--- Sub-functions -----------------------------------------------------------

function [X,G,y_corrected,y_fitted,bl,Xiter,S] = arts_failure
  %
  G.failure = 'ARTS crash';
  G.cost    = NaN;
  G.cost_x  = NaN;
  G.cost_y  = NaN;
  %
  X           = [];
  y_corrected = [];
  y_fitted    = [];
  bl          = [];
  Xiter       = [];
  S           = [];
  % 
return



function [X,G,y_corrected,y_fitted,bl,Xiter,S] = qpack_failure
  %
  [X,G,y_corrected,y_fitted,bl,Xiter,S] = arts_failure;
  G.failure = 'Qpack crash (singular matrix inversion)';
  % 
return

  


