%------------------------------------------------------------------------
% NAME:     qp_abs
%
%           A function to calculate the absorption corresponding to a Q 
%           structure, with the possibility to replace F_MONO and P_ABS. 
%
%           The optional arguments F_MONO and P_ABS can be empty, and
%           then original vectors are used.
%
%           Not all fields of Q must be set. 
%
% FORMAT:   [ Abs, f_mono, PTZ, Abs_per_tg ] = qp_abs( Q [, f_mono, p_abs ] )
%
% OUT:      Abs            Calculated absorption.
%           f_mono         Monochromatic frequencies.
%           PTZ            A matrix with p_abs, t_abs and z_abs as columns.
% IN:       Q              Setting structure.
% OPTIONAL: f_mono         Alternative monochromatic frequency grid.
%           p_abs          Alternative absorption pressure grid.
% TEMPLATE: basic.tmplt
%------------------------------------------------------------------------

% HISTORY: 2002.01.17  Created by Patrick Eriksson.


function [Abs,f_mono,PTZ,Abs_per_tg] = qp_abs( Q, f_mono, p_abs )


%=== Create temporary dir
%
tmpdir    = temporary_directory( Q.TMP_AREA );


%=== If necessary, use TMPDIR as CALCGRIDS_DIR
%
if nargin > 1
  %
  %= Copy all .aa and .ab files to TMPDIR
  D = dir( fullfile( Q.CALCGRIDS_DIR, '*.a*' ) );
  for i = 1:length(D)
    copyfile( fullfile( Q.CALCGRIDS_DIR, D(i).name), ...
                                               fullfile( tmpdir, D(i).name ) );
  end
  %
  Q.CALCGRIDS_DIR = tmpdir;
  %
  if exist('f_mono') & ~isempty(f_mono)
    write_datafile( fullfile( Q.CALCGRIDS_DIR, Q.F_MONO), f_mono, 'VECTOR' );
  end
  %
  if exist('p_abs') & ~isempty(p_abs)
    write_datafile( fullfile( Q.CALCGRIDS_DIR, Q.P_ABS ), p_abs, 'VECTOR' );
  end
end


%=== Get full path of template file and create cfile
%
template = which( 'abs.tmplt' );
%
if nargout > 3
  QE.ABS_PARTS = 1;
else
  QE.ABS_PARTS = 0;
end
%
QE = qp_hdf( Q, QE );
%
[cfile,basename] = qtool( Q, tmpdir, template, QE );


%=== Run ARTS
%
qp_arts( Q, cfile );


%=== Load spectrum
%
Abs = read_artsvar( basename, 'abs' );
Abs_per_tag = read_artsvar( basename, 'abs_per_tg' );
 z_abs = read_artsvar( basename, 'z_abs' );
if nargout > 1
  %
  f_mono = read_artsvar( basename, 'f_mono' );
  p_abs = read_artsvar( basename, 'p_abs' );
  t_abs = read_artsvar( basename, 't_abs' );
  z_abs = read_artsvar( basename, 'z_abs' );
  %
  np   = length( p_abs );
  PTZ  = zeros( np, 3 );
  PTZ(:,1) = p_abs;
  PTZ(:,2) = t_abs;
  PTZ(:,3) = z_abs;

  if QE.ABS_PARTS 
    Abs_per_tg = read_artsvar( basename, 'abs_per_tg' );
  end

end

%
if nargout < 1

  %- File type flag
  ftf = qp_hdf( Q );
  ftf = ftf(end);  
  
  write_artsvar(Q.OUT, 'abs', Abs, ftf );
  write_artsvar(Q.OUT, 'abs_per_tg', Abs_per_tag, ftf );
  write_artsvar(Q.OUT, 'z_abs',  z_abs, ftf );
end

%=== Delete the temporary directory
delete_tmp_dir( Q.TMP_AREA, tmpdir );
