%------------------------------------------------------------------------
% NAME:     qp_invchar
%
%           Performs a characterisation of the general retrieval performance
%           assuming linearity.
%
%           The following plots are produced if DO_PLOTS=1:
%
%           Measurement response 
%             The measurement response is the sum of the averaging kernels.
%             The sum is calculated only for the retrieval identity
%             sub-matrix. 
%
%           Total and observation error
%    
%           The FWHM of the averaging kernels
%             FWHM = full width at half maximum
%
%           The averaging kernels
%
%           Correlation FWHM
%             The distance over which the error correlation is >= 0.5.
%
%           Max/min correlation to other retrieval identities.
%             Correlation to all other altitudes etc. is considered.
%
% FORMAT:   [PTZ,measres,s_tot,s_obs,vertres,A,C] =
%                           qp_invchar( Q, Dy, Kx, kx_names, kx_index, kx_aux 
%                                                      [,do_plots, mr_limit] )
%
% OUT:      PTZ        Matrix with pressure, temperature and altitude.
%           measres    Measurement responses as a vector.
%           s_tot      Total retrieval error (sum of smoothing and observation
%                      errors).
%           s_obs      Observation error.
%           vertres    Vertical resolutions.
%           A          Averaging kernel matrix.
%           C          Error correlation matrix.
% IN:       Q          Setting structure.
%           Dy         Contribution function matrix
%           Kx         Weighting function matrix.
%           kx_index   Start and stop index for each retrieval identity.
%                      A 2 column matrix          
% kx_names   As the corresponding ARTS variable.
%           kx_aux     As the corresponding ARTS variable.
%           kx_aux_abs The same as kx_aux but in absolute values
% OPTIONAL: do_plots   Flag for doing plotting. Default is 1.
%           mr_limit   Do not include altitudes with a measurement response
%                      below this limit (only for plots). Default is 0.3.
%                      To set MR_LIMIT=0, can result in crash.
%------------------------------------------------------------------------

% HISTORY: 2001.08.23  Started by Patrick Eriksson
%          2001.10.14  Included do-level 1 in Se (PE)


function [PTZ, measres, s_tot, s_obs,  A,  C] = ...
    qp_invchar( Q, Dy, Kx, kx_names, kx_index, kx_aux, do_plots, mr_limit);


if ~exist('do_plots'),   do_plots = 1;   end
if ~exist('mr_limit'),   mr_limit = 0.3; end


%=== Some sizes
%
np  = size( Kx, 2 );
nrq = size( kx_names, 1 );


%=== Check that Kx and Dy have consistent sizes
%
if size(Dy,2)~=size(Kx,1) | size(Dy,1)~=np
  error('Sizes of Kx and Dy do not match. Recalculation needed?')
end


%=== Load Sx and check size
%
out(1,'Loading Sx ...');
%
load( [Q.OUT,'.sx'], '-mat' );
%
if size(Sx,1) ~= np
  error('Sizes of Kx and Sx do not match. Recalculation needed?')
end



%=== Calculate Se for do-levels 1 and 2
%
% Dy is here included in H and Hd to save memory.
% In this way, the observation error is obtained directly and the results
% is stored in what will be the error correlation matrix.
%
out(1,'Loading H matrices ...');
%
load( [Q.OUT,'.h'], '-mat' );
load( [Q.OUT,'.hd'], '-mat' );
%
out(1,'Calculating observation error ...');
%
H  = Dy * H;
Hd = Dy * Hd;
%
C = qp_Se( Q, H, Hd, [1 2] );
%
clear H Hd


%=== Calculate averaging kernels, errors and error correlation matrix.
%
out(1,'Calculating averaging kernels ...');
%
A	= Dy * Kx;
AI	= A - eye(size(A));


%== Calculate errors and error correlation
%
out(1,'Calculating total error ...');
%
s_obs	= sqrt( diag( C ));
C       = C + AI*Sx*AI';
s_tot   = sqrt( diag( C ));
%
out(1,'Calculating error correlation ...');
%
C       = C ./ (s_tot*s_tot');   
%
clear AI Sx


PTZ = read_datafile( Q.APRIORI_PTZ, 'MATRIX', 1 );
%=== Calculate measurement response
%
out(1,'Calculating measurement response ...');
%
z       = zeros(np,1);
measres = zeros(np,1);
vertres = zeros(np,1);
%
for i = 1:nrq
  ind = kx_index(i,1) : kx_index(i,2);
  measres(ind) = sum( A(ind,ind)' )';
end

%=== Return if not plots shall be made ===    %=== 
%                                             %===  
if ~do_plots                                  %=== Return ?
  return                                      %===
end                                           %=== 
out(1,'Preparing plots ...');


%=== Set general axes settings
%
A1 = axes_def; A2 = axes_def; A3 = axes_def;
A4 = axes_def; A5 = axes_def; A6 = axes_def;
%
A1.POSITION = [0.08 0.68 0.30 0.26];
A2.POSITION = [0.40 0.68 0.58 0.26];
A3.POSITION = [0.08 0.37 0.30 0.26];
A4.POSITION = [0.40 0.37 0.58 0.26];
A5.POSITION = [0.08 0.06 0.44 0.26];
A6.POSITION = [0.54 0.06 0.44 0.26];
%
A1.XGRID    = 'on';
A2.XGRID    = 'on';
A3.XGRID    = 'on';
A5.XGRID    = 'on';
A6.XGRID    = 'on';
%
A2.YTICKLABEL = [];
A4.YTICKLABEL = [];
A6.YTICKLABEL = [];



%=== First figure - 1
if isempty(get(0,'Children'))     % To handle the case when no window
  fig = 0;                        % is opened
else
  fig = gcf;
end



%=== Loop the retrieval identities ============================================
%
for i = 1:nrq


  %= Names of present retrieval identity
  [group,name] = qp_kinfo( kx_names{i} );
  %= Index in x of present retrieval identity
  ind = kx_index(i,1):kx_index(i,2);

  %= Get index for points with measurement response above limit
  ip = find( measres(ind) >= mr_limit );


  %= Do something only if there are points > mr_limit
  if isempty(ip)
    out(1,sprintf('Retrieval identity: %s', name ));
    out(1,'   all data below limit on measurement response');
  else

    %= Create figure
    fig = fig + 1;
    figure(fig);
    fig_size(18,24,'cm',0.8);


    %=== Scalar identity ======================================================
    if length( ip ) == 1

      row    = 0.016;
      fsize  = 10;
      y0     = 0.97;
      xoff   = 0.02;
      textw  = 0.35;

      set(gcf,'Unit','Cent');
      set(gca,'Position',[-0.01 -0.01 1 1]);
      set(gca,'Box','Off');
      set(gca,'TickLength',[0 0])
    
      irow = 1;
      text2( xoff, y0-irow*row, sprintf('%s',name),fsize*1.4,1);
  
      irow = irow+2;
      text2(xoff,y0-irow*row,'Measurement response',fsize,1);
      text2(xoff+textw,y0-irow*row,sprintf(': %.2f',measres(ind)),fsize,0);

      irow = irow+2;
      text2(xoff,y0-irow*row,'Total error',fsize,1);
      text2(xoff+textw,y0-irow*row,sprintf(': %.2e',s_tot(ind)),fsize,0);
      irow = irow+1;
      text2(xoff,y0-irow*row,'Observation error',fsize,1);
      text2(xoff+textw,y0-irow*row,sprintf(': %.2e',s_obs(ind)),fsize,0);

      irow = irow+2;
      text2(xoff,y0-irow*row,'Correlation with',fsize,1);
      irow = irow+2;
      for j = 1:nrq
        if j~=i
          nr    = j-(j>i);
          ind2 = kx_index(j,1):kx_index(j,2);
          [gg,nn] = qp_kinfo( kx_names{j} ); 
          text2(xoff,y0-irow*row,sprintf('%s',nn),fsize,0);
          if length(ind2) >= 2
            text2(xoff+textw,y0-irow*row,...
                          sprintf(': %.2f <-> %.2f',min(C(ind2,ind)),...
                                                    max(C(ind2,ind))),fsize,0);
          else
            text2(xoff+textw,y0-irow*row,sprintf(': %.2f',C(ind2,ind)),...
                                                                      fsize,0);
          end
          irow = irow + 1;
        end
      end


    %=== Vector identity ======================================================
    else

      ip = ind(min(ip)):ind(max(ip));

      %= Some plotting quantities
      %
      if strcmp( group, 'Species' )
        %
        z(ind)       = interpp( PTZ(:,1), PTZ(:,3), kx_aux(ind,1) );
        %
        yfac = 1000; ytext = 'Altitude [km]'; yunit = 'km';
        xfac = 100;  xunit = '%';
        %
      elseif strcmp( group, 'Temperature' )
	%
        z(ind)       = interpp( PTZ(:,1), PTZ(:,3), kx_aux(ind,1) );
        %
        yfac = 1000; ytext = 'Altitude [km]'; yunit = 'km';
        xfac = 1;    xunit = 'K';
        %    
      elseif strcmp( group, 'Continuum' )
	%
        z(ind)       = interpp( PTZ(:,1), PTZ(:,3), kx_aux(ind,1) );
        %
        yfac = 1000; ytext = 'Altitude [km]'; yunit = 'km';
        xfac = 1e-3; xunit = '1/m';    % Internal unit for x is here 1/km
        %    
      elseif strcmp( group, 'Ground emission' )
	%
        ne        = length( Q.EGROUND_LIMITS );
        f_mono    = read_datafile( fullfile( Q.CALCGRIDS_DIR, Q.F_MONO ), 'VECTOR' );
        z(ind)    = qp_eground_lims( f_mono, Q.EGROUND_LIMITS );
        %
        yfac = 1e9;  ytext = 'Frequency [GHz]'; yunit = 'GHz';
        xfac = 1;    xunit = '-';
        %    
      elseif strncmp( group, 'Baseline', 8 )
	%
        global EARTH_RADIUS
        %
        z(ind) = za2geomtan( EARTH_RADIUS, Q.PLATFORM_ALTITUDE, kx_aux(ind,1));
        %
        %= Revert IND and IP as tangent altitudes are coming in wrong order
        ind = flipdim(ind,2); 
        ip  = flipdim(ip,2); 
        %
        yfac = 1e3;  ytext = 'Geometrical ztan [km]'; yunit = 'km';
        xfac = 1;    xunit = 'K';
        %    
      elseif strcmp( name, 'Reference load temperatures' )
        %
        z(ind) = Q.TB_REFLOADS_NOMINAL;
        %
	yfac = 1;    ytext = 'Nominal value [K]'; yunit = 'K';
        xfac = 1;    xunit = 'K';
        %    
      else
	error(sprintf( 'Unknown retrieval identity: %s', name ));
      end

      %=== Figure 1 ===
      %
      A1.XLABEL   = 'Measurement response [%]';
      A1.YLABEL   = ytext;
      subplot(3,2,1)
      plot( measres(ip)*100, z(ip)/yfac );
      axis([mr_limit*100 119 0.99*min(z(ip))/yfac 1.01*max(z(ip))/yfac]);
      ax_set( A1 );
      

      %=== Figure 2 ===
      %
      A2.XLABEL   = ['Error [',xunit,']'];
      subplot(3,2,2)
      pline( s_tot(ip)*xfac, z(ip)/yfac, 1 ); 
      hold on
      pline( s_obs(ip)*xfac, z(ip)/yfac, 2 );
      axis([0 max(s_tot(ip)*xfac)*1.2 0.99*min(z(ip))/yfac 1.01*max(z(ip))/yfac])
      sleg = str2mat('Total','Observation');
      plineleg(sleg,0.65,0.7,0.12,10,0.07);
      ax_set( A2 );
      

      %=== Figure 3 ===
      %
      vertres(ind) = fwhm( z(ind), A(ind,ind) );
      %
      A3.XLABEL   = ['FWHM [',yunit,']'];
      A3.YLABEL   = ytext;
      subplot(3,2,3)
      plot( vertres(ip)/yfac, z(ip)/yfac )
      axis([0 max(vertres(ip)/yfac)*1.2 0.99*min(z(ip))/yfac 1.01*max(z(ip))/yfac])
      ax_set( A3 );
      

      %=== Figure 4 ===
      %
      A4.XLABEL   = 'Averaging kernels [-]';
      subplot(3,2,4)
      plot( A(ip,ip)', z(ip)/yfac )        % Transpose to plot the rows
      axis([-0.15 1.05 0.99*min(z(ip))/yfac 1.01*max(z(ip))/yfac]);
      hold on
      for j = ip
        plot( [-2 2], [z(j) z(j)]/yfac, ':' )
      end
      hold off
      ax_set( A4 );
      

      %=== Figure 5 ===
      %
      %A5.XLABEL   = ['Correlation FWHM [',yunit,']'];
      %A5.YLABEL   = ytext;
      %subplot(3,2,5)
      %cl = fwhm( z(ind), abs(C(ind,ind)) );
      %plot( cl/yfac, z(ind)/yfac )
      %axis([0 max(cl/yfac)*1.2 0.99*min(z(ip))/yfac 1.01*max(z(ip))/yfac])
      %ax_set( A5 );
      A5.XLABEL   = 'Correlation [-]';
      A5.YLABEL   = ytext;
      subplot(3,2,5)
      plot( C(ip,ip), z(ip)/yfac ) 
      axis([-0.99 1 0.99*min(z(ip))/yfac 1.01*max(z(ip))/yfac]);
      hold on
      for j = ip
        plot( [-2 2], [z(j) z(j)]/yfac, ':' )
      end
      hold off
      ax_set( A5 );
      

      %=== Figure 6 ===
      %
      A6.XLABEL   = 'Max/min correlation [-]';
      subplot(3,2,6)
      dtext = (max(z(ip))-min(z(ip)))/yfac/10;
      sleg = [];
      for j = 1:nrq
        if j~=i
          nr    = j-(j>i);
          ind2 = kx_index(j,1):kx_index(j,2);
          if length(ind2) > 2
            pline( max(C(ind2,ind)), z(ind)/yfac, nr );hold on
            pline( min(C(ind2,ind)), z(ind)/yfac, nr );
          else
            pline( C(ind2,ind), z(ind)/yfac, nr );hold on
          end
          [gg,nn] = qp_kinfo( kx_names{j} ); 
          sleg    = strvcat( sleg, nn );
        end
      end
      axis([-0.99 1 0.99*min(z(ip))/yfac 1.01*max(z(ip))/yfac]);
      plineleg(sleg,0.05,0.93,0.12,10,0.06);
      hold off
      ax_set( A6 );
      
      
      suptitle(name,0.97,14);

    end
  end
end

% plot the spectro errors if do_plots is specified
if Q.SPECTRO_DO & do_plots
  qp_plotspectro(Q,PTZ, Kx, kx_names, kx_index, kx_aux, measres, Dy, s_tot, mr_limit);
end


%==============================================================================
%===
%=== Sub-functions
%===
%==============================================================================

function A = axes_def() 
  %02;   
  A.BOX           = 'on';
  %
  A.FONTSIZE      = 15;
  A.FONTWEIGHT    = 'normal';
  %
  A.XLABEL_SIZE   = 15;
  A.XLABEL_WEIGHT = 'bold';
  %
  A.YLABEL_SIZE   = 15;
  A.YLABEL_WEIGHT = 'bold';
  %
return
