%------------------------------------------------------------------------
% NAME:     qp_x2X
%
%           Map a state vector x to an array of structures X 
%           which contains information for retrieval quantities. 
%           See qpcls.m for more details.
%
%           The present fields of X are just a first start. Some fields
%           can be changed and fields will be added. The X-fields will
%           be adjusted in such way that they can be used to create level
%           2 data for Odin-SMR.
%
%           The second output argument, *unitfac*, can be used to scale
%           errors to match the unit conversions made here. This vector
%           is set to for cases when no conversion is used.
%
% FORMAT:   [X,unitfac] = ...
%               qp_x2X(x,Q,tmpdir,basename,kx_names,kx_index,kx_aux,p_abs,vmrs)
%
% OUT:      X           An array of structures X containing information 
%                       for retrieved quantities. 
%           unitfac     Vector with same length as x that gives the 
%                       multiplicative factor to consider unit
%                       conversions.
%                       For the retrieved species it gives the unit of
%                       the avk's.
% IN:       x     
%           Q           Setting structure.
%           tmpdir      Name/path of temporary directory
%           basename    Basename for the performed ARTS run.
%           kx_names    Name of each retrieval identity.
%           kx_index    Index of first and last element in x for each
%                       retrieval identity.
%           kx_aux      As kx_aux in ARTS.
%           p_abs       Absorption pressure grid.
%           vmrs        As ARTS variable vmrs.
%------------------------------------------------------------------------

% HISTORY: 2001.10.12  Created by Patrick Eriksson.
%        : 2003.11.09  Updated by Sho Tsujimaru.

function [X,unitfac] = ...
               qp_x2X(x,Q,tmpdir,basename,kx_names,kx_index,kx_aux,p_abs,vmrs)


nrq      = size( kx_names, 1 );
X        = cell( nrq, 1 );
ispecies = 0;
unitfac  = ones( length(x), 1 );


%= Make sure that HSE is done if it should be done
%= (In many cases so has HSE been applied to z_abs, but instead of trying to
%= be smart, we do it every time here)
%
if Q.HSE_IN_ON
  template      = which( 'hse.tmplt' );
  %
  QE = [];
  QE = qp_hdf( Q, QE );
  %
  [cfile,basename] = qtool( Q, tmpdir, template, QE );
  qp_arts( Q, cfile ); 
end
z_abs = read_artsvar( basename ,'z_abs' );


for i = 1:nrq

  [group,name] = qp_kinfo(  kx_names{i} );
  X{i}.name = name;       

  ind = kx_index(i,1):kx_index(i,2);

  %=== Species
  %
  if strcmp( group, 'Species' )
    %
    ispecies     = ispecies + 1;
    X{i}.p       = kx_aux(ind,1);
    X{i}.z       = interpp( p_abs, z_abs, X{i}.p );
    X{i}.apriori = interpp( p_abs, vmrs(ispecies,:)', X{i}.p );
    if Q.CLS_ABS_SPECIES_ON 
      X{i}.x       = x(ind);
    else
      X{i}.x       = x(ind) .* X{i}.apriori;
      X{i}.xfrac   = x(ind);
      unitfac(ind) = X{i}.apriori;
    end
    
  %=== Sensor scalars such as pointing off-set
  %
  elseif strcmp( group, 'Sensor scalar' )
    %
    X{i}.apriori = 0;
    X{i}.x       = x(ind);
    %
    % For frequency is MHz used as internal unit.
    if strcmp( name, 'Frequency: off-set' )
      X{i}.x       = X{i}.x * 1e6;
      unitfac(ind) = 1e6;
    end

  %=== Temperature
  %
  elseif strcmp( group, 'Temperature' ) 
    %
    X{i}.p       = kx_aux(ind,1);
    X{i}.z       = interpp( p_abs, z_abs, X{i}.p );
    X{i}.apriori = kx_aux(ind,2);
    X{i}.x       = x(ind);

  %=== Continuum fit
  %
  elseif strcmp( group, 'Continuum' )
    %
    X{i}.p       = kx_aux(ind,1);
    X{i}.z       = interpp( p_abs, z_abs, X{i}.p );
    X{i}.apriori = kx_aux(ind,2);
    X{i}.x       = 0.001 * x(ind);

  %=== Ground emission
  %
  elseif strcmp( group, 'Ground emission' )
    %
    X{i}.f       = kx_aux(ind,1);
    X{i}.apriori = kx_aux(ind,2);
    X{i}.x       = x(ind);

  %=== Polynomial baseline fits
  %
  elseif strcmp( group, 'Baseline' )
    %
    X{i}.p       = kx_aux(ind,1);
    X{i}.z       = kx_aux(ind,1);
    X{i}.apriori = kx_aux(ind,2);
    X{i}.x       = x(ind);

  %=== Reference load temperatures
  %
  elseif strcmp( name, 'Reference load temperature(s)' )
    %
    X{i}.apriori = kx_aux(ind,2);
    X{i}.x       = x(ind);

  %=== Proportional calibration error
  %
  elseif strcmp( name, 'Proportional calibration error' )
    %
    X{i}.apriori = kx_aux(ind,2);
    X{i}.x       = x(ind);

  % Newly appended part. 
  %
  %= Pressure shift
  %
  elseif strcmp( name, 'Pressure shift' )
    %
    X{i}.apriori = kx_aux(ind,2);
    X{i}.x       = x(ind) * 1e4; % MHz/hPa --> Hz/Pa
    unitfac(ind) = 1e4;
  
  %= Frequency calibration
  %
  elseif strcmp( name, 'Freq. calibration' )
    %
    X{i}.apriori = kx_aux(ind,2);
    X{i}.x       = x(ind) * 1e6; % MHz --> Hz 
    unitfac(ind) = 1e6;

  else
    error(sprintf( 'Unknown retrieval identity: %s', X{i}.name ));
  end
end
