%------------------------------------------------------------------------
% NAME:     qp_x2y
%
%           Maps a state vector to corrections for the measurement vector.
%
%           Retrieved calibration variables are used to re-scale the
%           spectra to achieve a more correct calibration. 
%
%           The baseline corresponding to all baseline parameters is 
%	    determined and is returned as BL. The found baseline is removed
%	    from Y after the re-calibration has been performed.
%
% FORMAT:   [y,bl] = qp_x2y( Q, x, Kx, kx_names, kx_index, y )
%
% OUT:      y           Corrected measurement spectrum.
%           bl          Fitted baseline (is removed from y).
% IN:       Q           Q structure.
%           x           The state vector.
%           Kx          Weighting function matrix.
%           kx_names    As the corresponding ARTS variable.
%           kx_index    Start and stop index for each retrieval identity.
%                       A 2 column matrix.
%           y		Measurement spectrum.
%------------------------------------------------------------------------

% HISTORY: 2002.02.13  Started by by Patrick Eriksson.


function [y,bl] = qp_x2y( Q, x, Kx, kx_names, kx_index, y )


bl = 0;

do_tbrefload = 0;
do_propcal   = 0;

% If a new quantity is included here, it should also be included in
% qp_x2arts as an empty if statement in order to check that something is
% done for all quantities.


for i = 1:size(kx_names,1)

  [group,name] = qp_kinfo( kx_names{i} );

  ind = kx_index(i,1):kx_index(i,2);

  %=== Baseline fits
  %
  if strcmp( group, 'Baseline' )
    %
    bl = bl + Kx(:,ind) * x(ind);


  %=== Reference load temperatures
  %
  elseif strcmp( name, 'Reference load temperatures' )
    %
    do_tbrefload = 1;
    tbrefload    = x(ind);


  %=== Proportional calibration error
  %
  elseif strcmp( name, 'Proportional calibration error' )
    %
    do_propcal   = 1;
    propcal      = x(ind);

  end
end


y = y - bl;


%= Adjust for bad load temperatures
%
if do_tbrefload
  y1n = Q.TB_REFLOADS_NOMINAL(1);
  y2n = Q.TB_REFLOADS_NOMINAL(2);
  y1  = tbrefload(1);
  y2  = tbrefload(2);
  %
  y   = y1 + (y2-y1) * (y-y1n) / (y2n-y1n);
end


%= Adjust for bad calibration gain factor
%
if do_propcal
  y   = Q.PROPCAL_TB_REF + y / (1+propcal);
end
