%------------------------------------------------------------------------
% NAME:     qp_ympb
%
%           A function to calculate the monochromatic pencil beam spectrum 
%           corresponding to a Q structure, with the possibility to replace
%           F_MONO and ZA_PENCIL. 
%
%           The optional arguments F_MONO and ZA_PENCIL can be empty, and
%           then original vectors are used.
%
%           Not all fields of Q must be set. 
%
%           So far is no noise or baseline ripple added to the spectrum.
%
% FORMAT:   y = qp_ympb( Q [, f_mono, za_pencil ] )
%
% OUT:      y              Calculated spectrum.
% IN:       Q              Setting structure.
% OPTIONAL: f_mono         Alternative monochromatic frequency grid.
%           za_pencil      Alternative pencil beam zenith angle grid.
% TEMPLATE: basic.tmplt
%------------------------------------------------------------------------

% HISTORY: 2001.11.20  Created by Patrick Eriksson.


function y = qp_ympb( Q, f_mono, za_pencil )


%=== Create temporary dir
%
tmpdir    = temporary_directory( Q.TMP_AREA );


%=== If necessary, use TMPDIR as CALCGRIDS_DIR
%
if nargin > 1
  %
  %= Copy all .aa and .ab files to TMPDIR
  D = dir( fullfile( Q.CALCGRIDS_DIR, '*.a*' ) );
  for i = 1:length(D)
    copyfile( fullfile( Q.CALCGRIDS_DIR, D(i).name ), ...
                                               fullfile( tmpdir, D(i).name ) );
  end
  %
  Q.CALCGRIDS_DIR = tmpdir;
  %
  if exist('f_mono') & ~isempty(f_mono)
    write_datafile( fullfile( Q.CALCGRIDS_DIR, Q.F_MONO ), f_mono, 'VECTOR' );
  end
  %
  if exist('za_pencil') & ~isempty(za_pencil)
    write_datafile( fullfile( Q.CALCGRIDS_DIR, Q.ZA_PENCIL ), za_pencil, ...
                                                                    'VECTOR' );
  end
end


%=== Get full path of template file and create cfile
%
template = which( 'basic.tmplt' );
%
QE.ABS_SAVE = 0;
QE.ABS_EXIT = 0;
QE          = qp_hdf( Q, QE );
%
[cfile,basename] = qtool( Q, tmpdir, template, QE );



%=== Run ARTS
%
qp_arts( Q, cfile );



%=== Load spectrum
%
y = read_artsvar( basename, 'y' );


%=== Delete the temporary directory
delete_tmp_dir( Q.TMP_AREA, tmpdir );
