%------------------------------------------------------------------------
% NAME:     ycalc_q
%
%           A sample Q definition function for simulating a measurement.
%
%           For simulating measurements a sub-set of the Q-fields are
%           used (no new field names are introduced). It is not necessary
%           to specify the fields that are not used (what fields that 
%           actually are used is not documented, use trail and error!).
%
%           To simulate the measurement, type:
%
%           >> Qy = qpack( 'ycalc_q' ); 
%           >> qp_H( Qy );
%           >> y  = qp_ycalc( Qy ); 
%
%           Note that the field OUT must have an unique name to avoid
%           crashes with the pre-calculations.
%
% FORMAT:   Q = ycalc_q( Q )
%
% OUT:      Q    the setting structure for Qpack
% IN:       Q    the setting structure for Qpack
% OPTIONAL: -
%
% TEMPLATE: -
% Q-FIELDS: -
%------------------------------------------------------------------------

% HISTORY: 2001.04.17  Created by Patrick Eriksson


function Q = ycalc_q( Q )


%=== Report levels
%
Q.QP_LEVEL   = 2;
Q.ARTS_LEVEL = 0;


%=== Directories
%
top_dir          = '/home/patrick/ARTS/Qpack/Samples';
%
Q.ARTS           = 'arts1';
Q.TMP_AREA       = '/tmp';
%
Q.OUT            = fullfile(top_dir,'Out','out_y');
Q.SPECTRO_DIR    = fullfile(top_dir,'Spectroscopy');
Q.CALCGRIDS_DIR  = fullfile(top_dir,'CalcGrids');
Q.SENSOR_DIR     = fullfile(top_dir,'Sensor');



%=== Atmospheres
%
Q.APRIORI_VMR = ...
     '/home/patrick/ARTS/arts-data/atmosphere/fascod/tropical';
Q.APRIORI_PTZ     = ...
     '/home/patrick/ARTS/arts-data/atmosphere/fascod/subarctic-winter.tz.aa';


%=== Species
%
Q.RETRIEVAL_TAGS   = [ ...
      '"ClO",', ...
      '"O3",', ...
      '"N2O",', ...
      '"H2O-MPM89"' ]; 
%
Q.OTHER_TAGS       = [...
      '"N2-SelfContStandardType"' ];


%=== Spectroscopy
%
Q.LINEFILE         = 'lines501.4';
%
Q.LINESHAPE        = 'Voigt_Kuntz3';
Q.LINESHAPE_FACTOR = 'quadratic'; 
Q.LINESHAPE_CUTOFF = -1;
%
Q.CONTINUA         = 'continua_std';


%=== Hydrostatic eq.
Q.HSE_IN_ON          = 1;
Q.HSE_PREF           = 1.01e5;
Q.HSE_ZREF           = 0;


%=== RTE
%
Q.PLATFORM_ALTITUDE  = 625e3;
Q.STEPLENGTH_RTE     = 20e3;
%
Q.GROUND_ALTITUDE    = 200; 
Q.GROUND_EMISSION    = 1;
%
Q.REFRACTION_ON      = 0;
Q.REFR_METHOD        = 'Boudouris';
Q.REFR_LFAC          = ceil( Q.STEPLENGTH_RTE/2e3 );
%
Q.EMISSION_ON        = 1;


%=== Calculation grids
%
Q.P_ABS     = 'p_abs.aa';
Q.F_MONO    = 'f_mono.aa';
Q.ZA_PENCIL = 'za_pencil.aa';


%=== Sensor
%
Q.F_ORDER     = 1;
Q.ZA_ORDER    = 1;
%
Q.ANTENNA_ON    = 1;
Q.ANTENNA_ZA    = 'antenna_za.aa';
Q.ANTENNA_FILE  = 'antenna.aa';
Q.ANTENNA_ORDER = 1;
Q.ANTENNA_MOVE  = 0.011;
%
Q.DSB_ON        = 1;
Q.DSB_FILE      = 'sbfilter501.380.aa';
Q.DSB_ORDER     = 1;
Q.DSB_FPRIMARY  = 501.38e9;
Q.DSB_LO        = 497.38e9;
%
Q.BACKEND_ON    = 1;
Q.BACKEND_FREQS = 'backend_freqs.aa';
Q.BACKEND_FILE  = 'ac_channel_r1.16.ab';
Q.BACKEND_ORDER = 1;


