%------------------------------------------------------------------------
% NAME:     do_artscat
%
%           This function converts the Verdandi database to the ARTSCAT
%           format. AMI is required to execute this function. The function
%           shall be called when standing in the folder where the data 
%           shall be stored.
%
% FORMAT:   do_artscat( [ verdandi, mo0 ] )
%
% OPTIONAL: verdandi  Path to folder where the Verdandi database is placed.
%                     The function includes user specific hard-coded paths
%                     that are used as default if no argument is given.
%                     The default path is used also if the argument is empty.
%           mo0       Start at this molecule number. This option can be
%                     used to restart the conversion after a crash without
%                     starting from scratch.
%------------------------------------------------------------------------

% HISTORY: 2002-08-13  Created by Patrick Eriksson


function do_artscat( verdandi, mo0 )


%= Default value for path to Verdandi
%
if ~exist('verdandi','var') | isempty(verdandi)
  if strcmp( whoami, 'patrick' )
    verdandi = '/u/patrick/Projects/Norns/Verdandi/Catalogue';
  else
    error('Unknown user.');
  end
end


%= Start molecule number
%
if ~exist('mo0','var')
  mo0 = 0;
end


%= Copy Verdandi README file as README2
%
filename = fullfile( verdandi, 'README' );
%
if ~exist( filename, 'file' )
  error('Cannot find the Verdandi README file. Is path correct?');
end
%
copyfile( filename, 'README2' );


%= Copy the file species_table.txt
%
filename = fullfile( verdandi, 'species_table.txt' );
%
if ~exist( filename, 'file' )
  error('Cannot find the Verdandi species_table.txt file. Is path correct?');
end
%
copyfile( filename, 'species_table.txt' );


%= Convert all files starting with v
%
infiles = dir( fullfile(verdandi,'v*') );
%
for i = 1 : length(infiles)

  % Name of Verdandi file
  verdandifile = infiles(i).name;

  % Extract molecule number
  mo = str2num( verdandifile(2:3) );

  % Shall all molecule numbers be done
  if mo >= mo0

    % Name of ARTS file to create
    l = length( verdandifile );
    artsfile = [ vtag2arts( 10*str2num(verdandifile(l+(-3:-2))) + ...
				           str2num(verdandifile(l)) ), '.ac' ];
  
    fprintf( 'Converting %s to %s ...\n', verdandifile, artsfile );
  
    % Several species have too many transitions and it is not possible to read
    % the files as a block (by read_verdandi). Instead we are copying line for
    % line from the Verdandi files, make a conversion using a dummy file, read
    % the dummy file and appends the converted data to the ARTS file. Uh, but
    % that was the best solution I could find. And the function doesn't need
    % to be fast.
  
    % Open the Verdandi file
    name = fullfile( verdandi, verdandifile );
    verdandi_fid = fopen( name, 'r' );
    if verdandi_fid < 0
      error(sprintf('The file %s could not be opened.',name));
    end
  
    % Read the Verdandi file until EOF
    %
    il = 0;     % Line index 
    %
    s = fgets( verdandi_fid );
    %
    while ischar( s )
  
      il = il + 1;
  
      dummy_fid = fopen( 'dummy', 'w' );
      fprintf( dummy_fid, s );
      fclose( dummy_fid );
  
      L = read_verdandi( 'dummy' );
  
      if il == 1
	write_linefile( artsfile, L, 1, 1, 0 );
      else
	write_linefile( artsfile, L, 1, 1, 1 );
      end
    
      s = fgets( verdandi_fid );
  
    end
  
    fclose( verdandi_fid );

  end

end

