% ARTS_OEM_DEMO   Demonstration of inversions using ARTS and OEM
%
%    The example treats ground-based measurements of ozone at 110.8 GHz.
%
%    A figure with the inversion result is generated if the function is
%    called with no output arguments. Otherwise, the output from *oem* is
%    returned (work folder is deleted).
%
% FORMAT   [X,R] = arts_oem_demo
%
% OUT   X   As same output from *oem*.
%       R   As same output from *oem*.

% 2006-09-07   Created by Patrick Eriksson.

function [X,R] = arts_oem_demo


%- Check if needed atmlab settings are found
%
arts_xmldata_path = atmlab( 'ARTS_XMLDATA_PATH' );
if isnan( arts_xmldata_path )
  error('You need to ARTS_XMLDATA_PATH to run this example.');
end
%
arts_includes = atmlab( 'ARTS_INCLUDES' );
if isnan( arts_includes )
  error('You need to ARTS_INCLUDES to run this example.');
end



%----------------------------------------------------------------------------
%- Init control structures 
%----------------------------------------------------------------------------
%
Q      = qarts;
Q.J_DO = true;
O      = oem;


%----------------------------------------------------------------------------
%- OEM settings
%----------------------------------------------------------------------------
%
O.itermethod = 'GN';
O.stop_dx    = 0.01;
O.maxiter    = 10;
%
O.cost       = true;
O.e          = true; 
O.eo         = true;
O.es         = true;
O.yf         = true;



%----------------------------------------------------------------------------
%- Forward model parameters
%----------------------------------------------------------------------------

%- General 
%
Q.INCLUDES            = { fullfile( arts_includes, 'general.arts' ), ...
                          fullfile( arts_includes, 'continua.arts' ) };
Q.ATMOSPHERE_DIM      = 1;
Q.STOKES_DIM          = 1;

%- Atmosphere
%
Q.R_GEOID             = constants( 'EARTH_RADIUS' );
Q.Z_SURFACE           = 0;
Q.P_GRID              = z2p_simple( -250:250:90e3 )';
%
Q.USE_RAW_ATMOSPHERE  = true;
Q.RAW_ATMOSPHERE      = fullfile( arts_xmldata_path, 'atmosphere', ...
                                              'fascod', 'midlatitude-winter' );

%- RTE
%
Q.WSMS_BEFORE_RTE     = { 'atm_checkedCalc' };
Q.YCALC_WSMS          = { 'yCalc' };
%
Q.PPATH_LMAX          = 250;
Q.PPATH_STEP_AGENDA   = { 'ppath_stepGeometric' };
Q.Y_UNIT              = 'RJBT';
%
Q.SENSOR_LOS          = 70;
Q.SENSOR_POS          = Q.R_GEOID + Q.Z_SURFACE;

%- Exclude cosmic background radiation. A relative change of the spectrum
%- corresponds then to a relative change of the ozone profile.
%
Q.IY_SPACE_AGENDA     = { 'Ignore(rte_los)',  'Ignore(rte_pos)', ...
                          'nelemGet(nrows,f_grid)', ...
                          'Copy(ncols,stokes_dim)', ...
                          'MatrixSetConstant(iy,nrows,ncols,0)' };


%- Frequency, spectrometer and pencil beam antenna
%
% The hypothetical spectrometer has rectangular response functions
%
Q.F_GRID              = qarts_get( fullfile( atmlab_example_data , ...
                                              'f_grid_111ghz.xml' ) );
%
H                     = qartsSensor;
%
H.SENSOR_NORM         = true;
%
df                    = 0.5e6;
H.F_BACKEND           = [ min(Q.F_GRID)+df : df : max(Q.F_GRID)-df ]';
%
B.name                = 'Spectrometer channel response function';
B.gridnames           = { 'Frequency' };
B.grids               = { [-df/2 df/2] };
B.dataname            = 'Response';
B.data                = [1 1];
%
H.BACKEND_CHANNEL_RESPONSE{1} = B;
clear B
%
Q.SENSOR_DO           = true;
Q.SENSOR_RESPONSE     = H;
%
Q.ANTENNA_DIM         = 1;
Q.MBLOCK_ZA_GRID      = 0;



%----------------------------------------------------------------------------
%- Species
%----------------------------------------------------------------------------

% Water vapour has to be inluded (needed by absorption part)
%
Q.ABS_SPECIES(1).TAG      = { 'H2O' };
Q.ABS_SPECIES(1).RETRIEVE = false;

% Ozone, retrieved
%
Q.ABS_SPECIES(2).TAG      = { 'O3' };
Q.ABS_SPECIES(2).RETRIEVE = true;
Q.ABS_SPECIES(2).GRIDS    = { z2p_simple( 0e3:1e3:90e3 )', [], [] };
Q.ABS_SPECIES(2).UNIT     = 'vmr';
Q.ABS_SPECIES(2).SX       = covmat1d_from_cfun( Q.ABS_SPECIES(2).GRIDS{1}, ...
                                    2e-6, 'lin', 0.2, 0.00, @log10 ) + ...
                            covmat1d_from_cfun( Q.ABS_SPECIES(2).GRIDS{1}, ...
                                  0.5e-6, 'lin', 0.5, 0.00, @log10 );



%----------------------------------------------------------------------------
%- Set-up/create absorption look-up table
%----------------------------------------------------------------------------

Q.ABS_LINES_FORMAT         = 'Arts';
Q.ABS_LINES                = fullfile( atmlab_example_data , 'o3line111ghz' );
Q.ABS_NLS                  = [];
%
Q                          = qarts_abstable( Q ); 
% The absorption table will with the settings above be calculated
% automatically. If you want to pre-calculate the table, for later re-usage, 
% do: 
%Q.ABS_LOOKUP               = arts_abstable( Q ); 



%----------------------------------------------------------------------------
%- Init retrieval variables (such as xa) and create Sx
%----------------------------------------------------------------------------
  
[Qoem,O,R,xa] = arts_oem_init( Q, O );
%
Sx = arts_sx( Q, R );



%----------------------------------------------------------------------------
%- Create a measurement
%----------------------------------------------------------------------------

%- Noise statistics
%
si = 0.01;
%
SE.FORMAT    = 'param';
SE.SI        = si;
SE.CCO       = 0;  
SE.CFUN1     = 'drc';
SE.CL1       = 0; 

%- Create Se
%
Se = covmat3d( 1, SE, Q.SENSOR_RESPONSE.F_BACKEND );

%- Calculate a spectrum matching a priori
%
y = arts_y( Q );

%- Crate a test measurement
%
y = 1.1 * y + si*randn(size(Se,1),1);




%----------------------------------------------------------------------------
%- Calculations
%----------------------------------------------------------------------------

%- Perform retrieval
%
X = oem( O, Qoem, R, @arts_oem, Sx, Se, [], [], xa, y );
%
arts_oem( 'clean', R );
clear Qoem 


%- Plot
%
if ~nargout
  %
  z = p2z_simple( Q.ABS_SPECIES(2).GRIDS{1} );
  %
  if O.e & O.eo
    h= plot( 1e6*X.x, z/1e3, 'k-', ...
             1e6*(repmat(X.x,1,2)+[-X.e X.e]), z/1e3, 'b:', ...
             1e6*(repmat(X.x,1,2)+[-X.eo X.eo]), z/1e3, 'r--' );
    legend( h([1 4 2]), 'Retrieved profile', 'Observation error', ...
                                                               'Total error' );
  else
    plot( 1e6*X.x, z/1e3 )
  end
  xlabel( 'Ozone [ppm]' )
  ylabel( 'Approximative altitude [km]' );
end
