% N_WATER_SEGELSTEIN   Complex refractive index for water
%
%    The function is based on tabulated values from 
%       The complex refractive index of water / by David J. Segelstein.  1981.
%       Thesis (M.S.)--Department of Physics. University of
%       Missouri-Kansas City, 1981.
%
%    The tabulated values cover wavelengths from 1e-8 to 10 m, and
%    are interpolated to given wavelengths (*lambda*).
%
%    The refractive index is defined to have positive imaginary part.
%    
%    No temperature dependency of *n* is provided. Temperature for table
%    is unknown.
%
% FORMAT   n = n_water_segelstein( lambda )
%        
% OUT   n        Complex refractive index
% IN    lambda   Vector of wavelengths

% 2004-11-02   Created by Patrick Eriksson.


function n = n_water_segelstein(lambda)

A = get_n_data;

b = interp1( A(:,1), A(:,2:3), lambda*1e6 );

n = b(:,1) + i*b(:,2);

return



function A = get_n_data

%Segelstein, David J.
%The complex refractive index of water / by David J. Segelstein.  1981.
%ix, 167 leaves : ill. ; 29 cm.
%Thesis (M.S.)--Department of Physics. University of
%Missouri-Kansas City, 1981.			  
%
%Waveleng (mic)	Real Index	Imag Index
A=[
1.000E-02	0.968416	1.745E-03
1.099E-02	0.964778	2.370E-03
1.199E-02	0.960953	3.146E-03
1.300E-02	0.956954	4.072E-03
1.400E-02	0.952792	5.174E-03
1.600E-02	0.944124	7.958E-03
1.799E-02	0.934744	1.164E-02
2.000E-02	0.924583	1.636E-02
2.198E-02	0.913973	2.227E-02
2.399E-02	0.902694	2.950E-02
2.600E-02	0.890837	3.818E-02
2.799E-02	0.878766	4.851E-02
2.999E-02	0.866493	6.064E-02
3.199E-02	0.854141	7.461E-02
3.396E-02	0.842171	9.074E-02
3.597E-02	0.830720	1.093E-01
3.802E-02	0.819753	1.303E-01
3.999E-02	0.809997	1.534E-01
4.198E-02	0.802291	1.798E-01
4.395E-02	0.797737	2.088E-01
4.603E-02	0.797007	2.414E-01
4.797E-02	0.805579	2.766E-01
5.000E-02	0.820742	2.998E-01
5.200E-02	0.830957	3.153E-01
5.395E-02	0.835240	3.310E-01
5.598E-02	0.835295	3.498E-01
5.794E-02	0.831628	3.739E-01
5.998E-02	0.830901	4.119E-01
6.194E-02	0.840575	4.558E-01
6.397E-02	0.866994	5.033E-01
6.607E-02	0.903527	5.355E-01
6.808E-02	0.941801	5.634E-01
6.998E-02	0.981692	5.791E-01
7.194E-02	1.020921	5.859E-01
7.396E-02	1.049744	5.805E-01
7.603E-02	1.068724	5.859E-01
7.798E-02	1.087685	5.981E-01
7.998E-02	1.111682	6.135E-01
8.204E-02	1.140628	6.292E-01
8.395E-02	1.173382	6.453E-01
8.590E-02	1.214969	6.573E-01
8.790E-02	1.259495	6.573E-01
8.995E-02	1.302663	6.528E-01
9.204E-02	1.346760	6.439E-01
9.397E-02	1.387639	6.292E-01
9.594E-02	1.425425	6.050E-01
9.795E-02	1.455868	5.752E-01
1.000E-01	1.476628	5.430E-01
1.021E-01	1.493473	5.185E-01
1.040E-01	1.506677	4.929E-01
1.059E-01	1.516305	4.707E-01
1.079E-01	1.523589	4.485E-01
1.099E-01	1.528933	4.303E-01
1.119E-01	1.535363	4.148E-01
1.140E-01	1.543211	3.988E-01
1.159E-01	1.548070	3.826E-01
1.180E-01	1.553435	3.705E-01
1.199E-01	1.560870	3.596E-01
1.219E-01	1.570304	3.490E-01
1.239E-01	1.584638	3.387E-01
1.259E-01	1.606068	3.220E-01
1.279E-01	1.626822	2.876E-01
1.300E-01	1.633849	2.392E-01
1.321E-01	1.619420	1.870E-01
1.340E-01	1.586268	1.489E-01
1.361E-01	1.536403	1.333E-01
1.380E-01	1.496271	1.422E-01
1.400E-01	1.471129	1.678E-01
1.419E-01	1.461485	1.927E-01
1.439E-01	1.460977	2.167E-01
1.459E-01	1.469275	2.409E-01
1.479E-01	1.489551	2.641E-01
1.500E-01	1.521276	2.772E-01
1.521E-01	1.559942	2.772E-01
1.542E-01	1.596861	2.581E-01
1.560E-01	1.620422	2.338E-01
1.581E-01	1.641473	2.022E-01
1.600E-01	1.650184	1.670E-01
1.622E-01	1.652917	1.351E-01
1.641E-01	1.653100	1.039E-01
1.660E-01	1.647245	7.241E-02
1.679E-01	1.635062	3.998E-02
1.698E-01	1.605555	3.998E-03
1.722E-01	1.568183	2.004E-03
1.742E-01	1.549412	1.182E-03
1.750E-01	1.543062	8.391E-04
1.799E-01	1.513343	5.995E-05
1.849E-01	1.491881	1.250E-06
1.901E-01	1.475183	3.622E-07
1.950E-01	1.462543	1.850E-07
2.000E-01	1.451724	1.101E-07
2.051E-01	1.442296	6.711E-08
2.099E-01	1.434685	3.844E-08
2.148E-01	1.427828	1.999E-08
2.198E-01	1.421603	1.270E-08
2.249E-01	1.415921	1.158E-08
2.301E-01	1.410702	1.101E-08
2.350E-01	1.406358	1.071E-08
2.399E-01	1.402321	1.049E-08
2.449E-01	1.398535	9.904E-09
2.500E-01	1.394993	9.307E-09
2.553E-01	1.391674	8.606E-09
2.600E-01	1.388881	7.994E-09
2.649E-01	1.386239	7.444E-09
2.698E-01	1.383726	6.852E-09
2.748E-01	1.381341	6.292E-09
2.799E-01	1.379072	5.791E-09
2.851E-01	1.376902	5.405E-09
2.897E-01	1.375086	4.795E-09
2.951E-01	1.373098	4.403E-09
2.999E-01	1.371437	4.148E-09
3.048E-01	1.369839	3.826E-09
3.097E-01	1.368287	3.546E-09
3.148E-01	1.366812	3.325E-09
3.199E-01	1.365376	3.190E-09
3.251E-01	1.363990	3.082E-09
3.304E-01	1.362616	2.984E-09
3.350E-01	1.361513	2.883E-09
3.396E-01	1.360441	2.766E-09
3.451E-01	1.359231	2.653E-09
3.499E-01	1.358224	2.528E-09
3.548E-01	1.357247	2.420E-09
3.597E-01	1.356295	2.316E-09
3.648E-01	1.355370	2.217E-09
3.698E-01	1.354470	2.117E-09
3.750E-01	1.353594	2.031E-09
3.802E-01	1.352740	1.940E-09
3.846E-01	1.352046	1.840E-09
3.899E-01	1.351231	1.761E-09
3.954E-01	1.350438	1.663E-09
3.999E-01	1.349793	1.580E-09
4.046E-01	1.349159	1.489E-09
4.102E-01	1.348417	1.422E-09
4.150E-01	1.347811	1.339E-09
4.198E-01	1.347219	1.258E-09
4.246E-01	1.346636	1.169E-09
4.295E-01	1.346066	1.088E-09
4.345E-01	1.345505	1.018E-09
4.395E-01	1.344956	9.393E-10
4.446E-01	1.344418	8.685E-10
4.498E-01	1.343889	8.087E-10
4.550E-01	1.343368	7.795E-10
4.603E-01	1.342858	7.600E-10
4.645E-01	1.342455	7.495E-10
4.699E-01	1.341961	7.291E-10
4.753E-01	1.341475	7.011E-10
4.797E-01	1.341093	7.092E-10
4.853E-01	1.340620	7.158E-10
4.898E-01	1.340248	7.342E-10
4.955E-01	1.339791	7.849E-10
5.000E-01	1.339430	9.243E-10
5.047E-01	1.339073	1.078E-09
5.105E-01	1.338635	1.267E-09
5.152E-01	1.338288	1.461E-09
5.200E-01	1.337944	1.570E-09
5.248E-01	1.337607	1.640E-09
5.297E-01	1.337273	1.757E-09
5.346E-01	1.336943	1.887E-09
5.395E-01	1.336615	2.098E-09
5.445E-01	1.336292	2.269E-09
5.495E-01	1.335972	2.442E-09
5.546E-01	1.335656	2.659E-09
5.598E-01	1.335344	2.869E-09
5.649E-01	1.335035	3.132E-09
5.702E-01	1.334729	3.434E-09
5.754E-01	1.334425	3.844E-09
5.794E-01	1.334200	4.434E-09
5.848E-01	1.333902	5.221E-09
5.902E-01	1.333609	6.365E-09
5.957E-01	1.333316	7.723E-09
5.998E-01	1.333100	9.634E-09
6.053E-01	1.332813	1.132E-08
6.095E-01	1.332598	1.238E-08
6.152E-01	1.332317	1.330E-08
6.194E-01	1.332106	1.399E-08
6.252E-01	1.331826	1.472E-08
6.295E-01	1.331619	1.502E-08
6.353E-01	1.331345	1.552E-08
6.397E-01	1.331144	1.570E-08
6.457E-01	1.330877	1.606E-08
6.501E-01	1.330683	1.674E-08
6.546E-01	1.330490	1.777E-08
6.607E-01	1.330238	1.940E-08
6.653E-01	1.330052	2.031E-08
6.699E-01	1.329869	2.098E-08
6.745E-01	1.329690	2.177E-08
6.808E-01	1.329452	2.300E-08
6.855E-01	1.329278	2.471E-08
6.902E-01	1.329106	2.653E-08
6.950E-01	1.328938	2.963E-08
6.998E-01	1.328769	3.348E-08
7.047E-01	1.328603	4.100E-08
7.096E-01	1.328440	4.998E-08
7.145E-01	1.328279	5.995E-08
7.194E-01	1.328120	7.291E-08
7.244E-01	1.327963	9.137E-08
7.295E-01	1.327808	1.150E-07
7.345E-01	1.327652	1.348E-07
7.396E-01	1.327502	1.458E-07
7.447E-01	1.327350	1.530E-07
7.499E-01	1.327201	1.559E-07
7.551E-01	1.327055	1.580E-07
7.603E-01	1.326909	1.580E-07
7.656E-01	1.326764	1.570E-07
7.691E-01	1.326667	1.527E-07
7.745E-01	1.326524	1.478E-07
7.798E-01	1.326382	1.409E-07
7.852E-01	1.326244	1.339E-07
7.907E-01	1.326104	1.282E-07
7.943E-01	1.326012	1.258E-07
7.998E-01	1.325874	1.250E-07
8.054E-01	1.325739	1.270E-07
8.091E-01	1.325648	1.330E-07
8.147E-01	1.325512	1.448E-07
8.204E-01	1.325379	1.621E-07
8.241E-01	1.325290	1.819E-07
8.299E-01	1.325157	2.041E-07
8.356E-01	1.325025	2.243E-07
8.395E-01	1.324937	2.459E-07
8.453E-01	1.324805	2.690E-07
8.492E-01	1.324718	2.929E-07
8.551E-01	1.324590	3.153E-07
8.590E-01	1.324502	3.348E-07
8.650E-01	1.324373	3.546E-07
8.710E-01	1.324244	3.748E-07
8.750E-01	1.324159	3.907E-07
8.790E-01	1.324074	4.053E-07
8.851E-01	1.323946	4.234E-07
8.892E-01	1.323859	4.403E-07
8.954E-01	1.323732	4.622E-07
8.995E-01	1.323648	4.862E-07
9.057E-01	1.323520	5.150E-07
9.099E-01	1.323434	5.699E-07
9.141E-01	1.323351	6.696E-07
9.204E-01	1.323222	8.304E-07
9.247E-01	1.323138	1.060E-06
9.290E-01	1.323054	1.368E-06
9.354E-01	1.322926	1.771E-06
9.397E-01	1.322842	2.169E-06
9.441E-01	1.322757	2.557E-06
9.506E-01	1.322630	2.932E-06
9.550E-01	1.322546	3.190E-06
9.594E-01	1.322462	3.358E-06
9.661E-01	1.322333	3.464E-06
9.705E-01	1.322249	3.502E-06
9.750E-01	1.322165	3.480E-06
9.795E-01	1.322080	3.418E-06
9.840E-01	1.321994	3.336E-06
9.908E-01	1.321866	3.253E-06
9.954E-01	1.321780	3.131E-06
1.000E+00	1.321695	3.000E-06
1.009E+00	1.321521	2.688E-06
1.021E+00	1.321303	2.352E-06
1.030E+00	1.321128	2.001E-06
1.040E+00	1.320952	1.690E-06
1.050E+00	1.320775	1.419E-06
1.059E+00	1.320596	1.299E-06
1.069E+00	1.320416	1.259E-06
1.079E+00	1.320233	1.329E-06
1.089E+00	1.320051	1.499E-06
1.099E+00	1.319865	1.708E-06
1.109E+00	1.319678	2.038E-06
1.119E+00	1.319488	2.628E-06
1.130E+00	1.319296	3.869E-06
1.140E+00	1.319103	5.951E-06
1.151E+00	1.318909	9.306E-06
1.159E+00	1.318763	1.069E-05
1.169E+00	1.318566	1.120E-05
1.180E+00	1.318366	1.160E-05
1.191E+00	1.318162	1.181E-05
1.199E+00	1.318008	1.199E-05
1.211E+00	1.317799	1.191E-05
1.219E+00	1.317641	1.179E-05
1.230E+00	1.317427	1.160E-05
1.239E+00	1.317263	1.139E-05
1.250E+00	1.317042	1.100E-05
1.259E+00	1.316873	1.079E-05
1.271E+00	1.316645	1.090E-05
1.279E+00	1.316470	1.139E-05
1.291E+00	1.316233	1.221E-05
1.300E+00	1.316052	1.400E-05
1.309E+00	1.315868	1.639E-05
1.321E+00	1.315618	1.912E-05
1.330E+00	1.315425	2.251E-05
1.340E+00	1.315228	2.849E-05
1.349E+00	1.315031	4.047E-05
1.361E+00	1.314760	4.505E-05
1.371E+00	1.314547	5.804E-05
1.380E+00	1.314329	7.802E-05
1.390E+00	1.314104	1.060E-04
1.400E+00	1.313871	1.530E-04
1.409E+00	1.313671	2.540E-04
1.419E+00	1.313518	3.197E-04
1.429E+00	1.313373	3.538E-04
1.439E+00	1.313220	3.629E-04
1.449E+00	1.313055	3.637E-04
1.459E+00	1.312888	3.604E-04
1.469E+00	1.312715	3.387E-04
1.479E+00	1.312525	3.018E-04
1.489E+00	1.312318	2.659E-04
1.500E+00	1.312093	2.248E-04
1.510E+00	1.311852	1.958E-04
1.521E+00	1.311604	1.741E-04
1.531E+00	1.311352	1.602E-04
1.542E+00	1.311097	1.441E-04
1.549E+00	1.310923	1.348E-04
1.560E+00	1.310659	1.240E-04
1.570E+00	1.310387	1.140E-04
1.581E+00	1.310114	1.071E-04
1.589E+00	1.309928	9.940E-05
1.600E+00	1.309642	9.347E-05
1.611E+00	1.309352	8.804E-05
1.622E+00	1.309055	8.310E-05
1.629E+00	1.308855	8.096E-05
1.641E+00	1.308548	7.903E-05
1.648E+00	1.308341	7.591E-05
1.660E+00	1.308021	7.398E-05
1.671E+00	1.307672	7.404E-05
1.679E+00	1.307435	7.495E-05
1.690E+00	1.307073	7.601E-05
1.698E+00	1.306829	7.743E-05
1.710E+00	1.306453	8.050E-05
1.722E+00	1.306070	8.410E-05
1.730E+00	1.305809	8.900E-05
1.742E+00	1.305413	9.510E-05
1.750E+00	1.305142	1.000E-04
1.762E+00	1.304727	1.051E-04
1.770E+00	1.304442	1.120E-04
1.778E+00	1.304155	1.219E-04
1.791E+00	1.303718	1.330E-04
1.799E+00	1.303418	1.359E-04
1.811E+00	1.302947	1.371E-04
1.820E+00	1.302616	1.380E-04
1.828E+00	1.302269	1.418E-04
1.841E+00	1.301709	1.552E-04
1.849E+00	1.301291	1.861E-04
1.862E+00	1.300633	3.205E-04
1.871E+00	1.300214	5.209E-04
1.879E+00	1.299860	7.224E-04
1.888E+00	1.299545	9.221E-04
1.901E+00	1.298998	1.161E-03
1.910E+00	1.298791	1.678E-03
1.919E+00	1.298793	1.827E-03
1.932E+00	1.298681	1.922E-03
1.941E+00	1.298590	1.909E-03
1.950E+00	1.298472	1.848E-03
1.959E+00	1.298308	1.717E-03
1.968E+00	1.298051	1.548E-03
1.982E+00	1.297607	1.402E-03
1.991E+00	1.297292	1.250E-03
2.000E+00	1.296913	1.101E-03
2.009E+00	1.296499	9.904E-04
2.018E+00	1.296066	8.888E-04
2.028E+00	1.295606	8.050E-04
2.042E+00	1.294919	7.392E-04
2.051E+00	1.294457	6.742E-04
2.061E+00	1.293973	6.206E-04
2.070E+00	1.293476	5.725E-04
2.080E+00	1.292966	5.294E-04
2.089E+00	1.292438	4.884E-04
2.099E+00	1.291899	4.643E-04
2.109E+00	1.291353	4.403E-04
2.118E+00	1.290795	4.176E-04
2.128E+00	1.290221	3.970E-04
2.138E+00	1.289634	3.826E-04
2.148E+00	1.289033	3.705E-04
2.158E+00	1.288418	3.587E-04
2.168E+00	1.287787	3.506E-04
2.178E+00	1.287139	3.434E-04
2.188E+00	1.286474	3.395E-04
2.198E+00	1.285790	3.379E-04
2.208E+00	1.285087	3.387E-04
2.218E+00	1.284365	3.410E-04
2.228E+00	1.283619	3.458E-04
2.239E+00	1.282852	3.571E-04
2.249E+00	1.282064	3.739E-04
2.259E+00	1.281256	3.898E-04
2.270E+00	1.280421	4.081E-04
2.280E+00	1.279561	4.293E-04
2.291E+00	1.278675	4.506E-04
2.301E+00	1.277755	4.686E-04
2.312E+00	1.276797	4.918E-04
2.317E+00	1.276305	5.114E-04
2.328E+00	1.275295	5.430E-04
2.339E+00	1.274257	5.995E-04
2.350E+00	1.273184	6.365E-04
2.360E+00	1.272062	6.852E-04
2.371E+00	1.270902	7.427E-04
2.382E+00	1.269682	7.921E-04
2.388E+00	1.269059	8.488E-04
2.399E+00	1.267787	9.095E-04
2.410E+00	1.266450	9.904E-04
2.421E+00	1.265059	1.071E-03
2.432E+00	1.263577	1.150E-03
2.438E+00	1.262824	1.250E-03
2.449E+00	1.261297	1.348E-03
2.460E+00	1.259683	1.472E-03
2.472E+00	1.257969	1.580E-03
2.477E+00	1.257102	1.709E-03
2.489E+00	1.255347	1.810E-03
2.500E+00	1.253465	1.900E-03
2.512E+00	1.251445	1.953E-03
2.518E+00	1.250383	1.990E-03
2.529E+00	1.248127	2.017E-03
2.541E+00	1.245672	2.069E-03
2.553E+00	1.243014	2.142E-03
2.564E+00	1.240167	2.269E-03
2.570E+00	1.238670	2.311E-03
2.576E+00	1.237089	2.338E-03
2.582E+00	1.235426	2.387E-03
2.588E+00	1.233679	2.425E-03
2.594E+00	1.231834	2.476E-03
2.606E+00	1.227769	2.575E-03
2.612E+00	1.225483	2.703E-03
2.618E+00	1.223082	2.977E-03
2.624E+00	1.220535	3.302E-03
2.630E+00	1.218078	4.016E-03
2.636E+00	1.215699	4.363E-03
2.649E+00	1.209954	4.828E-03
2.655E+00	1.206519	5.368E-03
2.661E+00	1.202951	6.278E-03
2.667E+00	1.199289	7.325E-03
2.673E+00	1.195340	8.547E-03
2.679E+00	1.191390	1.049E-02
2.685E+00	1.188087	1.270E-02
2.698E+00	1.179962	1.451E-02
2.704E+00	1.174582	1.640E-02
2.710E+00	1.168874	1.861E-02
2.716E+00	1.160993	2.050E-02
2.723E+00	1.152876	2.817E-02
2.729E+00	1.149520	3.800E-02
2.742E+00	1.142068	4.622E-02
2.748E+00	1.136183	5.480E-02
2.754E+00	1.132860	6.483E-02
2.761E+00	1.131711	7.444E-02
2.767E+00	1.132778	8.352E-02
2.780E+00	1.130913	9.285E-02
2.786E+00	1.127959	1.020E-01
2.793E+00	1.127558	1.119E-01
2.799E+00	1.129478	1.210E-01
2.812E+00	1.128413	1.312E-01
2.818E+00	1.125532	1.422E-01
2.825E+00	1.125351	1.541E-01
2.831E+00	1.127523	1.670E-01
2.838E+00	1.133346	1.798E-01
2.851E+00	1.142386	1.940E-01
2.858E+00	1.145545	2.060E-01
2.864E+00	1.152284	2.182E-01
2.871E+00	1.162372	2.290E-01
2.884E+00	1.178446	2.392E-01
2.891E+00	1.185419	2.493E-01
2.897E+00	1.195889	2.581E-01
2.904E+00	1.208002	2.647E-01
2.917E+00	1.229654	2.715E-01
2.924E+00	1.240033	2.759E-01
2.931E+00	1.252073	2.798E-01
2.938E+00	1.263935	2.804E-01
2.951E+00	1.285942	2.824E-01
2.958E+00	1.297762	2.817E-01
2.965E+00	1.307891	2.785E-01
2.979E+00	1.326310	2.759E-01
2.985E+00	1.334533	2.721E-01
2.999E+00	1.352917	2.721E-01
3.048E+00	1.411876	2.398E-01
3.097E+00	1.452013	1.918E-01
3.148E+00	1.466753	1.348E-01
3.199E+00	1.461522	9.243E-02
3.251E+00	1.449409	6.106E-02
3.304E+00	1.432585	3.688E-02
3.350E+00	1.417064	2.611E-02
3.396E+00	1.404875	1.949E-02
3.451E+00	1.393260	1.321E-02
3.499E+00	1.384213	9.393E-03
3.548E+00	1.376092	6.789E-03
3.597E+00	1.368863	5.150E-03
3.648E+00	1.362546	4.234E-03
3.698E+00	1.356937	3.596E-03
3.750E+00	1.351891	3.402E-03
3.802E+00	1.347393	3.402E-03
3.846E+00	1.343958	3.530E-03
3.899E+00	1.340174	3.800E-03
3.954E+00	1.336658	4.157E-03
3.999E+00	1.333929	4.600E-03
4.046E+00	1.331403	5.067E-03
4.102E+00	1.328504	5.621E-03
4.150E+00	1.326183	6.220E-03
4.198E+00	1.323997	6.883E-03
4.246E+00	1.321906	7.600E-03
4.295E+00	1.319948	8.449E-03
4.345E+00	1.318113	9.307E-03
4.395E+00	1.316398	1.030E-02
4.446E+00	1.314920	1.140E-02
4.498E+00	1.313587	1.238E-02
4.550E+00	1.312483	1.361E-02
4.603E+00	1.311785	1.472E-02
4.645E+00	1.311588	1.548E-02
4.699E+00	1.311451	1.570E-02
4.753E+00	1.311148	1.552E-02
4.797E+00	1.310657	1.499E-02
4.853E+00	1.309721	1.441E-02
4.898E+00	1.308720	1.370E-02
4.955E+00	1.307228	1.312E-02
5.000E+00	1.305885	1.241E-02
5.047E+00	1.304258	1.180E-02
5.105E+00	1.301965	1.111E-02
5.152E+00	1.299910	1.061E-02
5.200E+00	1.297550	1.011E-02
5.248E+00	1.294933	9.904E-03
5.297E+00	1.292117	9.790E-03
5.346E+00	1.289015	9.881E-03
5.395E+00	1.285729	1.030E-02
5.445E+00	1.282194	1.078E-02
5.495E+00	1.278291	1.158E-02
5.546E+00	1.273883	1.258E-02
5.598E+00	1.268802	1.418E-02
5.649E+00	1.262994	1.659E-02
5.702E+00	1.256584	2.031E-02
5.754E+00	1.248370	2.482E-02
5.794E+00	1.242239	3.295E-02
5.848E+00	1.234896	4.323E-02
5.902E+00	1.229289	6.220E-02
5.957E+00	1.231892	8.646E-02
5.998E+00	1.242862	1.069E-01
6.053E+00	1.268459	1.250E-01
6.095E+00	1.295314	1.309E-01
6.152E+00	1.330121	1.172E-01
6.194E+00	1.341605	8.786E-02
6.252E+00	1.339863	6.947E-02
6.295E+00	1.335754	5.699E-02
6.353E+00	1.329242	4.952E-02
6.397E+00	1.325038	4.485E-02
6.457E+00	1.320468	4.176E-02
6.501E+00	1.317726	3.925E-02
6.546E+00	1.314837	3.731E-02
6.607E+00	1.311404	3.563E-02
6.653E+00	1.309021	3.450E-02
6.699E+00	1.306716	3.371E-02
6.745E+00	1.304521	3.310E-02
6.808E+00	1.301901	3.272E-02
6.855E+00	1.300125	3.242E-02
6.902E+00	1.298382	3.220E-02
6.950E+00	1.296751	3.212E-02
6.998E+00	1.295193	3.197E-02
7.047E+00	1.293609	3.190E-02
7.096E+00	1.292093	3.197E-02
7.145E+00	1.290696	3.205E-02
7.194E+00	1.289296	3.205E-02
7.244E+00	1.287944	3.220E-02
7.295E+00	1.286624	3.220E-02
7.345E+00	1.285242	3.227E-02
7.396E+00	1.283912	3.242E-02
7.447E+00	1.282606	3.249E-02
7.499E+00	1.281248	3.257E-02
7.551E+00	1.279895	3.272E-02
7.603E+00	1.278508	3.279E-02
7.656E+00	1.277123	3.302E-02
7.691E+00	1.276220	3.310E-02
7.745E+00	1.274794	3.325E-02
7.798E+00	1.273363	3.348E-02
7.852E+00	1.271952	3.371E-02
7.907E+00	1.270543	3.395E-02
7.943E+00	1.269613	3.410E-02
7.998E+00	1.268163	3.426E-02
8.054E+00	1.266657	3.450E-02
8.091E+00	1.265652	3.466E-02
8.147E+00	1.264125	3.490E-02
8.204E+00	1.262564	3.514E-02
8.241E+00	1.261488	3.530E-02
8.299E+00	1.259903	3.563E-02
8.356E+00	1.258240	3.579E-02
8.395E+00	1.257072	3.604E-02
8.453E+00	1.255384	3.637E-02
8.492E+00	1.254220	3.654E-02
8.551E+00	1.252405	3.688E-02
8.590E+00	1.251193	3.714E-02
8.650E+00	1.249353	3.748E-02
8.710E+00	1.247433	3.783E-02
8.750E+00	1.246095	3.809E-02
8.790E+00	1.244791	3.844E-02
8.851E+00	1.242789	3.880E-02
8.892E+00	1.241424	3.916E-02
8.954E+00	1.239322	3.952E-02
8.995E+00	1.237862	3.988E-02
9.057E+00	1.235657	4.035E-02
9.099E+00	1.234142	4.072E-02
9.141E+00	1.232659	4.110E-02
9.204E+00	1.230259	4.148E-02
9.247E+00	1.228589	4.196E-02
9.290E+00	1.226967	4.234E-02
9.354E+00	1.224439	4.293E-02
9.397E+00	1.222699	4.333E-02
9.441E+00	1.220909	4.373E-02
9.506E+00	1.218113	4.434E-02
9.550E+00	1.216115	4.475E-02
9.594E+00	1.214136	4.537E-02
9.661E+00	1.211068	4.600E-02
9.705E+00	1.208909	4.664E-02
9.750E+00	1.206729	4.718E-02
9.795E+00	1.204471	4.784E-02
9.840E+00	1.202228	4.851E-02
9.908E+00	1.198600	4.929E-02
9.954E+00	1.195932	4.998E-02
1.000E+01	1.193164	5.079E-02
1.005E+01	1.190334	5.174E-02
1.009E+01	1.187365	5.270E-02
1.014E+01	1.183900	5.380E-02
1.021E+01	1.180893	5.805E-02
1.026E+01	1.178360	5.634E-02
1.030E+01	1.174182	5.845E-02
1.035E+01	1.170827	5.995E-02
1.040E+01	1.167354	6.191E-02
1.045E+01	1.163960	6.394E-02
1.050E+01	1.160584	6.619E-02
1.054E+01	1.157248	6.852E-02
1.059E+01	1.153843	7.092E-02
1.064E+01	1.150368	7.359E-02
1.069E+01	1.146959	7.652E-02
1.074E+01	1.143601	7.958E-02
1.079E+01	1.140345	8.294E-02
1.084E+01	1.137372	8.646E-02
1.089E+01	1.134419	8.970E-02
1.094E+01	1.131445	9.328E-02
1.099E+01	1.128640	9.678E-02
1.104E+01	1.125466	9.995E-02
1.109E+01	1.122010	1.039E-01
1.114E+01	1.118841	1.083E-01
1.119E+01	1.116059	1.129E-01
1.125E+01	1.113289	1.172E-01
1.130E+01	1.110334	1.218E-01
1.135E+01	1.107674	1.270E-01
1.140E+01	1.105361	1.321E-01
1.146E+01	1.103057	1.370E-01
1.151E+01	1.100705	1.422E-01
1.156E+01	1.097503	1.472E-01
1.159E+01	1.096584	1.520E-01
1.164E+01	1.096068	1.570E-01
1.169E+01	1.094339	1.621E-01
1.175E+01	1.092339	1.678E-01
1.180E+01	1.090622	1.741E-01
1.186E+01	1.089062	1.802E-01
1.191E+01	1.086474	1.865E-01
1.194E+01	1.086163	1.927E-01
1.199E+01	1.087480	1.990E-01
1.205E+01	1.087926	2.055E-01
1.211E+01	1.087993	2.112E-01
1.216E+01	1.086723	2.177E-01
1.219E+01	1.087212	2.238E-01
1.225E+01	1.089721	2.295E-01
1.230E+01	1.090913	2.359E-01
1.236E+01	1.091270	2.420E-01
1.239E+01	1.092375	2.476E-01
1.245E+01	1.095643	2.528E-01
1.250E+01	1.098011	2.593E-01
1.256E+01	1.099603	2.641E-01
1.259E+01	1.100816	2.690E-01
1.265E+01	1.104624	2.740E-01
1.271E+01	1.107403	2.791E-01
1.276E+01	1.108999	2.837E-01
1.279E+01	1.110319	2.883E-01
1.285E+01	1.114243	2.929E-01
1.291E+01	1.116753	2.977E-01
1.294E+01	1.118262	3.012E-01
1.300E+01	1.122067	3.060E-01
1.306E+01	1.124841	3.103E-01
1.309E+01	1.126485	3.139E-01
1.315E+01	1.130583	3.183E-01
1.321E+01	1.133825	3.227E-01
1.324E+01	1.135773	3.257E-01
1.330E+01	1.139515	3.295E-01
1.334E+01	1.141428	3.325E-01
1.340E+01	1.145850	3.363E-01
1.346E+01	1.149628	3.402E-01
1.349E+01	1.151643	3.426E-01
1.355E+01	1.156338	3.466E-01
1.361E+01	1.160150	3.490E-01
1.365E+01	1.161869	3.514E-01
1.371E+01	1.165763	3.546E-01
1.374E+01	1.167947	3.571E-01
1.380E+01	1.172049	3.596E-01
1.384E+01	1.174089	3.621E-01
1.390E+01	1.178513	3.646E-01
1.396E+01	1.182458	3.680E-01
1.400E+01	1.184740	3.696E-01
1.406E+01	1.189086	3.722E-01
1.409E+01	1.191399	3.739E-01
1.416E+01	1.195603	3.757E-01
1.419E+01	1.197623	3.774E-01
1.426E+01	1.201594	3.791E-01
1.429E+01	1.203552	3.809E-01
1.435E+01	1.207465	3.826E-01
1.439E+01	1.209428	3.844E-01
1.445E+01	1.213645	3.862E-01
1.449E+01	1.215328	3.871E-01
1.455E+01	1.218762	3.898E-01
1.459E+01	1.220973	3.916E-01
1.466E+01	1.225566	3.934E-01
1.469E+01	1.227627	3.943E-01
1.476E+01	1.231631	3.961E-01
1.479E+01	1.233597	3.970E-01
1.486E+01	1.237500	3.988E-01
1.489E+01	1.239445	3.998E-01
1.496E+01	1.243348	4.016E-01
1.500E+01	1.245318	4.025E-01
1.507E+01	1.249380	4.044E-01
1.510E+01	1.251704	4.053E-01
1.514E+01	1.253631	4.053E-01
1.521E+01	1.256977	4.072E-01
1.524E+01	1.258880	4.081E-01
1.531E+01	1.263173	4.100E-01
1.535E+01	1.265082	4.100E-01
1.542E+01	1.268440	4.119E-01
1.545E+01	1.270391	4.128E-01
1.549E+01	1.272559	4.138E-01
1.556E+01	1.276473	4.148E-01
1.560E+01	1.278233	4.157E-01
1.567E+01	1.282639	4.176E-01
1.570E+01	1.284709	4.176E-01
1.574E+01	1.286576	4.186E-01
1.581E+01	1.290576	4.196E-01
1.585E+01	1.292723	4.205E-01
1.589E+01	1.294706	4.205E-01
1.596E+01	1.298872	4.225E-01
1.600E+01	1.301310	4.225E-01
1.603E+01	1.303145	4.225E-01
1.611E+01	1.306556	4.234E-01
1.614E+01	1.308540	4.244E-01
1.622E+01	1.313112	4.254E-01
1.626E+01	1.315327	4.254E-01
1.629E+01	1.317122	4.254E-01
1.637E+01	1.320901	4.264E-01
1.641E+01	1.322675	4.264E-01
1.644E+01	1.324631	4.274E-01
1.648E+01	1.326773	4.274E-01
1.656E+01	1.330870	4.283E-01
1.660E+01	1.333056	4.283E-01
1.663E+01	1.334869	4.283E-01
1.671E+01	1.338869	4.293E-01
1.675E+01	1.341074	4.293E-01
1.679E+01	1.342949	4.293E-01
1.687E+01	1.347481	4.303E-01
1.690E+01	1.349696	4.293E-01
1.694E+01	1.351233	4.293E-01
1.698E+01	1.352834	4.293E-01
1.706E+01	1.356772	4.303E-01
1.710E+01	1.359301	4.303E-01
1.714E+01	1.361083	4.293E-01
1.722E+01	1.364655	4.303E-01
1.726E+01	1.367219	4.303E-01
1.730E+01	1.369211	4.293E-01
1.734E+01	1.370751	4.293E-01
1.742E+01	1.374519	4.293E-01
1.746E+01	1.376758	4.293E-01
1.750E+01	1.378598	4.283E-01
1.754E+01	1.380029	4.283E-01
1.762E+01	1.383660	4.283E-01
1.766E+01	1.385875	4.283E-01
1.770E+01	1.387734	4.274E-01
1.774E+01	1.389417	4.274E-01
1.778E+01	1.390838	4.264E-01
1.786E+01	1.394313	4.274E-01
1.791E+01	1.396377	4.264E-01
1.795E+01	1.398169	4.264E-01
1.799E+01	1.399826	4.254E-01
1.803E+01	1.401123	4.254E-01
1.811E+01	1.404604	4.254E-01
1.816E+01	1.406786	4.254E-01
1.820E+01	1.408657	4.244E-01
1.824E+01	1.410419	4.244E-01
1.828E+01	1.412092	4.234E-01
1.837E+01	1.415276	4.234E-01
1.841E+01	1.417548	4.234E-01
1.845E+01	1.419809	4.225E-01
1.849E+01	1.421557	4.215E-01
1.854E+01	1.422820	4.205E-01
1.862E+01	1.426178	4.205E-01
1.866E+01	1.428308	4.196E-01
1.871E+01	1.429982	4.186E-01
1.875E+01	1.431240	4.176E-01
1.879E+01	1.432797	4.176E-01
1.884E+01	1.434643	4.167E-01
1.888E+01	1.435881	4.157E-01
1.897E+01	1.439563	4.157E-01
1.901E+01	1.441618	4.138E-01
1.905E+01	1.442846	4.128E-01
1.910E+01	1.444197	4.119E-01
1.914E+01	1.445486	4.110E-01
1.919E+01	1.446666	4.100E-01
1.923E+01	1.447502	4.091E-01
1.932E+01	1.450255	4.091E-01
1.936E+01	1.452188	4.081E-01
1.941E+01	1.453825	4.072E-01
1.945E+01	1.455604	4.062E-01
1.950E+01	1.456898	4.044E-01
1.954E+01	1.457713	4.035E-01
1.959E+01	1.458719	4.025E-01
1.963E+01	1.459690	4.016E-01
1.968E+01	1.460391	4.007E-01
1.977E+01	1.463349	4.007E-01
1.982E+01	1.465400	3.988E-01
1.986E+01	1.466543	3.970E-01
1.991E+01	1.467000	3.952E-01
1.995E+01	1.467249	3.943E-01
2.000E+01	1.467642	3.934E-01
2.099E+01	1.483693	3.818E-01
2.198E+01	1.499422	3.722E-01
2.301E+01	1.516402	3.629E-01
2.399E+01	1.529309	3.482E-01
2.500E+01	1.537967	3.356E-01
2.600E+01	1.544080	3.227E-01
2.698E+01	1.546670	3.103E-01
2.799E+01	1.546272	2.991E-01
2.897E+01	1.542658	2.889E-01
2.999E+01	1.535500	2.817E-01
3.097E+01	1.527225	2.791E-01
3.199E+01	1.519076	2.798E-01
3.304E+01	1.511879	2.830E-01
3.396E+01	1.505906	2.863E-01
3.499E+01	1.498932	2.916E-01
3.597E+01	1.492960	2.991E-01
3.698E+01	1.486740	3.068E-01
3.802E+01	1.481006	3.190E-01
3.899E+01	1.478232	3.317E-01
3.999E+01	1.476571	3.442E-01
4.102E+01	1.475642	3.587E-01
4.198E+01	1.477194	3.739E-01
4.295E+01	1.480747	3.880E-01
4.395E+01	1.485266	4.025E-01
4.498E+01	1.491543	4.176E-01
4.603E+01	1.499424	4.323E-01
4.699E+01	1.508821	4.465E-01
4.797E+01	1.520272	4.579E-01
4.898E+01	1.531473	4.675E-01
5.000E+01	1.542270	4.773E-01
5.200E+01	1.567492	4.975E-01
5.395E+01	1.594131	5.079E-01
5.598E+01	1.619157	5.162E-01
5.794E+01	1.643739	5.233E-01
5.998E+01	1.669053	5.258E-01
6.194E+01	1.690223	5.246E-01
6.397E+01	1.709762	5.246E-01
6.607E+01	1.729441	5.233E-01
6.808E+01	1.747333	5.209E-01
6.998E+01	1.762824	5.174E-01
7.194E+01	1.777162	5.138E-01
7.396E+01	1.790800	5.103E-01
7.603E+01	1.805539	5.079E-01
7.798E+01	1.819110	5.021E-01
7.998E+01	1.830882	4.964E-01
8.204E+01	1.842330	4.907E-01
8.395E+01	1.851943	4.839E-01
8.590E+01	1.859854	4.773E-01
8.790E+01	1.867327	4.718E-01
8.995E+01	1.874242	4.654E-01
9.204E+01	1.880545	4.600E-01
9.397E+01	1.886330	4.548E-01
9.594E+01	1.891384	4.485E-01
9.795E+01	1.895435	4.434E-01
1.000E+02	1.899131	4.383E-01
1.099E+02	1.907505	4.176E-01
1.199E+02	1.911671	4.167E-01
1.300E+02	1.919973	4.186E-01
1.400E+02	1.927412	4.205E-01
1.500E+02	1.934154	4.264E-01
1.600E+02	1.941655	4.333E-01
1.698E+02	1.948419	4.403E-01
1.799E+02	1.955736	4.506E-01
1.901E+02	1.965156	4.611E-01
2.000E+02	1.974559	4.697E-01
2.099E+02	1.983438	4.784E-01
2.198E+02	1.992287	4.873E-01
2.301E+02	2.001418	4.964E-01
2.399E+02	2.010446	5.056E-01
2.500E+02	2.020318	5.138E-01
2.600E+02	2.029224	5.209E-01
2.698E+02	2.037243	5.282E-01
2.799E+02	2.045135	5.355E-01
2.897E+02	2.052476	5.430E-01
2.999E+02	2.059773	5.505E-01
3.199E+02	2.073976	5.660E-01
3.396E+02	2.086956	5.805E-01
3.597E+02	2.099543	5.967E-01
3.802E+02	2.112811	6.135E-01
3.999E+02	2.125742	6.292E-01
4.198E+02	2.139507	6.453E-01
4.395E+02	2.153213	6.589E-01
4.603E+02	2.166254	6.727E-01
4.797E+02	2.177335	6.852E-01
5.000E+02	2.188736	6.995E-01
5.200E+02	2.200349	7.125E-01
5.395E+02	2.210869	7.241E-01
5.598E+02	2.220374	7.359E-01
5.794E+02	2.228339	7.478E-01
5.998E+02	2.236685	7.617E-01
6.501E+02	2.254575	7.921E-01
6.998E+02	2.270109	8.314E-01
7.499E+02	2.290196	8.725E-01
7.998E+02	2.312599	9.116E-01
8.492E+02	2.337241	9.501E-01
8.995E+02	2.363856	9.835E-01
9.506E+02	2.385313	1.011E+00
1.000E+03	2.399111	1.042E+00
1.100E+03	2.436760	1.120E+00
1.200E+03	2.481153	1.191E+00
1.300E+03	2.527536	1.261E+00
1.400E+03	2.577344	1.330E+00
1.500E+03	2.629097	1.393E+00
1.600E+03	2.679108	1.452E+00
1.700E+03	2.729264	1.511E+00
1.800E+03	2.781861	1.567E+00
1.900E+03	2.831974	1.617E+00
2.000E+03	2.881863	1.670E+00
2.100E+03	2.933900	1.718E+00
2.200E+03	2.983258	1.763E+00
2.300E+03	3.032401	1.809E+00
2.400E+03	3.084049	1.853E+00
2.500E+03	3.133464	1.891E+00
2.600E+03	3.179887	1.931E+00
2.700E+03	3.228984	1.973E+00
2.800E+03	3.279470	2.009E+00
2.900E+03	3.326631	2.043E+00
3.000E+03	3.374610	2.079E+00
3.100E+03	3.422465	2.110E+00
3.200E+03	3.468221	2.142E+00
3.300E+03	3.516889	2.174E+00
3.400E+03	3.563346	2.199E+00
3.500E+03	3.607096	2.228E+00
3.600E+03	3.650102	2.255E+00
3.700E+03	3.695213	2.286E+00
3.800E+03	3.741930	2.310E+00
3.900E+03	3.785136	2.333E+00
4.000E+03	3.829496	2.360E+00
4.100E+03	3.873564	2.380E+00
4.200E+03	3.917021	2.404E+00
4.300E+03	3.960586	2.423E+00
4.400E+03	4.003601	2.445E+00
4.500E+03	4.045111	2.460E+00
4.600E+03	4.084851	2.481E+00
4.700E+03	4.125763	2.500E+00
4.800E+03	4.166540	2.518E+00
4.900E+03	4.207585	2.535E+00
5.000E+03	4.248425	2.551E+00
5.100E+03	4.288766	2.567E+00
5.200E+03	4.328263	2.581E+00
5.300E+03	4.367201	2.595E+00
5.400E+03	4.403706	2.607E+00
5.500E+03	4.442443	2.625E+00
5.600E+03	4.482558	2.636E+00
5.700E+03	4.518750	2.647E+00
5.800E+03	4.555811	2.662E+00
5.901E+03	4.593558	2.671E+00
6.001E+03	4.631138	2.685E+00
6.100E+03	4.667698	2.692E+00
6.200E+03	4.704528	2.705E+00
6.299E+03	4.740422	2.711E+00
6.400E+03	4.776470	2.723E+00
6.500E+03	4.811400	2.727E+00
6.599E+03	4.844068	2.737E+00
6.700E+03	4.881015	2.747E+00
6.800E+03	4.915201	2.750E+00
6.899E+03	4.946351	2.758E+00
7.000E+03	4.979800	2.766E+00
7.101E+03	5.013994	2.774E+00
7.199E+03	5.047771	2.780E+00
7.300E+03	5.083439	2.787E+00
7.399E+03	5.116001	2.786E+00
7.501E+03	5.146161	2.792E+00
7.600E+03	5.177179	2.796E+00
7.700E+03	5.209531	2.801E+00
7.800E+03	5.241539	2.805E+00
7.900E+03	5.273172	2.808E+00
8.000E+03	5.304929	2.811E+00
8.100E+03	5.336323	2.814E+00
8.200E+03	5.367389	2.816E+00
8.300E+03	5.398286	2.824E+00
8.400E+03	5.428878	2.825E+00
8.500E+03	5.459208	2.826E+00
8.600E+03	5.489262	2.827E+00
8.700E+03	5.519027	2.827E+00
8.800E+03	5.548489	2.833E+00
8.900E+03	5.577699	2.833E+00
9.000E+03	5.606586	2.832E+00
9.100E+03	5.635201	2.830E+00
9.200E+03	5.663535	2.835E+00
9.300E+03	5.691521	2.833E+00
9.400E+03	5.719272	2.831E+00
9.500E+03	5.746661	2.835E+00
9.600E+03	5.773802	2.832E+00
9.700E+03	5.800631	2.835E+00
9.800E+03	5.827179	2.831E+00
9.900E+03	5.853423	2.828E+00
1.000E+04	5.879378	2.830E+00
1.100E+04	6.131865	2.807E+00
1.200E+04	6.346035	2.773E+00
1.300E+04	6.538143	2.733E+00
1.400E+04	6.711149	2.685E+00
1.500E+04	6.867192	2.630E+00
1.600E+04	7.007965	2.576E+00
1.700E+04	7.135674	2.519E+00
1.800E+04	7.252419	2.461E+00
1.900E+04	7.358822	2.397E+00
2.000E+04	7.455943	2.338E+00
2.100E+04	7.544423	2.280E+00
2.200E+04	7.625553	2.224E+00
2.300E+04	7.701126	2.170E+00
2.400E+04	7.768902	2.113E+00
2.500E+04	7.831158	2.064E+00
2.600E+04	7.889643	2.011E+00
2.700E+04	7.941322	1.961E+00
2.800E+04	7.989355	1.914E+00
2.900E+04	8.033791	1.868E+00
3.000E+04	8.074469	1.824E+00
3.100E+04	8.112180	1.781E+00
3.200E+04	8.147128	1.741E+00
3.300E+04	8.179843	1.701E+00
3.400E+04	8.209818	1.663E+00
3.500E+04	8.238281	1.627E+00
3.600E+04	8.264599	1.591E+00
3.700E+04	8.288448	1.557E+00
3.800E+04	8.311297	1.525E+00
3.900E+04	8.332788	1.493E+00
4.000E+04	8.352700	1.463E+00
4.100E+04	8.371979	1.434E+00
4.200E+04	8.389151	1.405E+00
4.300E+04	8.405156	1.378E+00
4.400E+04	8.420858	1.352E+00
4.500E+04	8.435660	1.327E+00
4.600E+04	8.449912	1.302E+00
4.700E+04	8.462521	1.278E+00
4.800E+04	8.474673	1.256E+00
4.900E+04	8.486470	1.233E+00
5.000E+04	8.497290	1.212E+00
5.100E+04	8.507814	1.191E+00
5.200E+04	8.517490	1.171E+00
5.300E+04	8.526995	1.152E+00
5.400E+04	8.535947	1.133E+00
5.500E+04	8.545560	1.115E+00
5.600E+04	8.554652	1.097E+00
5.700E+04	8.563363	1.080E+00
5.800E+04	8.571605	1.063E+00
5.901E+04	8.579613	1.047E+00
6.001E+04	8.587191	1.031E+00
6.100E+04	8.594362	1.016E+00
6.200E+04	8.601342	1.001E+00
6.299E+04	8.607948	9.867E-01
6.400E+04	8.614382	9.727E-01
6.500E+04	8.620473	9.589E-01
6.599E+04	8.626323	9.456E-01
6.700E+04	8.632030	9.328E-01
6.800E+04	8.637431	9.202E-01
6.899E+04	8.642623	9.078E-01
7.000E+04	8.647693	8.960E-01
7.101E+04	8.652568	8.845E-01
7.199E+04	8.657181	8.728E-01
7.300E+04	8.661691	8.618E-01
7.399E+04	8.666029	8.509E-01
7.501E+04	8.670273	8.406E-01
7.600E+04	8.674287	8.302E-01
7.700E+04	8.678217	8.201E-01
7.800E+04	8.681999	8.104E-01
7.900E+04	8.685638	8.007E-01
8.000E+04	8.689204	7.914E-01
8.100E+04	8.692637	7.822E-01
8.200E+04	8.695941	7.733E-01
8.300E+04	8.699181	7.645E-01
8.400E+04	8.702404	7.560E-01
8.500E+04	8.705507	7.476E-01
8.600E+04	8.708553	7.394E-01
8.700E+04	8.711486	7.314E-01
8.800E+04	8.714366	7.236E-01
8.900E+04	8.717139	7.159E-01
8.999E+04	8.719808	7.083E-01
9.099E+04	8.722432	7.009E-01
9.200E+04	8.725011	6.938E-01
9.300E+04	8.727494	6.867E-01
9.399E+04	8.729884	6.798E-01
9.499E+04	8.732236	6.730E-01
9.601E+04	8.734549	6.663E-01
9.701E+04	8.736776	6.598E-01
9.799E+04	8.738919	6.534E-01
9.899E+04	8.741029	6.471E-01
1.000E+05	8.743107	6.409E-01
1.099E+05	8.761392	5.845E-01
1.199E+05	8.776133	5.380E-01
1.300E+05	8.788091	4.983E-01
1.400E+05	8.797836	4.636E-01
1.500E+05	8.804337	4.337E-01
1.600E+05	8.809677	4.073E-01
1.698E+05	8.814082	3.837E-01
1.799E+05	8.817867	3.630E-01
1.901E+05	8.821129	3.448E-01
2.000E+05	8.823834	3.277E-01
2.099E+05	8.826184	3.123E-01
2.198E+05	8.828230	2.983E-01
2.301E+05	8.830103	2.860E-01
2.399E+05	8.831656	2.741E-01
2.500E+05	8.833091	2.634E-01
2.600E+05	8.834348	2.535E-01
2.698E+05	8.835449	2.441E-01
2.799E+05	8.836475	2.356E-01
2.897E+05	8.837375	2.274E-01
2.999E+05	8.838141	2.202E-01
3.097E+05	8.838809	2.132E-01
3.199E+05	8.839434	2.064E-01
3.304E+05	8.840020	2.008E-01
3.396E+05	8.840491	1.944E-01
3.499E+05	8.840974	1.891E-01
3.597E+05	8.841394	1.840E-01
3.698E+05	8.841790	1.790E-01
3.802E+05	8.842166	1.745E-01
3.899E+05	8.842491	1.701E-01
3.999E+05	8.842801	1.659E-01
4.102E+05	8.843095	1.621E-01
4.198E+05	8.843349	1.580E-01
4.295E+05	8.843591	1.545E-01
4.395E+05	8.843823	1.509E-01
4.498E+05	8.844044	1.478E-01
4.603E+05	8.844254	1.448E-01
4.699E+05	8.844436	1.415E-01
4.797E+05	8.844609	1.386E-01
4.898E+05	8.844776	1.358E-01
5.000E+05	8.844936	1.333E-01
5.105E+05	8.845089	1.309E-01
5.200E+05	8.845220	1.282E-01
5.297E+05	8.845345	1.258E-01
5.395E+05	8.845467	1.233E-01
5.495E+05	8.845583	1.213E-01
5.598E+05	8.845696	1.191E-01
5.702E+05	8.845804	1.172E-01
5.794E+05	8.845896	1.150E-01
5.902E+05	8.845997	1.132E-01
5.998E+05	8.846082	1.114E-01
6.095E+05	8.846164	1.093E-01
6.194E+05	8.846244	1.076E-01
6.295E+05	8.846321	1.061E-01
6.397E+05	8.846396	1.044E-01
6.501E+05	8.846468	1.030E-01
6.607E+05	8.846538	1.013E-01
6.699E+05	8.846596	9.995E-02
6.808E+05	8.846662	9.858E-02
6.902E+05	8.846716	9.700E-02
6.998E+05	8.846769	9.567E-02
7.096E+05	8.846821	9.436E-02
7.194E+05	8.846871	9.285E-02
7.295E+05	8.846920	9.179E-02
7.396E+05	8.846967	9.053E-02
7.499E+05	8.847013	8.929E-02
7.603E+05	8.847058	8.827E-02
7.691E+05	8.847094	8.705E-02
7.798E+05	8.847137	8.586E-02
7.907E+05	8.847178	8.488E-02
7.998E+05	8.847212	8.391E-02
8.091E+05	8.847244	8.275E-02
8.204E+05	8.847283	8.181E-02
8.299E+05	8.847314	8.087E-02
8.395E+05	8.847344	7.976E-02
8.492E+05	8.847374	7.885E-02
8.590E+05	8.847403	7.795E-02
8.710E+05	8.847437	7.723E-02
8.790E+05	8.847459	7.617E-02
8.892E+05	8.847485	7.547E-02
8.995E+05	8.847512	7.461E-02
9.099E+05	8.847537	7.375E-02
9.204E+05	8.847563	7.308E-02
9.290E+05	8.847582	7.224E-02
9.397E+05	8.847606	7.142E-02
9.506E+05	8.847630	7.076E-02
9.594E+05	8.847648	6.995E-02
9.705E+05	8.847671	6.931E-02
9.795E+05	8.847688	6.852E-02
9.908E+05	8.847710	6.804E-02
1.000E+06	8.847727	6.727E-02
1.099E+06	8.847889	6.121E-02
1.199E+06	8.848013	5.621E-02
1.300E+06	8.848109	5.197E-02
1.400E+06	8.848183	4.828E-02
1.500E+06	8.848243	4.506E-02
1.600E+06	8.848291	4.234E-02
1.698E+06	8.848330	3.979E-02
1.799E+06	8.848364	3.765E-02
1.901E+06	8.848392	3.571E-02
2.000E+06	8.848415	3.395E-02
2.099E+06	8.848435	3.234E-02
2.198E+06	8.848452	3.089E-02
2.301E+06	8.848467	2.957E-02
2.399E+06	8.848480	2.837E-02
2.500E+06	8.848491	2.721E-02
2.600E+06	8.848501	2.623E-02
2.698E+06	8.848510	2.522E-02
2.799E+06	8.848518	2.437E-02
2.897E+06	8.848524	2.348E-02
2.999E+06	8.848531	2.274E-02
3.097E+06	8.848536	2.202E-02
3.199E+06	8.848541	2.132E-02
3.304E+06	8.848546	2.074E-02
3.396E+06	8.848550	2.008E-02
3.499E+06	8.848554	1.953E-02
3.597E+06	8.848557	1.900E-02
3.698E+06	8.848560	1.848E-02
3.802E+06	8.848563	1.802E-02
3.899E+06	8.848566	1.757E-02
3.999E+06	8.848568	1.713E-02
4.102E+06	8.848570	1.670E-02
4.198E+06	8.848572	1.632E-02
4.295E+06	8.848574	1.591E-02
4.395E+06	8.848576	1.559E-02
4.498E+06	8.848577	1.523E-02
4.603E+06	8.848579	1.492E-02
4.699E+06	8.848580	1.458E-02
4.797E+06	8.848581	1.428E-02
4.898E+06	8.848582	1.399E-02
5.000E+06	8.848584	1.373E-02
5.105E+06	8.848585	1.348E-02
5.200E+06	8.848585	1.321E-02
5.297E+06	8.848586	1.297E-02
5.395E+06	8.848587	1.273E-02
5.495E+06	8.848588	1.250E-02
5.598E+06	8.848589	1.227E-02
5.702E+06	8.848589	1.207E-02
5.794E+06	8.848590	1.185E-02
5.902E+06	8.848591	1.166E-02
5.998E+06	8.848591	1.148E-02
6.095E+06	8.848592	1.129E-02
6.194E+06	8.848592	1.111E-02
6.295E+06	8.848593	1.091E-02
6.397E+06	8.848593	1.076E-02
6.501E+06	8.848594	1.061E-02
6.607E+06	8.848594	1.044E-02
6.699E+06	8.848594	1.030E-02
6.808E+06	8.848595	1.016E-02
6.902E+06	8.848595	9.995E-03
6.998E+06	8.848595	9.858E-03
7.096E+06	8.848596	9.700E-03
7.194E+06	8.848596	9.567E-03
7.295E+06	8.848596	9.458E-03
7.396E+06	8.848596	9.328E-03
7.499E+06	8.848597	9.200E-03
7.603E+06	8.848597	9.095E-03
7.691E+06	8.848597	8.949E-03
7.798E+06	8.848597	8.847E-03
7.907E+06	8.848598	8.746E-03
7.998E+06	8.848598	8.626E-03
8.091E+06	8.848598	8.527E-03
8.204E+06	8.848598	8.429E-03
8.299E+06	8.848598	8.333E-03
8.395E+06	8.848598	8.218E-03
8.492E+06	8.848598	8.124E-03
8.590E+06	8.848599	8.031E-03
8.710E+06	8.848599	7.958E-03
8.790E+06	8.848599	7.849E-03
8.892E+06	8.848599	7.759E-03
8.995E+06	8.848599	7.688E-03
9.099E+06	8.848599	7.600E-03
9.204E+06	8.848599	7.530E-03
9.290E+06	8.848599	7.444E-03
9.397E+06	8.848600	7.359E-03
9.506E+06	8.848600	7.291E-03
9.594E+06	8.848600	7.208E-03
9.705E+06	8.848600	7.142E-03
9.795E+06	8.848600	7.060E-03
9.908E+06	8.848600	7.011E-03
1.000E+07	8.848600	6.931E-03
];