"""This module deals with control file creation and data input/output for the
stable arts 1 package.  This is used only for the creation of GasAbsLookup
tables. To use this module you need to set the envoronment variable ARTS1_PATH
or manually set arts1.ARTS1_EXEC after importing this module"""
from PyARTS.arts_file_components import *

ARTS1_EXEC=os.getenv('ARTS1_PATH')

class cont_descriptionAppend(WSM):
    def __init__(self,tagname,model):
        self.name='cont_descriptionAppend'
        self.g_args=[]
        self.k_args={'tagname':quotify(tagname),
                     'model':quotify(model),
                     'userparameters':'[ ]'}
        self.keylist=['tagname','model','userparameters']

def abs_file(tags,hitran_filename,line_fmin,line_fmax,atm_basename,p_file,
             T_offset,f_file):
    """creates a command list corresponding to the calculation of absorption
    coefficients for specified 1D atmospheric fields. This function is used by
    the GasAbsLookup class"""
    tagstring='['
    for tag in tags[:-1]:
        tagstring+=quotify(tag)+','
    tagstring+=quotify(tags[-1])+']'
    command_list=[WSM('tgsDefine',k_args={'tags':tagstring}),
                  WSM('lineshapeDefine',
                      k_args={'shape':quotify('Voigt_Kuntz6'),
                              'normalizationfactor':quotify('quadratic'),
                              'cutoff':750e9},
                      keylist=['shape','normalizationfactor','cutoff']),
                  WSM('linesReadFromHitran',
                      k_args={'filename':quotify(hitran_filename),
                              'fmin':line_fmin,
                              'fmax':line_fmax},
                      keylist=['filename','fmin','fmax']),
                  WSM('lines_per_tgCreateFromLines'),
                  WSM('cont_descriptionInit'),
                  cont_descriptionAppend("H2O-SelfContStandardType","Rosenkranz"),
                  cont_descriptionAppend("H2O-ForeignContStandardType","Rosenkranz"),
                  cont_descriptionAppend("H2O-ContMPM93","MPM93"),
                  cont_descriptionAppend("O2-SelfContStandardType","Rosenkranz"),
                  cont_descriptionAppend("N2-SelfContStandardType","Rosenkranz"),
                  cont_descriptionAppend("N2-SelfContMPM93","MPM93"),
                  cont_descriptionAppend("CO2-SelfContPWR93","Rosenkranz"),
                  cont_descriptionAppend("CO2-ForeignContPWR93","Rosenkranz"),
                  cont_descriptionAppend("H2O-CP98","CruzPol"),
                  cont_descriptionAppend("H2O-PWR98","Rosenkranz"),
                  cont_descriptionAppend("H2O-MPM87","MPM87"),
                  cont_descriptionAppend("H2O-MPM89","MPM89"),
                  cont_descriptionAppend("H2O-MPM93","MPM93"),
                  cont_descriptionAppend("O2-PWR93","Rosenkranz"),
                  cont_descriptionAppend("O2-MPM93","MPM93"),
                  WSM('MatrixReadAscii',
                      g_args=['raw_ptz'],
                      k_args={'filename':quotify(atm_basename+'.tz.aa')}),
                  WSM('raw_vmrsReadFromScenario',
                      k_args={'basename':quotify(atm_basename)}),
                  WSM('VectorReadAscii',
                      g_args=['p_abs'],
                      k_args={'filename':quotify(p_file)}),
                  WSM('AtmFromRaw'),
                  WSM('VectorAdd',
                      g_args=['t_abs','t_abs'],
                      k_args={'value':T_offset}),
                  WSM('h2o_absSet'),
                  WSM('n2_absSet'),
                  WSM('VectorWriteAscii',
                      g_args=['t_abs'],
                      k_args={'filename':'""'}),
                  WSM('MatrixWriteAscii',
                      g_args=['vmrs'],
                      k_args={'filename':'""'}),
                  WSM('VectorReadAscii',
                      g_args=['f_mono'],
                      k_args={'filename':quotify(f_file)}),
                  WSM('absCalc'),
                  WSM('ArrayOfMatrixWriteAscii',
                      g_args=['abs_per_tg'],
                      k_args={'filename':'""'})
                  ]
    return command_list

def arts1_file_from_command_list(command_list,filename):
    """takes a list of WSM and AgendaSet objects and creates an arts control file"""
    outstring = "#DEFINITIONS:  -*-sh-*-\n#This file was generated by " + \
                "arts_file_from_command_list \n#in the PyARTS python module\n\n"
    for command in command_list:
        outstring+=command.textify()
    
    outfile=open(filename,'w')
    outfile.write(outstring)
    outfile.close()

def arts1_save(x,filename):
    """save a numpy array (1D or 2D) in arts 1 ascii format"""
    dim=len(x.shape)
    assert(0<dim<3)
    outfile=file(filename,'w')
    outfile.write('1\n')
    shape=[1,1]
    for i in range(dim):
        shape[i]=x.shape[i]
    outfile.write(str(shape[0])+' '+str(shape[1])+'\n')
    for i in range(x.shape[0]):
        if dim==1:
            outfile.write(str(x[i])+'\n')
        else:
            for j in range(x.shape[1]):
                outfile.write(str(x[i,j])+'  ')
            outfile.write('\n')
    outfile.close()

def arts1_load(filename):
    """load an array aor a list of arrays from an arts 1 ascii file"""
    f=file(filename)
    skipping_header=True
    while skipping_header:
        s=f.readline()
        skipping_header=(s[0]=='#')
    #first number is number of array elements
    nelem=int(s.strip())
    arraylist=[]
    for i in range(nelem):
        #first row is array shape
        s=f.readline().strip().split()
        shape=(int(s[0]),int(s[1]))
        data=zeros(shape,Float)
        for j in range(shape[0]):
            line=f.readline().strip().split()
            for k in range(shape[1]):
                data[j,k]=float(line[k])
        arraylist.append(data)
    if nelem==1:
        #just return the single array
        return arraylist[0]
    else:
        return arraylist
