"""ARTS Single Scattering Data calculations.

Contains low-level function to calculate single-scattering properties.
The higher-level classes are arts_types.SingleScatteringData and the
various types in the clouds module.

"""

import os
import functools
import pickle
import subprocess
import copy

import numpy
import scipy

from warnings import warn   

from . import artsXML
from . import arts_math
#from . import arts_types
import arts_types
from . import general
from . import REFICE
from . import physics
from . import scatsubs
from . import tmd
from . import tmatrix

from .physics import c

#from IPython.Debugger import Tracer; debug_here = Tracer()
verbosity=0		  # Controls the verbosity of the whole module

class TMatConvError(general.PyARTSError): pass
class ReficeWError(general.PyARTSError): pass

def calc_SSP(f_grid, T_grid, za_grid, aa_grid, r_v, aspect_ratio, NP, precision,
             phase, ptype, mrr=None, mri=None):
    """Calculates single scattering properties:
    - extinction matrix
    - phase matrix
    - absorption vector

    Not meant to be called directly; called by SingleScatteringData.calc
    """
    # some name shortening
    if NP is None:
        raise general.PyARTSError('"NP" is None, cannot use t-matrix code')

    nf = len(f_grid)
    nT = len(T_grid)
    nza = len(za_grid)
    lam = numpy.zeros(nf, numpy.float32) # wavelength in microns
    for fi in range(nf):
        freq = f_grid[fi]
        lam[fi] = c/freq*1e6	
    ## refractive index data is either supplied in the initialisation parameters
    ## or calculated from temperature frequency and phase information
    if mrr is None:
        # Calculated refractive index data
        mrr = numpy.zeros([nf, nT], float)
        mri = numpy.zeros([nf, nT], float)
        for fi in range(nf):
            freq = f_grid[fi]
            for Ti in range(nT):
                T = T_grid[Ti]
                if phase == "ice":
                    # use Stephen Warren's code
                    m = refice(freq, T)
                elif phase == "liquid":
                    m = refliquid(freq, T)
                else:
                    raise PyARTSError("Invalid phase: %s. Must be ice or " \
                                      "liquid." % phase)
                mrr[fi,Ti] = m.real
                mri[fi,Ti] = m.imag

    # Calculation of optical properties begins

    if ptype == 20:

        pha_mat_data = numpy.zeros([nf, nT, nza, 6], numpy.float32)
        ext_mat_data = numpy.zeros([nf, nT, 1, 1, 1], numpy.float32)
        abs_vec_data = numpy.zeros([nf, nT, 1, 1, 1], numpy.float32)

        for f_index in range(nf):
            for Ti in range(nT):
                # use partial to avoid code duplication
                calc_tmat = functools.partial(tmat_rnd,
                    r_v, aspect_ratio, NP, lam[f_index], mrr[f_index, Ti],
                    mri[f_index, Ti], precision, nza)
                try:
                    tup = calc_tmat(use_quad=0)
                except TMatConvError:
                    """try quad precision"""
                    if 'tmq' in dir(tmd):
                        tup = calc_tmat(use_quad=1)
                    else:
                        raise
                Cext, Csca, F11, F22, F33, F44, F12, F34 = tup
                ext_mat_data[f_index, Ti, 0, 0, 0] = Cext*1e-12 # um^2 -> m^2
                abs_vec_data[f_index, Ti, 0, 0, 0] = (Cext-Csca)*1e-12 # um^2 -> m^2

                mono_pha_mat_data = numpy.zeros([nza, 6],float)
                mono_pha_mat_data[:, 0] = F11
                mono_pha_mat_data[:, 1] = F12
                mono_pha_mat_data[:, 2] = F22
                mono_pha_mat_data[:, 3] = F33
                mono_pha_mat_data[:, 4] = F34
                mono_pha_mat_data[:, 5] = F44

                # Convert from scattering matrix to phase matrix in m^2
                pha_mat_data[f_index, Ti, :, :] = mono_pha_mat_data*Csca/4/numpy.pi*1e-12

        pha_mat_data.shape = (nf, nT, nza, 1, 1, 1, 6)
    elif ptype == 30:
        nza_inc = (nza-1)/2+1
        naa = len(aa_grid)
        pha_mat_data = []
        ext_mat_data = []
        abs_vec_data = numpy.zeros([nf, nza_inc, 1, 2], numpy.float32)
        Csca_data = []
        for f_index in range(nf):
            lam_f = lam[f_index]
            for Ti in range(nT):
                mrr_T = mrr[f_index, Ti]
                mri_T = mri[f_index, Ti]
                # Calculate Tmatrix
                calc_tmat = functools.partial(tmat_fxd, r_v, aspect_ratio,
                                              NP, lam_f, mrr_T, mri_T,
                                              precision)
                try:
                    calc_tmat(use_quad=0)
                except TMatConvError:
                    if 'tmatrixq' in dir(tmatrix):
                        calc_tmat(use_quad=1)
                    else:
                        raise
                # calculate phase matrix data
                if verbosity:
                    print "Calculating phase matrix data"
                for za_scat in za_grid:
                    for delta_aa in aa_grid:
                        for za_inc in za_grid[:nza_inc]:
                            phasmat_beta_alpha = functools.partial(phasmat,
                                  lam_f, za_inc, za_scat, 0.0, delta_aa)
                            if aspect_ratio < 1.0:
                                # We have a prolate particle, which will be
                                # laying on its side
                                phasmat_alpha = functools.partial(phasmat_beta_alpha, 90)
##                                def phasmat_alpha(alpha):
##                                    Z_alpha = phasmat(lam, za_inc,
##                                                      za_scat, 0.0,
##                                                      delta_aa,
##                                                      BETA, alpha)
##                                    return Z_alpha
                                Z = arts_math.gauss_leg(phasmat_alpha, 0.0, 180.0, 10)
                                Z /= 180.0
                            else:
                                # We have an oblate particle that remains in the
                                # original orientation.  Integration over
                                # orientations is not required
                                Z = phasmat_beta_alpha(0, 0.0)
##                                Z = phasmat(lam_f, za_inc, za_scat,0.0,delta_aa,
##                                          BETA,0.0)
                            pha_mat_data.append(Z)
                # Calculate Csca type integral for eventual calculation of K_abs
                for za_scat in za_grid[:nza_inc]:
                    if aspect_ratio < 1.0:
                        # We have a prolate particle, which will be
                        # laying on its side
                        BETA = 90
                        def phasmatsintheta(arglist):
                            za_inc1 = arglist[0]
                            delta_aa1 = arglist[1]
                            alpha1 = arglist[2]
                            Zsintheta = phasmat(lam_f, za_inc1, za_scat, 0.0,
                                                delta_aa1, BETA, alpha1) * \
                                                numpy.sin(za_inc1*numpy.pi/180)
                            return Zsintheta[:2, 0]
                        integral = arts_math.multi_gauss_leg(phasmatsintheta,
                                                             [[0.0,180.0], [0.0,180.0], [0.0,180.0]],
                                                             n=6)
                        Csca_data.append(2*integral/180.0)
                    else:
                        BETA = 0
                        def phasmatsintheta(arglist):
                            za_inc1 = arglist[0]
                            delta_aa1 = arglist[1]
                            Zsintheta = phasmat(lam_f, za_inc1, za_scat, 0.0,
                                              delta_aa1, BETA, 0.0) * \
                                              numpy.sin(za_inc1*numpy.pi/180)
                            return Zsintheta[:2, 0]
                        integral = arts_math.multi_gauss_leg(phasmatsintheta, 
                                                             [[0.0,180.0], [0.0,180.0]],
                                                             n=10)

                        Csca_data.append(2*integral)
                # Calculate extinction matrix data
                if verbosity:
                    print "Calculating extinction matrix data"
                BETA = 0
                alpha = 0
                if aspect_ratio < 1.0:
                    # Numerical method
                    # We have a prolate particle, which will be
                    # laying on its side

                    ##------------------------
                    #Using analytic averaging of the tmatrix
                    tmatrix.avgtmatrix(tmatrix.nmax)
                    for za_inc in za_grid[:nza_inc]:
                        K = numpy.take(extmat(tmatrix.nmax, lam_f, za_inc,
                                              0.0, BETA, alpha),
                                       [0, 1, 6])
                        ext_mat_data.append(K)

                else:
                    # We have an oblate particle that remains in the
                    # original orientation.  Integration over
                    # orientations is not required
                    for za_inc in za_grid[:nza_inc]:
                        K = numpy.take(extmat(tmatrix.nmax, lam_f, za_inc, 
                                              0.0, BETA, 0.0),
                                       [0, 1, 6])
                        ext_mat_data.append(K)

        #resize everything
        pha_mat_data = numpy.array(pha_mat_data)
        pha_mat_data.shape = (nf, nT, nza, naa, nza_inc, 1, 16)
        ext_mat_data = numpy.array(ext_mat_data)
        ext_mat_data.shape = (nf, nT, nza_inc, 1, 3)
        Csca_data = numpy.array(Csca_data)
        Csca_data.shape = (nf, nT, nza_inc, 1, 2)
        Csca_data *= numpy.pi**2/32400
        # Calculate Absorption coefficient vector
        if verbosity:
            print "Calculating absorption coefficient vector data"
        abs_vec_data = ext_mat_data[..., :2] - Csca_data
    else:
        raise PyARTSError("Only ptype = 20 and 30 are currently working")
    return ext_mat_data, pha_mat_data, abs_vec_data

def batch_generate(argdict,num_proc):
    """This function takes a dictionary with keys:
    ['T_grid', 'aa_grid', 'NP', 'equiv_radius', 'aspect_ratio', 'f_grid',
    'ptype', 'za_grid'] (just like the SingleScatteringData parameters).
    However in this case the values for ['T_grid', 'NP', 'equiv_radius',
    'aspect_ratio', 'f_grid','ptype'] are lists.  batch_generate cascades
    through these settings generating scattering data files for each
    combination.  The order of calculation is according to the hierarchy:
    ['f_grid','T_grid', 'ptype', 'NP', 'aspect_ratio', 'equiv_radius'] with
    'equiv_radius' changing the fastest.  These calculations can be done in
    num_proc parallel processes. A list of filenames (with an order
    corresponding to the above hierarchy) is returned
    """
    import cPickle
    params={"za_grid":argdict["za_grid"],"aa_grid":argdict["aa_grid"],
            'phase':argdict['phase']}
    ##########################################################
    #generate list of input strings
    stdinlist=[]
    stdoutlist=[]
    for f_grid in argdict["f_grid"]:
        params["f_grid"]=f_grid
        for T_grid in argdict["T_grid"]:
            params["T_grid"]=T_grid
            for ptype in argdict["ptype"]:
                params["ptype"]=ptype
                for NP in argdict["NP"]:
                    params["NP"]=NP
                    for ar in argdict["aspect_ratio"]:
                        params["aspect_ratio"]=ar
                        for r in argdict["equiv_radius"]:
                            params["equiv_radius"]=r
                            stdinlist.append([cPickle.dumps(params)])

    #define function calling scat_file_batch
    def call_proc(inarg):
        p = subprocess.Popen("nice scat_file_batch", shell=True,
          stdin=subprocess.PIPE, stdout=subprocess.PIPE, close_fds=True)
        p.stdin.write(inarg)
        p.stdin.close()

        out_text=p.stdout.readlines()[-3:]
        p.stdout.close()
        return out_text[-3]+out_text[-2]+out_text[-1]

    stdoutlist = general.multi_thread2(call_proc,stdinlist,num_proc,1)
    outdata=range(len(stdoutlist))
    for i in range(len(stdoutlist)):
        try:
            outdata[i]=cPickle.loads(stdoutlist[i])
        except:
            outdata[i]=stdoutlist[i]
    return outdata

def phasmat(LAM,THET0,THET,PHI0,PHI,BETA,alpha):
    """Calculates the phase matrix and returns it in m\ :sup:`2`.  This requires
    that the Tmatrix has already been calculated. See arts_scat.extmat for
    argument descriptions"""
    S11,S12,S21,S22=tmatrix.ampl(tmatrix.nmax,LAM,THET0,THET,PHI0,PHI,alpha,BETA)
    #Z=ampmat2phasmat(S11,S12,S21,S22)
    Z=scatsubs.phasmat(S11,S12,S21,S22)
    return Z*1e-12                      #convert to m^2

def ampmat2phasmat(S11,S12,S21,S22):
    """calculates the phase matrix from the amplitude matrix elements S11,etc
    The units of the returned phase matrix will be the units of Sij squared.
    This was separated from phasmat to clarify profiling"""
    Z=numpy.zeros([4,4],float)
    Z[0,0]=(0.5*(S11*S11.conjugate()+S12*S12.conjugate() \
                 +S21*S21.conjugate()+S22*S22.conjugate())).real
    Z[0,1]=(0.5*(S11*S11.conjugate()-S12*S12.conjugate()
                 +S21*S21.conjugate()-S22*S22.conjugate())).real
    Z[0,2]=(-S11*S12.conjugate()-S22*S21.conjugate()).real
    Z[0,3]=(1j*(S11*S12.conjugate()-S22*S21.conjugate())).real
    Z[1,0]=(0.5*(S11*S11.conjugate()+S12*S12.conjugate()
                 -S21*S21.conjugate()-S22*S22.conjugate())).real
    Z[1,1]=(0.5*(S11*S11.conjugate()-S12*S12.conjugate()
                 -S21*S21.conjugate()+S22*S22.conjugate())).real
    Z[1,2]=(-S11*S12.conjugate()+S22*S21.conjugate()).real
    Z[1,3]=(1j*(S11*S12.conjugate()+S22*S21.conjugate())).real
    Z[2,0]=(-S11*S21.conjugate()-S22*S12.conjugate()).real
    Z[2,1]=(-S11*S21.conjugate()+S22*S12.conjugate()).real
    Z[2,2]=(S11*S22.conjugate()+S12*S21.conjugate()).real
    Z[2,3]=(-1j*(S11*S22.conjugate()+S21*S12.conjugate())).real
    Z[3,0]=(1j*(S21*S11.conjugate()+S22*S12.conjugate())).real
    Z[3,1]=(1j*(S21*S11.conjugate()-S22*S12.conjugate())).real
    Z[3,2]=(-1j*(S22*S11.conjugate()-S12*S21.conjugate())).real
    Z[3,3]=(S22*S11.conjugate()-S12*S21.conjugate()).real
    return Z


def extmat(NMAX,LAM,THET0,PHI0,BETA,alpha):
    """Calculate the extinction matrix for a given wavelength (LAM), and
    a propagation direction given by zenith angle (THET0) and azimuthal
    angle (PHI0) , both in degrees.  BETA and alpha give the particles
    orientation (These angles are defined as in Mishchenkos paper [mishchenko00]_).
    The output is a 1D array with the 7 independent extinction matrix elements
    [KJJ,K12,K13,K14,K23,K24,K34] in m\ :m:`$^2$`. To call this method you
    must first call tmat_fxd_"""
    S11,S12,S21,S22=tmatrix.ampl(NMAX,LAM,THET0,THET0,PHI0,PHI0,alpha,BETA)
    KJJ = (-1j*(S11+S22)).real*LAM        #!in microns^2
    K12 = (1j*(S22-S11)).real*LAM
    K13 = (1j*(S12+S21)).real*LAM
    K14 = (S21-S12).real*LAM
    K23 = (-1j*(S21-S12)).real*LAM
    K24 = -(S21+S12).real*LAM
    K34 = (S22-S11).real*LAM
    return  numpy.array([KJJ,K12,K13,K14,K23,K24,K34])*1e-12 #convert to m^2


def combine(scat_data_list,pnd_vec):
    """Returns a single SingleScatteringData object obtained by summing over
    a list of SingleScatteringData objects, each one multiplied by the
    corresponding element in a vector of particle number densities.  Currently
    this function requires that all members of scat_data_list have the same
    value of ptype, and the same angular grids"""
    n = len(scat_data_list)
    assert n==len(pnd_vec), "scat_data_list and pnd_vec must be the same length"
#    params=scat_data_list[0].params
#    new_scat_data=arts_types.SingleScatteringData(params)#This is an example of default values
#                                        #not being helpful.
    new_scat_data = copy.copy(scat_data_list[0])
    new_scat_data.pha_mat_data=pnd_vec[0]*scat_data_list[0].pha_mat_data
    new_scat_data.ext_mat_data=pnd_vec[0]*scat_data_list[0].ext_mat_data
    new_scat_data.abs_vec_data=pnd_vec[0]*scat_data_list[0].abs_vec_data
    for i in range(1,n):
        assert scat_data_list[i].ptype == scat_data_list[0].ptype, "ptype must be the same"
        for k in ["f_grid","T_grid","za_grid","aa_grid"]:
            assert (getattr(scat_data_list[i], k) ==
                    getattr(scat_data_list[0], k)).all(), k + " must be the same"
        new_scat_data.pha_mat_data+=pnd_vec[i]*scat_data_list[i].pha_mat_data
        new_scat_data.ext_mat_data+=pnd_vec[i]*scat_data_list[i].ext_mat_data
        new_scat_data.abs_vec_data+=pnd_vec[i]*scat_data_list[i].abs_vec_data
    new_scat_data.description="This single scattering data file was made by "+\
                 "combining several particle types - some of the "+\
                 "following settings may not apply."
    return new_scat_data

def compress(pnd_file,scat_data_list,new_pnd_file,scat_file_basename):
    """Takes a 1D pnd_file, and a list of SingleScatteringData objects, and
    creates an artificial SingleScatteringData object for each non-zero pnd
    level by using the combine function, and outputs the new
    SingleScatteringData object in arts XML format.  A new pnd file is created,
    which only contains zeros and ones, effectively placing each new artificial
    SingleScatteringObject in at the correct level. The function returns the
    list of new SingleScatteringData objects, and a list of lists of Tensor3
    representing the new pnd field.  This function can save CPU time and memory
    in ARTS simulations when there are more particle types, than non-zero ice
    water content levels """
    pnd1D = arts_types.ArrayOfGriddedField3.load(pnd_file)
    N_pt=len(scat_data_list)
    assert(len(pnd1D.data)==N_pt),"pnd_file and scat_data_list have different numbers of particle types"
    pnd_vec=numpy.zeros(N_pt,float)
    p_grid=pnd1D.data[0]["p_grid"]
    N_plevels=len(p_grid)
    lat_grid=pnd1D.data[0]["lat_grid"]
    lon_grid=pnd1D.data[0]["lon_grid"]
    new_scat_data_list=[]
    new_pnd_data = arts_types.ArrayOfGriddedField3()
    non_zero_p_indices=[]
    for plevel_index in range(N_plevels):
        #pnd_profile=zeros([N_plevels,1,1],float)
        for pt_index in range(N_pt):
            pnd_vec[pt_index]=pnd1D.data[pt_index]["data"][plevel_index,0,0]
        if sometrue(pnd_vec!=0):
            new_scat_data_list.append(combine(scat_data_list,pnd_vec))
            #pnd_profile[plevel_index,0,0]=1
            #new_pnd_data.append([p_grid,lat_grid,lon_grid,pnd_profile])
            non_zero_p_indices.append(plevel_index)
    newN_pt=len(new_scat_data_list)

    #Trim unnecessary p_levels
    new_p_indices=copy.deepcopy(non_zero_p_indices)
    if (non_zero_p_indices[0]>0):
        #include p_level just above 1st non-zero pnd_level
        new_p_indices.insert(0,non_zero_p_indices[0]-1)
    if (non_zero_p_indices[0]>1):
        #inclued top p_level
        new_p_indices.insert(0,0)
    if (non_zero_p_indices[-1]<N_plevels-2):
        #append p_level just below the last non_zero pnd_level
        new_p_indices.append(non_zero_p_indices[-1]+1)
    if (non_zero_p_indices[-1]<N_plevels-1):
        #append bottom p_level
        new_p_indices.append(N_plevels-1)

    p_grid=numpy.take(p_grid,new_p_indices,0)
    new_N_plevels=len(new_p_indices)
    for pt_index in non_zero_p_indices:
        pnd_profile=numpy.zeros([new_N_plevels,1,1],float)
        pnd_profile[new_p_indices.index(pt_index),0,0]=1
        new_pnd_data.append(GriddedField3({"p_grid":p_grid,
                                           "lat_grid":lat_grid,
                                           "lon_grid":lon_grid,
                                           "data":pnd_profile}))	


    #Do file generation once everything else is done, so we don't have rubbish
    #files all over the show from failed calculations

    new_pnd_data.save(new_pnd_file)

    for i in range(newN_pt):
        new_scat_data_list[i].save(scat_file_basename+str(i)+".xml")

    return new_pnd_data,new_scat_data_list

def expand1Dpnd_field(old_pnd_field,lat1,lat2,lon1,lon2):
    """takes 1D pnd_raw data, which is an ArrayOfGriidedField3, and creates a
    3D version, with non-zero values confined to
    within lat1,lat2,lon1,lon2"""
    latmin=-90
    latmax=90
    lonmin=-180
    lonmax=180
    N_pt=len(old_pnd_field)
    lat_grid= numpy.array([latmin,lat1-0.01,lat1,lat2,lat2+0.01,latmax])
    lon_grid= numpy.array([lonmin,lon1-0.01,lon1,lon2,lon2+0.01,lonmax])
    p_grid=old_pnd_field[0]["p_grid"]
    N_plevels=len(p_grid)
    new_pnd_field = arts_types.ArrayOfGriddedField3()
    for old_field in old_pnd_field:
        new_pnd_data=numpy.zeros([N_plevels,6,6],float)
        old_pnd_data=old_field["data"][:,0,0]
        for lt_index in [2,3]:
            for ln_index in [2,3]:
                new_pnd_data[:,lt_index,ln_index]=old_pnd_data
        new_pnd_field.append(GriddedField3({"p_grid":p_grid,
                                            "lat_grid":lat_grid,
                                            "lon_grid":lon_grid,
                                            "data":new_pnd_data}))
    return new_pnd_field

def refice(freq,temp):
    """A wrapper for the REFICE fortran program - this time with frequency
    and temperature as the arguments, and a python exception is raised if
    inputs are out of range. freq in Hz, temp in K"""
    wavlen=c/freq*1e6		# wavelength in microns
    if ((wavlen<0.0443) or (wavlen>8.6e6)):
        raise ReficeWError("wavelength %s out of range (0.0043, 8.6e6)" % wavlen)

    if temp<213.16:
        #allow linear extrapolation of real part and logarithmic
        #extrapolation of imaginary part.
        warn("Temperature %.2f is below tabulated data range: returned value is " \
             "extrapolated" % temp)
        t0=213.16
        t1=220
        m0=REFICE.refice(wavlen,t0)
        m1=REFICE.refice(wavlen,t1)
        #imaginary part is extrapolated logarithmically
        a=numpy.log(m1.imag/m0.imag)/(t1-t0)
        b=m0.imag/numpy.exp(a*t0)
        mi=b*numpy.exp(a*temp)
        #real part is extrapolated linearly
        mr=m0.real+(temp-t0)*(m1.real-m0.real)/(t1-t0)
        return complex(mr,mi)

    elif temp>272.16:
        warn("Temperature %.2f is above tabulated data range: returned value is " \
             "that for 272.16 K" % temp)
        return REFICE.refice(wavlen,272.16)
    else:
        return REFICE.refice(wavlen,temp)

def refliquid(freq,temp):
    """Calculates the refractive index of liquid water, according to a
    model based on those of Liebe and Hufford, as used in the EOSMLS
    scattering code.  This has been checked with Table C-1 in the cloudy
    ATBD. freq in Hz, temp in K"""
    f=freq/1e9

    #code copied from RefractiveIndex.f90
    #calculate dielectric constant
    th=300./temp
    fp=20.09-142.4*(th-1.)+294*(th-1.)**2
    fs=590.-1500.*(th-1.)
    e0=77.66+103.3*(th-1.)
    e1=5.48
    e2=3.51
    x=(e0-e1)/(1+(f/fp)**2)+(e1-e2)/(1+(f/fs)**2)+e2
    y=(e0-e1)*f/fp/(1+(f/fp)**2)+(e1-e2)*f/fs/(1+(f/fs)**2)
    e=complex(x,y)

    #calculate refractive index
    return e**0.5

def tmat_fxd(equiv_radius,aspect_ratio,NP,lam,mrr,mri,precision,use_quad=0):
    """A simplified interface to the tmatrix.tmatrix function"""
    if mri<0:
        raise ValueError,"imaginary part of the refractive index (mri) must be positive"
    if use_quad:
        tmatrix.nmax,Csca,Cext,errmsg = tmatrix.tmatrixq(equiv_radius,NP,
                                                         lam,
                                                         aspect_ratio,
                                                         mrr,
                                                         mri,
                                                         precision,
                                                         not verbosity)
        error_log_file= general.DATA_PATH+'/scat/tmat_fxdq_'
    else:
        tmatrix.nmax,Csca,Cext,errmsg = tmatrix.tmatrix(equiv_radius,NP,
                                                        lam,
                                                        aspect_ratio,
                                                        mrr,
                                                        mri,
                                                        precision,
                                                        not verbosity)
        error_log_file= general.DATA_PATH+'/scat/tmat_fxd_'

    if (len(errmsg.strip())>0):
        if not os.path.exists( general.DATA_PATH):
            os.mkdir( general.DATA_PATH)
        if not os.path.exists( general.DATA_PATH+'/scat'):
            os.mkdir( general.DATA_PATH+'/scat')

        print errmsg
        if tmatrix.choice.ichoice==1:
            outfilename=error_log_file+'NAG_failures.pickle'
        else:
            outfilename=error_log_file+'failures.pickle'
        if os.path.exists(outfilename):
            mode='a'
        else:
            mode='w'
        outfile=open(outfilename,mode)
        pickle.dump({'equiv_radius':equiv_radius,
                     'aspect_ratio':aspect_ratio,
                     'NP':NP,
                     'lam':lam,
                     'mrr':mrr,
                     'mri':mri,
                     'precision':precision},outfile)
        outfile.close()
        raise TMatConvError

def tmat_rnd(equiv_radius,aspect_ratio,NP,lam,mrr,mri,precision,nza,use_quad=0):
    """A simplified interface to the tmd.tmd function"""
    if mri<0:
        raise ValueError,"imaginary part of the refractive index (mri) must be positive"
    if use_quad:
        reff,veff,Cext,Csca,w,asymm,F11,F22,F33,F44,F12,F34,errmsg = \
            tmd.tmq(1.0,          #meaning radius is equivalent volume
                    4,          #Gamma distribution
                    equiv_radius,
                    1,          # NPNAX PARAMETER
                    0.1,                # B
                    1.0,                # GAM is ignored
                    -1,         # NKMAX
                    aspect_ratio,
                    NP,
                    lam,
                    mrr,
                    mri,
                    precision,              # DDELT
                    nza,
                    2, # NDGS APPERNETLY SOME CARE IS REQUIRED HERE
                    0.9999999,  # R1RAT
                    1.0000001,  # R2RAT
                    1)  #quiet
        error_log_file= general.DATA_PATH+'/scat/tmat_rndq_'
    else:
        reff,veff,Cext,Csca,w,asymm,F11,F22,F33,F44,F12,F34,errmsg = \
            tmd.tmd(1.0,          #meaning radius is equivalent volume
                    4,          #Gamma distribution
                    equiv_radius,
                    1,          # NPNAX PARAMETER
                    0.1,                # B
                    1.0,                # GAM is ignored
                    -1,         # NKMAX
                    aspect_ratio,
                    NP,
                    lam,
                    mrr,
                    mri,
                    precision,              # DDELT
                    nza,
                    2, # NDGS APPERNETLY SOME CARE IS REQUIRED HERE
                    0.9999999,  # R1RAT
                    1.0000001,  # R2RAT
                    1)  #quiet

        error_log_file= general.DATA_PATH+'/scat/tmat_rnd_'

    if (len(errmsg.strip())>0):
        print errmsg
        if tmd.choice.ichoice==1:
            outfile=open(error_log_file+'NAG_failures.pickle','a')
        else:
            outfile=open(error_log_file+'failures.pickle','a')
        pickle.dump({'equiv_radius':equiv_radius,
                     'aspect_ratio':aspect_ratio,
                     'NP':NP,
                     'lam':lam,
                     'mrr':mrr,
                     'mri':mri,
                     'precision':precision},outfile)
        outfile.close()
        raise TMatConvError

    return Cext,Csca,F11,F22,F33,F44,F12,F34


##from arts_types import SingleScatteringData
