"""This file includes interpreter or general purpose functions
"""

import os
import cPickle
#from scipy import *
import sys
import copy
import numbers
import ConfigParser
from collections import namedtuple

import numpy
import scipy
import os.path
from . import constants

DATA_PATH=os.getenv('DATA_PATH', "/tmp")
##MYCODE_PATH=os.getenv('MYCODE_PATH', os.getcwd())

class PyARTSError(Exception):
    pass

class PyARTSWarning(UserWarning):
    pass

def edit(filename):
    """Opens filename in emacs. One day could be modified to take an existing
    module name as an argument"""
    pid=os.spawnlp(os.P_NOWAIT,'emacs','emacs','-bg','midnightblue','-fg',
                   'white','-cr','yellow','-ms','yellow', '-fn',
                   'lucidasanstypewriter-14',filename)
    return pid

def dict_combine_with_default(in_dict,default_dict):
    """A useful function for dealing with dictionary function input.  Combines
    parameters from in_dict with those from default_dict with the output
    having default_dict values for keys not present in in_dict"""
    out_dict=copy.deepcopy(in_dict)
    for key in default_dict.keys():
        out_dict[key]=copy.deepcopy(in_dict.get(key,default_dict[key]))
    return out_dict

def quickpickle(object,filename):
    """pickle.dump <object> to <filename>. If filename ends with .gz the pickle file
    is gzipped """
    if filename.endswith(".gz"):
        # Then compress the pickled file
        outfile=open(filename[:-3],'w')
    else:
        outfile=open(filename,'w')
    cPickle.dump(object,outfile)
    outfile.close()
    if filename.endswith(".gz"):
        # Then compress the pickled file
        os.system('gzip -f '+filename[:-3])

def quickunpickle(filename):
    """pickle.load an object from <filename> If an absolute path is not included
    in filename, then environment variable DATA_PATH is used.  Saves a few lines
    of code."""
    if filename.endswith('.gz'):
        # then we need to uncompress the pickle file
        infile=os.popen('gunzip -c '+filename)
    else:
        infile=open(filename)
    return cPickle.load(infile)

def arrayfromascii(filename,headlines=0):
    """Loads an ascii file containing numeric data, Returns a 2-D array of floats"""
    infile=open(filename)
    strlist=infile.readlines()
    header=strlist[:headlines]
    strlist=strlist[headlines:]
    nrows=len(strlist)
    ncols=len(strlist[0].split())
    print "Loading array with dimensions ["+str(nrows)+","+str(ncols)+"]"
    y=zeros([nrows,ncols], dtype="float32")
    for i in range(nrows):
        rowstrlist=strlist[i].split()
        for j in range(ncols):
            y[i,j]=float(rowstrlist[j].replace('D','E'))
    if headlines > 0:
        return y,header
    else:
        return y

def dict_print(indict):
    """prints a dictionary object in a nicer way than the default. ie one key
    per line"""
    outstr="{\n"
    for thiskey in indict.keys():
        outstr+=str(thiskey)+" : "+str(indict[thiskey])+"\n"
    outstr+="}"
    print outstr

def array2csv(data,filename):
    """Saves a 2D array in comma delimited format.  Useful for getting data
    into spreadsheet applications"""
    outfile=open(filename,'w')
    if data.ndims!=2:
        raise PyARTSError("array must be 2D")
    for i in range(data.shape[0]):
        for j in range(data.shape[1]-1):
            outfile.write(str(data[i,j])+',')
        outfile.write(str(data[i,data.shape[1]-1])+'\n')
    outfile.close()

def multi_thread(func,inarglist,num_proc,logging):
    """executes <func> for every argument list in inarglist and returns a
    list of output objects. num_proc specifies the
    desired number of concurrent processes (usually the number of CPUs)"""
    import cPickle
    import Queue
    import threading
    outlist=range(len(inarglist))
    #create job queue
    queue=Queue.Queue(num_proc)
    #define is_empty Event
    is_empty=threading.Event()
    #define function that does the work
    def gruntwork(i):
        try:
            outlist[i]= func(*inarglist[i])
        except TypeError:
            print "inarglist must be a list of lists"
        except:
            print 'Error executing ' + func.__name__
            print sys.exc_info()
        if logging:
            quickpickle(outlist,DATA_PATH+'/multi_thread_log.pickle')
        #make space in the queue
        queue.get()
        if queue.empty():
            is_empty.set()

    #Now execute the processes
    for i in range(len(inarglist)):
        thr=threading.Thread(target=gruntwork, args=(i,))
        is_empty.clear()
        queue.put(thr,1)
        thr.start()

    #while not queue.empty():
        #pass
    is_empty.wait()
    return outlist

def multi_thread2(func,inarglist,num_proc,logging):
    """executes <func> for every argument list in inarglist and returns a
    list of output objects. num_proc specifies the
    desired number of concurrent processes (usually the number of CPUs)"""

    import Queue
    import threading
    outlist=range(len(inarglist))
    #define is_empty Event
    is_empty=threading.Event()
    #define function that does the work
    class Worker(threading.Thread):
        def __init__(self,queue):
            self.__queue=queue
            threading.Thread.__init__(self)

        def run(self):
            while 1:
                i=self.__queue.get()
                if i is None:
                    break # reached end of queue
                outlist[i]=apply(func,inarglist[i])
                # GH 2011-03-04: disabled pickling, it doesn't work
##                if logging:
##                    try:
##                        quickpickle(outlist,DATA_PATH+'/multi_thread_log.pickle')
##                    except TypeError, m:
##                        print >>sys.stderr, "Unable to pickle: %s (%s)" % \
##                                (outlist, m.args)

    #create job queue
    queue=Queue.Queue(0)

    #Put jobs on the queue
    for i in range(len(inarglist)):
        queue.put(i)

    #Signal end of queue
    for i in range(num_proc):
        queue.put(None)

    workers=[]
    #Now start Workers
    for i in range(num_proc):
        workers.append(Worker(queue))
        workers[i].start()

    #Wait until the workers are finished
    for wrkr in workers:
        wrkr.join()
    return outlist

def array2arts(arr):
    """Converts a n-dimensional array to an ARTS-literal.

    For a 0-dimensional array, this is [arr].

    For a 1-dimensional array, this is [a, b, c].

    For a 2-dimensional array, this is [a, b, c; d, e, f].

    For higher dimensions, there is no way to specify an ARTS literal.
    Write the object to a file instead (this should also be done for large
    amounts of data).
    """

    arr = numpy.asarray(arr)
    if arr.ndim == 0:
        return "[%s]" % arr
    elif arr.ndim == 1:
        return \
            '[' + \
            ", ".join(str(s) for s in arr) + \
            ']'
    elif arr.ndim == 2:
        return \
            '[' + \
            '; '.join(", ".join(str(elem) for elem in row) for row in arr) + \
            ']'
    else:
        raise PyARTSError("Cannot represent %d-dim object as ARTS literal" % arr.ndim)

def merge_arrays((arr1, arr2)):
    """Merges two record-arrays.
    """
    t1 = arr1.dtype
    t2 = arr2.dtype
    newdtype = numpy.dtype(t1.descr + t2.descr)
    newarray = numpy.empty(shape=arr1.shape, dtype=newdtype)
    for field in t1.names:
        newarray[field] = arr1[field]
    for field in t2.names:
        newarray[field] = arr2[field]
    return newarray

def quotify(s):
    """Adds quotation marks around the string, if not present.

    This is necessary for string literals in ARTS.
    Does not take care of escaping anything.
    
    >>> print quotify("Hello, world!")
    "Hello, world!"
    """
    if s.startswith('"') and s.endswith('"'):
        return s
    else:
        return '"%s"' % s

##from IPython.Debugger import Tracer; debug_here = Tracer()
def arts_repr_list(L):
    """Arts-readable string-representation of list of strings
    """
    return "[" + ", ".join(quotify(s) for s in L) + "]"

##    debug_here()
    return [quotify(s) for s in L]

def get_fascod_path(mode, species):
    """Get fascod path for (mode, species).

    Mode can be tropical, subarctic-winter, etc.
    Species can be CH4, CO, etc.

    Does not verify the file exists.
    """
    if constants.fascod_dir is None:
        raise PyARTSError(\
            "Cannot find Fascod data; is environment ARTSXML_DATA_PATH " \
            "set?")
    base = os.path.join(constants.fascod_dir, "%s.%s.xml")
    return base % (mode, species)

def get_fascod_for_ir(mode, dataset):
    required = constants.sensor_species["hirs_reference"]
    provided = constants.dataset_species[dataset]
    d = {}
    for spec in required:
        if spec not in provided:
            d[spec] = get_fascod_path(mode, spec)
    return d
    
def get_fascod_for_chevalier_hirs(mode):
    """Fascod-paths needed for HIRS-simul with Chevalier.
    """
    return get_fascod_for_ir(mode, "cheval")
#    hirs_species = constants.sensor_species["hirs_reference"]
#    cheval_species = constants.dataset_species["cheval"]
#    d = {}
#    for spec in hirs_species:
#        if spec not in cheval_species:
#            d[spec] = get_fascod_path(mode, spec)
#    return d

def get_fascod_for_nicam_ir(mode):
    """Fascod-paths needed for IR-simul with Chevalier/Nicam.
    """

    return get_fascod_for_ir(mode, "nicam")
#    ir_species = constants.sensor_species["hirs_reference"]
#    nicam_species = constants.dataset_species["nicam"]

class Configurator(object):
    config = None

    def init(self):
        config = ConfigParser.RawConfigParser()
        config.read(os.path.expanduser('~/.PyARTSrc'))
        self.config = config
    
    def __call__(self, arg):
        if self.config is None:
            self.init()
        return self.config.get("main", arg)

get_config = Configurator()


# OrderedDict was added in 2.7, but is used by artsXML; repeated here
if sys.version_info < (2, 7, 0):
    # source: http://code.activestate.com/recipes/576693/
    from UserDict import DictMixin

    class OrderedDict(dict, DictMixin):

        def __init__(self, *args, **kwds):
            if len(args) > 1:
                raise TypeError('expected at most 1 arguments, got %d' % len(args))
            try:
                self.__end
            except AttributeError:
                self.clear()
            self.update(*args, **kwds)

        def clear(self):
            self.__end = end = []
            end += [None, end, end]         # sentinel node for doubly linked list
            self.__map = {}                 # key --> [key, prev, next]
            dict.clear(self)

        def __setitem__(self, key, value):
            if key not in self:
                end = self.__end
                curr = end[1]
                curr[2] = end[1] = self.__map[key] = [key, curr, end]
            dict.__setitem__(self, key, value)

        def __delitem__(self, key):
            dict.__delitem__(self, key)
            key, prev, next = self.__map.pop(key)
            prev[2] = next
            next[1] = prev

        def __iter__(self):
            end = self.__end
            curr = end[2]
            while curr is not end:
                yield curr[0]
                curr = curr[2]

        def __reversed__(self):
            end = self.__end
            curr = end[1]
            while curr is not end:
                yield curr[0]
                curr = curr[1]

        def popitem(self, last=True):
            if not self:
                raise KeyError('dictionary is empty')
            if last:
                key = reversed(self).next()
            else:
                key = iter(self).next()
            value = self.pop(key)
            return key, value

        def __reduce__(self):
            items = [[k, self[k]] for k in self]
            tmp = self.__map, self.__end
            del self.__map, self.__end
            inst_dict = vars(self).copy()
            self.__map, self.__end = tmp
            if inst_dict:
                return (self.__class__, (items,), inst_dict)
            return self.__class__, (items,)

        def keys(self):
            return list(self)

        setdefault = DictMixin.setdefault
        update = DictMixin.update
        pop = DictMixin.pop
        values = DictMixin.values
        items = DictMixin.items
        iterkeys = DictMixin.iterkeys
        itervalues = DictMixin.itervalues
        iteritems = DictMixin.iteritems

        def __repr__(self):
            if not self:
                return '%s()' % (self.__class__.__name__,)
            return '%s(%r)' % (self.__class__.__name__, self.items())

        def copy(self):
            return self.__class__(self)

        @classmethod
        def fromkeys(cls, iterable, value=None):
            d = cls()
            for key in iterable:
                d[key] = value
            return d

        def __eq__(self, other):
            if isinstance(other, OrderedDict):
                return len(self)==len(other) and self.items() == other.items()
            return dict.__eq__(self, other)

        def __ne__(self, other):
            return not self == other
else:
    from collections import OrderedDict


