#!/usr/bin/env python
#! -*- coding: utf-8 -*-

"""This test-file illustrates a small example using Daniels code
"""

import pprint
import os.path
import datetime
now = datetime.datetime.now

import numpy

import PyARTS.arts_file_components as afc

from PyARTS.arts_geometry import EARTH_RADIUS as R
from PyARTS.arts_types import AbsSpecies
from PyARTS.arts_math import integrate_quantity
from PyARTS.arts import ArtsRun
from PyARTS.io import read_chevalier, get_chevalier_file
from PyARTS.general import get_fascod_for_chevalier_hirs

settings_common = dict(
    y_unit = "PlanckBT",
    stokes_dim = 1,
    dim = 1,
    limb = False, # needed because sensor_pos/los not explicitly given
    batch_write = "/tmp/batch_output", # just to track progress :)
    surface_reflectivity = 0.3,
#    cheval_selection = high_IWP, # only those atmospheres
    )

settings_amsu = dict(
#        abs_species = AbsSpecies(H2O=["PWR98"], O3=None, O2=["PWR93"],
#                                 N2=["SelfContStandardType"],
#                                 order=["H2O", "O3", "N2", "O2"]),
    sensor = "amsub_fast",
    abs_from_sensor = True,
    scat_data_raw = "/storage3/user_data/gerrit/scattering_artsXML/hong/ssd_rosette.xml.gz",
    scat_data_meta_array = "/storage3/user_data/gerrit/scattering_artsXML/hong/smd_rosette.xml.gz",
    channels = 5,
    views = 45,
    )

settings_ir_mono = dict(
    abs_from_sensor = False,
    surface_emissivity = 0.75,
    abs_lines = "/storage3/data/catalogue/hitran/hitran2004/HITRAN04.par",
    abs_lines_format = "Hitran2004",
    f_grid = numpy.array([2.5e13]), # 2.5e13 Hz ~= 12 µm
    extra_fields = get_fascod_for_chevalier_hirs("tropical"),
    y_unit = "PlanckBT",
    scat_data_raw = "/storage3/user_data/gerrit/scattering_artsXML/yang_2005/ssd_SOL.xml.gz",
    scat_data_meta_array = "/storage3/user_data/gerrit/scattering_artsXML/yang_2005/smd_SOL.xml.gz",
)

settings_hirs = dict(
    sensor = "hirs_fast",
    abs_from_sensor = True,
    surface_emissivity = 0.75,
    satellite = "noaa19",
    abs_lines = "/storage3/data/catalogue/hitran/hitran2004/HITRAN04.par",
    abs_lines_format = "Hitran2004",
    channels = numpy.array([11])-1, # channel 11 (position 10)
    views = numpy.array([27]), # only nadir
    extra_fields = get_fascod_for_chevalier_hirs("tropical"),
    y_unit = "PlanckBT", # applied after yCalc
    scat_data_raw = "/storage3/user_data/gerrit/scattering_artsXML/yang_2005/ssd_SOL.xml.gz",
    scat_data_meta_array = "/storage3/user_data/gerrit/scattering_artsXML/yang_2005/smd_SOL.xml.gz",
    N_za_grid = 180,
    )

settings_cloudy = dict(
    cheval_file = get_chevalier_file("all", "ccol"), # Daniels file
    )

settings_clearsky = dict(
    cheval_file = get_chevalier_file("clear", "ccol"), # in arts-xml-data
    )

table = {
    "amsu": settings_amsu,
    "hirs": settings_hirs,
    "cloud": settings_cloudy,
    "clear": settings_clearsky,
    "ir_mono": settings_ir_mono,
}

def simulate(sensor, cloud, **more):
    """Simulates for amsu/hirs for cloud/clear
    """

    settings = settings_common.copy()
    settings.update(table[sensor])
    settings.update(table[cloud])
    settings.update(more)
    run = ArtsRun(params=settings, run_type=
            "batch_clear" if cloud=="clear" else "batch_doit")

    # prepare to reshape output according to actual dimensions
    n_chan = len(settings["channels"])
    n_pol = settings["stokes_dim"]
    n_pos = 1
    n_los = len(settings["views"])

    print now(), "Chevalier atmospheres", settings["cheval_selection"]
    print now(), "running", sensor, cloud, run.filename
    pprint.pprint(settings)
    run.run()
    print now(), "saving results"
    output = run["ybatch"].reshape(n_los, n_pos, n_pol, n_chan, -1)
    numpy.save("ybatch_%s_%s_result.npy" % (sensor, cloud), output)
    numpy.save("ybatch_%s_%s_raw_result.npy" % (sensor, cloud), run["ybatch"])
    return output

def main():
    # read Chevalier data
    print now(), "Reading Chevalier data"
    M = read_chevalier("ccol")
    IWP = integrate_quantity(M, "CIW")
    # find highest IWPs
    print now(), "Selecting Cheval profiles"
    #high_IWP = IWP.argsort()[-25:]
    high_IWP = IWP.argsort()[-5:]


    numpy.save("IWP.npy", IWP)

    #clear = simulate("hirs", "clear", cheval_selection=high_IWP)
    #cloud = simulate("hirs", "cloud", cheval_selection=high_IWP)
    cloud = simulate("hirs", "cloud", cheval_selection=numpy.array([0, 1000, 2000, 3000, 4000]))
    #cloud = simulate("ir_mono", "cloud", cheval_selection=numpy.array([0, 1000, 2000, 3000, 4000]))

    print "IWP, input", IWP
    print "ybatch, clear", clear
    print "ybatch, cloudy", cloud
    print "ybatch, difference", clear - cloud

if __name__ == "__main__":
    main()
