"""
Regression tests and unittests
"""

import sys, os, re, unittest

import numpy

import scipy.integrate
import PyARTS.io

# this part based on
# This program is part of "Dive Into Python", a free Python book for
# experienced programmers.  Visit http://diveintopython.org/ for the
# latest version.

def regressionTest():
    path = os.path.abspath(os.path.dirname(sys.argv[0]))
    files = os.listdir(path)
    test = re.compile("test\.py$", re.IGNORECASE)
    files = filter(test.search, files)
    filenameToModuleName = lambda f: os.path.splitext(f)[0]
    moduleNames = map(filenameToModuleName, files)
    modules = map(__import__, moduleNames)
    load = unittest.defaultTestLoader.loadTestsFromModule
    return unittest.TestSuite(map(load, modules))

class TestReadHongYang(unittest.TestCase):
    """Test reading routines for Hong, Yang data.
    """

    yang2005 = ["/storage3/data/scattering_databases/yang/Yang_2005_IR"
                "/%s.out" % s for s in
                    ("AGG", "BUL", "DRO", "HOL", "PLA", "SOL")]
    hong = ["/storage3/data/scattering_databases/hong/single"
            "/%s.single_mm_21" % s for s in
                ("aggregate", "column", "droxtal",
                 "hollow", "plate", "rosette")]
    def setUp(self):
        for f in self.yang2005 + self.hong:
            if not os.path.exists(f):
                print "Skipping TestReadHongYang"
                self.dotest = False
                break
        else:
            self.dotest = True

    def test_integration(self):
        if not self.dotest:
            return
        for y in self.yang2005+self.hong:
            if y in self.yang2005:
                (size, wavelength, angles), SSD = PyARTS.io.readYang2005(y)
                integrand = (2 * numpy.pi * numpy.sin(numpy.deg2rad(angles)) 
                               * SSD["P11"][..., :])
            elif y in self.hong:
                (size, wavelength, angles), SSD = PyARTS.io.readHong(y)
                integrand = (2 * numpy.pi * numpy.sin(numpy.deg2rad(angles)) 
                               * SSD["Z"][..., 0, :])
            integrated = scipy.integrate.trapz(integrand, numpy.deg2rad(angles))
            ratio = integrated/(SSD["K"] - SSD["A"])
            maxratio = numpy.nanmax(abs(ratio-1))
            self.assertAlmostEqual(maxratio, 0.0, 1,
                "Integration gave wrong results for %s, ratio up to %f" %
                    (y, maxratio))

if __name__ == "__main__":
    unittest.main()
    unittest.main(defaultTest="regressionTest")

