% CREATE_TMPFOLDER   Creation of a temporary folder in the work area.
%
%    The function creates a folder in the folder set as work area in
%    *atmlab*. See CONFIGURE for information about *atmlab*. A work area 
%    is a required setting.
%
%    The folder name is generated by matlab's tempname function and a
%    -atmlab is prepended to it.
%
%    
%
% FORMAT   tmpfolder = create_tmpfolder
%
% OUT      tmpfolder   Full path to created folder.
% ATMLAB   WORK_AREA   The work area is required as a personal setting.
%
% See also: onCleanup (e.g. c = onCleanup(@()delete(tmpfolder))

% 2011-04-04  Updated by Oliver Lemke. Use matlab's tempname to generate
%             folder name. Append -atmlab to tmpfolder name.
% 2002-12-20  Created by Patrick Eriksson, based on older version
%             in AMI (part of arts-1).


function tmpfolder = create_tmpfolder

%=== Require that a work area is set as a personal setting
%
atmlab( 'require', {'WORK_AREA'} );
workarea = atmlab( 'WORK_AREA' );
%
if ~ischar(workarea)
    error( 'atmlab:create_tmpfolder', 'WORKAREA must be a string' );
elseif length(workarea) == 1 && strcmp(workarea,'.')
    workarea = pwd();
elseif length(workarea) > 1 && strcmp(workarea(1:2),'./')
    workarea = atmlab('WORK_AREA',[pwd(),workarea(2:end)]);
elseif length(workarea) > 1 && strcmp(workarea(1:2),'~/')
    workarea = atmlab('WORK_AREA',[getenv('HOME'),workarea(2:end)]);
elseif ~strcmp(workarea(1), '/')
    error( 'atmlab:create_tmpfolder', 'WORKAREA must be an absolute path' );
end

if ~isdir(workarea)
    error( 'atmlab:create_tmpfolder', ...
           'The given work area (%s) does not exist.', workarea );
end

%=== Loop until a new folder with unique name has been created
%
ready = 0;
count = 100;
%
username = getenv('USER');
if (~isempty(username)), username = [username '-']; end
tmpfolderparts = regexp(tempname(workarea), ...
    ['(?<dirname>.*' filesep ')(?<basename>[^' filesep ']*)$'], ...
    'names');
tmpfolder = [tmpfolderparts.dirname 'atmlab-' username tmpfolderparts.basename];

% Has happened that whitespace has entered, at least from username
tmpfolder = tmpfolder( ~isspace(tmpfolder) );

while count && ~ready
  if ~exist( tmpfolder, 'dir' )
    [a,b] = mkdir(tmpfolder);      %#ok<ASGLU>
    if isempty(b)           % Check again if folder exist, for extra safety
      ready = 1;
    end
  end
  count = count - 1;
end

if ~ready
    error( 'atmlab:tmpfolder', ...
           'Unable to create temporary folder in %s', workarea );
end
