/**
 * @file utility.h
 * @author spfrnd
 * @date 2016-02-25
 * @brief Utility functions for testing.
 *
 * This file contains utility functions for testing, such as computing the
 * maximum relative error of a matrix or a vector or the generation of
 * random vectors and matrices.
 *
 */

#ifndef TEST_UTILITY_H
#define TEST_UTILITY_H

#include <random>
#include <cmath>
#include <iostream>
#include "invlib/algebra.h"

namespace invlib
{

/**
 * \brief Set matrix to identity matrix.
 *
 * Function template to set a function to the matrix identity. Assumes that
 * elements can be accessed using <tt>operator(unsigned int, unsigend int)</tt>
 * with indexing starting at zero and that the number of rows and colums of the
 * matrix are returned by the member functions <tt>rows()</tt> and <tt>cols()</tt>,
 * respectively. The given matrix must be square.
 *
 * \tparam MatrixType The type of the matrix to set to the identity matrix.
 */
template
<
typename MatrixType
>
void set_identity(MatrixType &A)
{
    assert( A.cols() == A.rows() );

    for (unsigned int i = 0; i < A.rows(); i++ )
    {
        for (unsigned int j = 0; j < A.cols(); j++ )
        {
            A(i,j) = 0.0;
        }
        A(i,i) = 1.0;
    }
}

/**
 * \brief Generate random vector.
 *
 * Generate a random of length @n with uniformly distributed elements in the
 * range [-1.0,1.0]. Assumes that the provided vector type can be resized
 * using the memeber function <tt>resize(int)</tt>.
 *
 * \tparam Vector The vector type of the vector to generate.
 */
template
<
typename Vector
>
Vector random(unsigned int n)
{
    Vector v; v.resize(n);
    std::random_device rd;
    std::mt19937 gen(rd());
    std::uniform_real_distribution<> dis(-10,10);

    for (unsigned int i = 0; i < n; i++)
    {
        v(i) = dis(gen);
    }

    return v;
}

/**
 * \brief Generate random vector with given norm.
 *
 * Generate a random vector of dimension @n and norm l with uniformly
 * distributed elements in the  range [-1.0,1.0]. Assumes that the
 * provided vector type can be resized using the memeber function
 * <tt>resize(int)</tt> and that the vector can be scaled using
 * the member function <tt>operator*=</tt>.
 *
 * \tparam Vector The vector type of the vector to generate.
 */
template
<
typename Vector
>
Vector random_normalized(unsigned int n, double l)
{
    Vector v = random<Vector>(n);
    v *= l / std::sqrt(dot(v,v));
    return v;
}

/**
 * \brief Generate random matrix.
 *
 * Generate a random @m times @n matrix uniformly distributed elements in the
 * range [-1.0,1.0]. Assumes that the provided matrix type can be resized
 * using the memeber function <tt>resize(unsigned int, unsigned int)</tt> and
 * that elements can be accessed using
 * <tt>operator()(unsigned int, unsigned int)</tt>.
 *
 * \tparam Matrx The type of the matrix to generate.
 */
template
<
typename MatrixType
>
MatrixType random(unsigned int m, unsigned int n)
{
    MatrixType M; M.resize(m, n);
    std::random_device rd;
    std::mt19937 gen(rd());
    std::uniform_real_distribution<> dis(-1,1);

    for (unsigned int i = 0; i < m; i++)
    {
        for (unsigned int j = 0; j < n; j++)
        {
            M(i,j) = dis(gen);
        }
    }

    return M;
}

/**
 * \brief Generate random positive definite matrix.
 *
 * Generate a random @n times @n positive definite matrix. The matrix is
 * generated by generating a random matrix \f$M\f$ using random(n,n) and then
 * computing \f$M'=M^TM + I\f$.
 *
 * \tparam MatrixType The type of the matrix to generate.
 */
template
<
typename MatrixType
>
MatrixType random_positive_definite(unsigned int m)
{
    MatrixIdentity<MatrixType> I{};
    MatrixType M = random<MatrixType>(m, m);
    return transp(M)*M + I;
}

/**
 * \brief Element-wise maximum error.
 *
 * Given two objects @A and @B of a type that provides an element iterator, the
 * function computes the element-wise maximum relative error of A w.r.t B given
 * the the current element of B is non-zero. Otherwise the absolute error for
 * this element is used.
 *
 * \tparam T The iterable type.
 */
template
<
typename T
>
typename T::RealType maximum_error( T &A,
                                    T &B )
{
    using std::max;
    using std::abs;

    using RealType = typename T::RealType;
    RealType max_err = 0.0;

    for( auto&& a = A.begin(), b = B.begin();
         a != A.end(), b != B.end();
         ++a, ++b )
    {
        RealType err = abs(*a - *b);
        err = (*b == 0.0) ? err : err / abs(*b);
        max_err = max(err, max_err);
    }
    return max_err;
}

/**
 * \brief Fill matrix or vector.
 *
 * Given an objects @A a type that provides an element iterator, the
 * function fills the object with the given object of the associated RealType.
 *
 * \tparam T The iterable type.
 * \tparam c The value to fill the object with.
 */
template
<
typename T
>
void fill(T &A, typename T::RealType c)
{
    for(auto a = A.begin(); a != A.end(); ++a)
    {
        *a = c;
    }
}

}

#endif // TEST_UTILITY_H
