"""

PyARTS: an ARTS related Python package
======================================

Introduction
------------

PyARTS is a python package, which has been developed to complement the
Atmospheric Radiative Transfer System - ARTS
(http://www.sat.ltu.se/arts/). Although ARTS is very flexible
software, it's primary function currently is to perform radiative
transfer simulations for a given atmospheric state.  PyARTS simplifies
the process of creating these atmospheric scenarios, and also provides
a front-end to the ARTS software for convenient configuration and
execution of ARTS radiative transfer calculations.

PyARTS includes two high-level modules that provide most of the
functionality needed for the preparation and execution of ARTS
simulations:

clouds
  produces
  arbitrarily complex multi-phase multi-habit cloud fields for arts
  simulations. This includes the generation of single scattering
  properties of non-spherical ice particles and the generation of
  particle number density fields for given ice and liquid water
  content fields. *clouds* also provides convenience functions for producing
  simple 1D and 3D box cloud scenarios.

  The module provides a high-level Cloud class and several lower level
  objects such as Crystal or Droplet. They can generate both single
  scattering properties and particle number density fields. As of
  arts-1-14-139, arts can calculate particle number densities internally
  and there is less need to do so in PyARTS.

arts
  contains classes and functions that
  actually perform ARTS simulations.  The ArtsRun class provides general
  functionality for configuring, performing, and managing the output
  of ARTS simulations.

There are several lower-level modules that, as well as serving the
arts and cloud modules, are also useful in their own right:

arts_file_components
  Contains an ArtsCodeGenerator class that provides low- and mid-level
  access to generate ARTS control-files.

arts_geometry
  Various functions dealing with 3D spherical geometry.

arts_math
  provides several interpolation, quadrature, and
  grid creation functions.

arts_scat
  provides functions and classes for the calculation of
  single scattering properties of ice and liquid water hydrometeors.

arts_types
  provides support for the manipulation, loading, and
  saving in ARTS XML format of some ARTS classes, e.g,
  ArrayOfGriddedField3, GriddedField3, and also the generation of gaseous
  absorption lookup tables

artsXML
  provides general XML input and output that can be used
  for all ARTS objects.

general
  a general purpose module that includes simplified
  pickling/unpickling functions for saving arbitrarily complex python
  objects, and functions for performing multi-threaded calculations.

physics
  Some physical constants, Rayleigh Jeans brightness temperature function,
  and psychrometric functions.

plotting
  general purpose plotting functions and functions for plotting
  ARTS related quantities (requires matplotlib)

sli
  contains the SLIData2 class which generates almost optimal grids
  for 2D sequential linear interpolation.  SLI can be used by the ARTS-MC
  algorithm for the rapid calculation of incoming radiation at the
  cloudbox boundary.

io
  Contains functions to locate and read various kinds of data associated
  with PyARTS or ARTS, such as the Chevalier data or scattering databases.


An example
----------

Here is a simple example python session that demonstrates what
can be done with the PyARTS package.  In this case we perform 3D
polarized radiative transfer in an atmosphere containing a uniform box
shaped cloud.

FIXME: the example below needs updating!

1. First import the most commonly used PyARTS modules.

>>> from PyARTS import *

2. Start by defining a simple box shaped cloud filled with horizontally 
   aligned oblate spheroids.

>>> a_cloud=clouds.boxcloud(ztopkm=14.0,zbottomkm=13.0,lat1=-2.0,lat2=2.0,
... lon1=-2.0,lon2=2.0,cb_size={'np':5,'nlat':5,'nlon':5},
... zfile='PyARTS/data/tropical.z.xml',tfile='PyARTS/data/tropical.t.xml',
... IWC=0.1)
>>> horizontal_plate=clouds.Crystal(ptype=30,NP=-1,aspect_ratio=2.0)
>>> a_cloud.addHydrometeor(horizontal_plate)
<Cloud, hydrometeors=1>

3. Generate single scattering data files, and particle number density 
   fields.

>>> a_cloud.scat_file_gen(f_grid=[500e9,503e9],num_proc=2)
<Cloud, hydrometeors=1>
>>> a_cloud.pnd_field_gen('pnd_field.xml')
<Cloud, hydrometeors=1>

4. Generate grids for ARTS RT simulation.  For the pressure grid, latitude
   grid and longitude grid, a fine grid spanning the cloudbox is merged with a
   course grid covering the modelled atmosphere.

>>> p_grid=arts_math.gridmerge(arts_math.nlogspace(101325.0,0.1,100),
... a_cloud.p_grid[1:-2])
>>> artsXML.save(p_grid,'p_grid.xml')
>>> lat_grid=arts_math.gridmerge(arts_math.nlinspace(-16.0,16.0,100),
... a_cloud.lat_grid[1:-2])
>>> artsXML.save(lat_grid,'lat_grid.xml')
>>> lon_grid=lat_grid
>>> artsXML.save(lon_grid,'lon_grid.xml')

5. Now define parameters for ARTS run, giving the Monte Carlo algorithm
   a maximum execution time of 10 seconds (you can also specify a desired
   accuracy or a fixed number of iterations)

>>> arts_params={
...     "atm_basename":"PyARTS/data/tropical",
...     "cloud_box":a_cloud.cloudbox,
...     "freq":501.18e9,
...     "abs_species":["ClO","O3","H2O,H2O-MPM89","N2-SelfContStandardType"],
...     "abs_lookup":"PyARTS/data/gas_abs_lookup.xml",
...     "lat_grid":"lat_grid.xml",
...     "lon_grid":"lon_grid.xml",
...     "max_time":10,
...     "p_grid":"p_grid.xml",
...     "pnd_field_raw":a_cloud.pnd_file,
...     "rte_pos":{'r_or_z':95000.1,'lat':9.1,'lon':0},
...     "rte_los":{'za':99.14,'aa':180},
...     "scat_data_file":a_cloud.scat_files,
...     "mc_seed": 42,
...     "stokes_dim":4
...     }

6. And perform RT calculations (using 2 processors)...

>>> my_run=arts.ArtsRun(arts_params,'mcgeneral','cfile.arts')
>>> my_run.run_parallel(2)
<ArtsRun cfile.arts>

7. And here is the simulated Stokes vector...

>>> print my_run.output['y']
[  1.17613500e+02   5.57757000e+00  -7.19482500e-02  -2.69899500e-01]
>>> print my_run.output['mc_error']
[ 1.66664512  1.23597316  0.4771547   0.43988175]

"""

#the following modules are imported with "from PyARTS import *"

__all__=['arts_scat','arts','artsXML','arts_math','arts_types',
         'clouds','general','physics','plotting','sli',
         'arts_geometry','training_data_generator']
