"""PyARTS is designed as a wrapper for ARTS - ie an alternative to
writing a control file for each arts run.  The module assumes that arts
is in the default path.  This can be overridden by setting the environment
variable ARTS_PATH (e.g. /home/user/test/myarts, where myarts is the name
of the executable).

The most useful class is ArtsRun. 
"""
import os
#from os import *
import scipy
#from scipy import *
from physics import c,k
import general
#from general import *
import artsXML
import arts_file_components
#from arts_file_components import *
import time
import tempfile
from sli import SLIData2
from warnings import warn
import sys
import traceback
import subprocess
import numpy
from .constants import EARTH_RADIUS

# Base directory for arts installation
if (os.getenv('ARTS_PATH')):
    ARTS_EXEC=os.getenv('ARTS_PATH')
    arts_in_default_path = False
else:
    ARTS_EXEC='arts'
    arts_in_default_path = True

ARTS_FLAGS = "-r 000"

required_arts_version = (1, 14, 161)

def get_arts_version():
    cmd = ARTS_EXEC + ' -v'
    p = subprocess.Popen(cmd, shell=True, stdout=subprocess.PIPE,
                         stderr=subprocess.STDOUT)
    output = p.stdout.read()
    if p.wait() != 0:
        raise ARTSRunError("Arts failed while quering version. "\
                           "Output was:\n" + output)
    v_line = output.splitlines()[0]
    return tuple(int(i) for i in v_line.split()[0][5:].strip().split('.'))

class ARTSRunError(general.PyARTSError):
    pass

class ARTSPathError(general.PyARTSError):
    pass

class ARTSVersionError(general.PyARTSError):
    pass


#check if arts is around
p = subprocess.Popen(ARTS_EXEC+' -v', shell=True, 
          stdin=subprocess.PIPE,
          stdout=subprocess.PIPE,
          stderr=subprocess.PIPE, close_fds=True)

errstr=p.stderr.read()
stdoutstr=p.stdout.read()
if len(errstr)>0:
    if arts_in_default_path:
        errorstr="ARTS executable not found in default path. "\
                "Please set the ARTS_PATH environment variable."
    else:
        errorstr="ARTS executable ("+ARTS_EXEC+") not found. "\
                "Please correct the ARTS_PATH environment variable."

    raise ARTSPathError, errorstr
#and if it is the right version
else:
    arts_version=get_arts_version()
    errorstr='Required arts version is '+str(required_arts_version[0])+'.'+\
            str(required_arts_version[1])+'.'+str(required_arts_version[2])+'\n'+\
            'Installed arts version is '+str(arts_version[0])+'.'+\
            str(arts_version[1])+'.'+str(arts_version[2])+'\n'
    if arts_version < required_arts_version:
        raise ARTSVersionError, errorstr

class ArtsRun(object):
    """A class representing a single arts simulation.
    
    Initialisation
    ~~~~~~~~~~~~~~

    params: mapping
      Full parameters. For documentation, see
      arts_file_components.ArtsCodeGenerator.

    run_type: string-like
      Determines the type of run. ArtsRun will initiate an
      ArtsCodeGenerator by calling the classmethod constructor
      ArtsCodeGenerator.do_run_type. For example, if run_type
      is doit_batch, ArtsRun will initiate
      ArtsCodeGenerator.do_doit_batch. Any do_x method in
      arts_file_components.ArtsCodeGenerator, e.g.  clear, montecarlo,
      doit_batch, will work here. Different 

    filename: string (optional)
      The name of the generated arts control file.
      If not given, a random, temporary file is used.

    catch_stdout: boolean (optional, defaults to True)
      If set to False, stdout is copied to the screen.

    Some examples for using this class are given in the docstring for
    the PyARTS package (__init__.py)

    Attributes
    ~~~~~~~~~~

        params: mapping
            Stores parameters passed to constructor.

        run_type: string
            Stores run_type passed to constructor.

        filename: string
            Stores the filename used .

        code: ArtsCodeGenerator object
            Stores the instance of ArtsCodeGenerator, after executing
            .file_gen().

        proc: subprocess.Popen object
            Stores the pipe to the arts process, after running .start().

        output: dictionary
            Dictionary storing all variables output by ARTS. Available
            after running .process_output().

        time_elapsed: float
            Stores run-time (user space) in seconds.
    """
    def __init__(self, params, run_type, filename=None,
                 catch_stdout=True, flags=ARTS_FLAGS):
        if filename is None:
            filename = tempfile.mktemp(".arts")
        self.params = params
        self.run_type = run_type
        self.filename = filename
        self.output = {}
        self.catch_stdout = catch_stdout
        self.flags = flags

    def file_gen(self):
        """Generates the arts control file
        """
        method = getattr(arts_file_components.ArtsCodeGenerator,
                         "do_%s" % self.run_type)

        codegen = method(filename=self.filename, **self.params)
        self.code = codegen

    def start(self):
        """Start the arts run.
        
        Sets self.proc to the subprocess.Popen object to handle output
        etc.

        """
        #just to be sure. Make sure control file is up to date
        self.file_gen()
#        self.out_stream = popen("nice "+ARTS_EXEC+" "+\
#                                self.filename+" | tee " \
#                                + self.filename + ".out")
        cmd = "nice " + ARTS_EXEC + " " + self.flags + " " + self.filename
        if self.catch_stdout:
            proc = subprocess.Popen(cmd,
                        stdout=subprocess.PIPE, 
                        stderr=subprocess.PIPE,
                        shell=True)
        else:
            print 'executing', cmd
            rv = os.system(cmd)
            if rv != 0:
                raise ARTSRunError("ARTS run failed (error code: %d). Hopefully, "
                                   "the output above gives a clue." % rv)
            proc = None
        self.proc = proc

    def clean(self):
        """Removes any files created by the ArtsRun instance
        """

        base = self.filename[:-4]
        d = os.path.dirname(self.filename)
        for f in os.listdir(d):
            if f.startswith(base):
                os.remove(os.path.join(d, f))

    def process_output(self):
        """Wait for ARTS to complete (if run with pipe) and process output (stdout, stderr)
        
        For all variables output by the controlfile (available in
        self.code.out), retrieve the value and put them in a hash
        self.output.
        """
        if self.proc is not None:
            self.std_out = self.proc.stdout.read()
            std_err = self.proc.stderr.read()
            retval = self.proc.wait()
            if retval != 0:
                raise ARTSRunError("ARTS error running %s, error code: %d. " \
                                   "Full ARTS stderr:\n%s" % \
                                    (self.filename, retval, std_err))
        for varname in self.code.out:
            outbase = "%s.%s.xml" % (self.filename[:-5], varname)
            if not os.path.exists(outbase):
                outbase += ".gz"
            self.output[varname]=artsXML.load(outbase)

    def refine(self,max_time=-1,max_iter=-1,combined_error=-1):
        """Improve the accuracy an already completed ArtsRun (montecarlo only)
        by adding more photons. As in the initial parameters, a combination
        of max_time, max_iter, and combined_error can be specified, with
        the calcualtion termining when the first termination criteria is met.
        max_time and max_iter refer to the additional calculation, while
        combined_error is the desired standard error for the whole simulation"""
        if max_time<=0 and max_iter<=0 and combined_error<=0:
            raise InputError,"at least one of max_time, max_iter, combined_error must be positive"
        y1=self.y
        error1=self.error
        N1=self.iterations
        if combined_error>0:
            N2=N1*((error1/combined_error)**2-1)
            if max_iter>0:
                max_iter=numpy.minimum(max_iter,N2)
            else:
                max_iter=N2
        self.params.update({"std_err":-1,"max_time":max_time,"max_iter":max_iter})
        self.run()
        N2=self.iterations
        y2=self.y
        error2=self.error
        self.y=(N1*y1+N2*y2)/(N1+N2)
        self.error=(N1**2*error1**2+N2**2*error2**2)**0.5/(N1+N2)
        self.iterations=N1+N2

    def run(self):
        """Run ARTS according to the settings when instantiating ArtsRun.

        This method calls self.start() and self.process_output(). The
        total runtime is stored in self.time_elapsed. Returns self.
        """

        start_time = time.time()
        self.start()
        self.process_output()
        end_time = time.time()
        self.time_elapsed = end_time-start_time
        return self

    def run_in_xterm(self):
        """Runs arts in an xterm with no output processing"""
        self.file_gen()
        print "arts is running in an xterm, to return to python interpreter\n" + \
              "destroy xterm."
        proc = subprocess.Popen("xterm -exec "+ARTS_EXEC+" "+self.filename+\
                  " | tee " + self.filename + ".out  -hold", shell=True)

    def print_results(self):
        """Prints results of arts simulation"""
        print "Run type:"
        print self.run_type
        for varname in arts_output_vars[self.run_type]:
            print arts_output_var_desc[varname]+' ('+varname+'): '+str(self.output[varname])
        print "Time elapsed (s):"
        print self.time_elapsed

    def time_estimate(self):
        """Gives an estimate of run time. It only makes sense to use this for
        run_type = "montecarlo". An estimate of the expected error in the
        final calculation is also given"""
        import copy
        dummy_argdict=copy.deepcopy(self.params)
        dummy_argdict["maxiter"]=1000
        dummy_arts_run=ArtsRun(dummy_argdict,self.run_type)
        start_time=time.time()
        dummy_arts_run.run()
        end_time=time.time()
        maxiter=float(self.params["maxiter"])
        time_estimate=(end_time-start_time) * maxiter/1000.0
        print "Single processor time for current arguments (in s): "
        print time_estimate
        print "Error estimate for proposed calculation (K):"
        print dummy_arts_run.error/sqrt(maxiter/1000.0)

    def run_parallel(self,number_of_processes):
        """Only for monte carlo simulations.  Divides self.params["max_iter"]
        by number_of_processes and runs number_of_processes arts processes.  The
        results are then combined.  This is particularly worthwhile on multiple
        processor machines."""
        warn("ARTS is multi-threaded, do not run more than one ARTS.", DeprecationWarning)
        stokes_dim=self.params["stokes_dim"]

        self.output['y']=numpy.zeros(stokes_dim, dtype='float32')
        if self.run_type=='montecarlo':
            self.output['iy']=numpy.zeros(stokes_dim, dtype='float32')
        errorsquared = numpy.zeros(stokes_dim, dtype='float32')
        #create new instances with divided maxiter
        import copy

        start_time=time.time()

        dummy_argdict=copy.deepcopy(self.params)
        if self.params.has_key("max_iter"):
            if self.params["max_iter"]>0:
                dummy_argdict["max_iter"]=self.params["max_iter"]/number_of_processes
        self.run_list=[]
        for i in range(number_of_processes):
            self.run_list.append(ArtsRun(dummy_argdict,self.run_type,
                                         self.filename))
            self.run_list[i].start()
            time.sleep(3)
        for i in range(number_of_processes):
            self.run_list[i].process_output()
            errorsquared+=self.run_list[i].output['mc_error']**2
            self.output['y']+=self.run_list[i].output['y']
            if self.run_type=='montecarlo':
                self.output['iy']+=self.run_list[i].output['iy']
        self.output['mc_error']=errorsquared**0.5/number_of_processes
        self.output['y']/=number_of_processes
        if self.run_type=='montecarlo':
            self.output['iy']/=number_of_processes

        end_time=time.time()
        self.time_elapsed=end_time-start_time
        return self

    def __getitem__(self, item):
        return self.output[item]

    def __repr__(self):
        return "<ArtsRun %s>" % self.filename


def artsGetAtmFields(argdict):
    """Interpolate atmospheric fields on p_grid

    Usage:
    t_field,z_field,vmr_field=artsGetAtmFields(argdict)

    ARTS is used to interpolate 1D t, z, and vmr fields on a given p_grid.
    The outputs are 3D arrays.  Only 1D fields are supported.
    
    argdict is a dictionary of arguments required by
    arts_file_components.ArtsCodeGenerator, eg
    'atm_basename','gas_species','p_grid'. Currently p_grid must be specified
    either as a filename, or a dictionary with 'start'','stop', and 'n' arguments
    (used for nlogspace in arts).  See examples/get_atm_fields.py
    """ 
    code = arts_file_components.ArtsCodeGenerator(generic=False, **argdict)
    code.add(arts_file_components.WSM('output_file_formatSetAscii'))
    code.add(arts_file_components.WSM("AtmosphereSet1D"))
    code.setup_gas_abs()
    code.setup_grids()
    code.setup_single_gas_species()
    #code.atmosphere1D_settings()
    #code.add(arts_file_components.WSM("AtmFieldsCalc"))
    code.add(arts_file_components.WriteXML("t_field"))
    code.add(arts_file_components.WriteXML("z_field"))
    code.add(arts_file_components.WriteXML("vmr_field"))
    basename=tempfile.mkstemp()[1]
    code.toControlFile(basename + ".arts")
    proc = subprocess.Popen(ARTS_EXEC + ' ' + basename + ".arts",
                stdout=subprocess.PIPE,
                stderr=subprocess.PIPE,
                stdin=subprocess.PIPE, shell=True)
    errstr = proc.stderr.read()
    if errstr:
        raise ARTSRunError("Error running ARTS control file %s: %s" % \
                            (basename + ".arts", errstr))
    t_field=artsXML.load(basename+'.t_field.xml')
    z_field=artsXML.load(basename+'.z_field.xml')
    vmr_field=artsXML.load(basename+'.vmr_field.xml')
    #delete files
    # FIXME GH 2011-03-04: there appears to be no .rep
    for suffix in ['.arts','.t_field.xml','.z_field.xml','.vmr_field.xml','.rep']:
        try:
            os.remove(basename+suffix)
        except OSError:
            # it's okay
            pass
    return t_field,z_field,vmr_field

def create_incoming_lookup(arts_params,zkm0,za0):
    """Initialises an SLIData2 object suitable for the generation of
    the ARTS WSV mc_incoming.  See examples/mc_incoming_gen.py
    """

    #define function needed by SLIData2
    r0=zkm0*1e3+EARTH_RADIUS
    def getradiance(r,za):
        #create sensor_pos
        sensor_pos=r.reshape(len(r),1)
        sensor_los=za.reshape(len(za),1)
        spfname=tempfile.mktemp('sensor_pos.xml')
        slfname=tempfile.mktemp('sensor_los.xml')

        artsXML.save(sensor_pos,spfname)

        artsXML.save(sensor_los,slfname)
        arts_params.update({'sensor_pos':spfname,
                            'sensor_los':slfname})
        a_run=ArtsRun(arts_params,"clear")
        a_run.run()
        #convert back to radiance from RJBT
        return 2*arts_params['freq']**2*k*a_run.output['y']/c**2
    return SLIData2(getradiance,r0,za0)


def IWP_cloud_opt_pathCalc(settings):
    """Calculates FOV averaged ice water path and FOV averaged optical path.
    The standard error for each calculation (Monte Carlo for Gaussian antennas)
    Returns iwp,tau,iwp_error,tau_error
    """

    arts_run = ArtsRun(settings, run_type='iwp_tau').run()
    return arts_run.output['mc_IWP'],arts_run.output['mc_cloud_opt_path'],arts_run.output['mc_IWP_error'],arts_run.output['mc_cloud_opt_path_error']

def limb_spectrum(f_grid,params={}):
    """A function that plots a clear sky limb spectrum with frequencies
    given in array f_grid
    """

    radiance_vec=[]
    for freq in f_grid:
        params["freq"]=freq
        a_run=ArtsRun(params,"clear")
        a_run.run()
        print "f = "+str(freq)+", y = "+str(a_run.y)
        radiance_vec.append(a_run.y)
    y_data=numpy.array(radiance_vec)
    return y_data

def xml_ascii_to_binary(old_file,var_name,new_file):
    """Use ARTS to convert from xml to binary.
    """

    code = arts_file_components.ArtsCodeGenerator(generic=False)
    code.add(arts_file_components.WSM("output_file_formatSetBinary"))
    code.add(arts_file_components.ReadXML(var_name,old_file))
    code.add(arts_file_components.WriteXML(var_name,new_file))
    artsfile=tempfile.mktemp('.arts')
    code.toControlFile(artsfile)
    p = subprocess.Popen(ARTS_EXEC+" "+artsfile, shell=True,
          stdin=subprocess.PIPE,
          stdout=subprocess.PIPE,
          stderr=subprocess.PIPE, close_fds=True)

    errstr=p.stderr.read()
    if len(errstr)>0:
        raise ARTSRunError(errstr)

def xml_binary_to_ascii(old_file,var_name,new_file):
    """Use ARTS to convert from binary to xml.
    """

    code = arts_file_components.ArtsCodeGenerator(generic=False)
    code.add(arts_file_components.WSM("output_file_formatSetAscii"))
    code.add(arts_file_components.ReadXML(var_name,old_file))
    code.add(arts_file_components.WriteXML(var_name,new_file))
    artsfile=tempfile.mktemp('.arts')
    code.toControlFile(artsfile)
    p = subprocess.Popen(ARTS_EXEC + ' ' + artsfile, shell=True)
    errstr = p.stderr.read()
    if errstr:
        raise ARTSRunError("Error running ARTS: " + errstr)

def pnd_fieldCalc(pnd_field_raw_file,cloudbox,p_file,lat_file,lon_file,
                  pnd_field_file,cloudbox_limits_file=''):
    """Use ARTS to calculate pnd_field.
    
    Uses arts to calculate the pnd_field WSV, which is interpolated onto the
    arts atmospheric grids.
    """

    code = arts_file_components.ArtsCodeGenerator(generic=False)
    if pnd_field_file.endswith(".bin"):
        code.add(arts_file_components.WSM("output_file_formatSetBinary"))
    else:
        code.add(arts_file_components.WSM("output_file_formatSetAscii"))
    code.add(arts_file_components.WSM('AtmosphereSet3D'))
    code.add(arts_file_components.ReadXML('p_grid',p_file))
    code.add(arts_file_components.ReadXML('lat_grid',lat_file))
    code.add(arts_file_components.ReadXML('lon_grid',lon_file))
    code.add(arts_file_components.WSM("cloudboxSetManually", "cloudbox_on", "cloudbox_limits", "atmosphere_dim", "p_grid", "lat_grid", "lon_grid", 
                                      *[cloudbox[s] for s in ("p1", "p2", "lat1", "lat2", "lon1", "lon2")])) 
    code.add(arts_file_components.ReadXML('pnd_field_raw',pnd_field_raw_file))
    code.add(arts_file_components.WSM('pnd_fieldCalc'))
    code.add(arts_file_components.WriteXML('pnd_field',pnd_field_file))
    code.add(arts_file_components.WriteXML('cloudbox_limits',cloudbox_limits_file))
    artsfile=tempfile.mktemp('.arts')
    code.toControlFile(artsfile)
    p = subprocess.Popen(ARTS_EXEC + ' ' + artsfile, shell=True, stderr=subprocess.PIPE)
    errstr = p.stderr.read()
    if errstr:
        raise ARTSRunError("Error running ARTS: " + errstr)


def scat_data_monoCalc(scat_data_raw_file,freq,scat_data_mono_file):
    """Use ARTS to calculate *scat_data_mono*.
    
    Uses arts to calculate the scat_data_mono WSV, which is interpolated
    at frequency freq.
    """

    code = arts_file_components.ArtsCodeGenerator(generic=False)
    if scat_data_mono_file.endswith(".bin"):
        code.add(arts_file_components.WSM("output_file_formatSetBinary"))
    else:
        code.add(arts_file_components.WSM("output_file_formatSetAscii"))
    code.add(arts_file_components.VectorSet("f_grid",freq))
    code.add(arts_file_components.IndexSet("f_index",0))
    code.add(arts_file_components.ReadXML('scat_data_raw',scat_data_raw_file))
    code.add(arts_file_components.WSM('scat_data_monoCalc'))
    code.add(arts_file_components.WriteXML('scat_data_mono',scat_data_mono_file))

    artsfile=tempfile.mktemp('.arts')
    code.toControlFile(artsfile)
    p = subprocess.Popen(ARTS_EXEC + ' ' + artsfile, shell=True)
    errstr = p.stderr.read()
    if errstr:
        raise ARTSRunError("Error running ARTS: " + errstr)



def ppathCalc(argdict):
    """Use ARTS to calculate 3D propagation path.
    
    **WARNING: Currently not working!**

    Uses arts to calculate a 3D propagation path, with atmospheric field
    settings as described in argdict.  The returned object is a dictionary
    holding all the Ppath member data."""

    fname=tempfile.mktemp('.arts')
    arts_file_components.ArtsCodeGenerator.do_ppath(filename=fname, **argdict)
    os.system(ARTS_EXEC+' '+fname+' > /dev/null')
    return artsXML.load(fname[:-5]+'.ppath.xml')


