#!/usr/bin/python
# -*- coding: utf-8 -*-

"""Basic classes to generate ARTS-code.

This is a low-level module usually not used directly. It can be used to
generate any valid ARTS-code, but also contains some methods that shorten
setups of e.g. usual settings for atmospheres, sensors, etc.

Some examples:

(work in progress, to be filled in later)
"""

# $Id: arts_file_components.py 8010 2012-11-16 15:02:37Z mendrok $

# TODO:
#
# Cory Davis 2004-09-16:
# FIXME: Make defaults a file lookup, and phase out all the dict.gets, assuming
# that any merging with the default arguments is done at a higher level

import os
import functools
import contextlib
import time
import numbers
import pprint
import datetime
now = datetime.datetime.now
import cPickle

import types
import artsXML
import arts_math
from .constants import EARTH_RADIUS
from cStringIO import StringIO
from warnings import warn

import numpy

import datetime

from . import arts_geometry
from . import constants
from . import general
from . import io
from . import artsXML

from .arts_types import AbsSpecies, ArtsType, ArrayOf

from IPython.core.debugger import Tracer; debug_here = Tracer()

verbosity=0#Controls the verbosity of the module

## defaults={
##     "3Dfields":0,
##     "lmax":-1,
##     "r_geoid":-1,
##     "stokes_dim":4,
##     "z_surface":0
## }


from .general import quotify

class ArtsCode(object):
    """Base-class for anything representing ARTS code.

    Not meant to be used directly.
    >>> ArtsCode()
    Traceback (most recent call last):
        ...
    PyARTSError: Do not instantiate this class directly

    """

    def __init__(self):
        if self.__class__ is ArtsCode:
            raise general.PyARTSError("Do not instantiate this class directly")

    def textify(self):
        """Alias for str(obj)"""
        return str(self)

class WSM(ArtsCode):
    """Represents an ARTS workspace method.

    The first argument is the name of the workspace method.
    The rest of the arguments are passed on to the workspace method.

    >>> print WSM("Test", "this", quotify("method"), 42, [1, 2, 3])
    Test(this, "method", 42, [1, 2, 3])
    <BLANKLINE>

    """
    def __init__(self, name="", *args):
        self.name = name
        self.args = args

    def __str__(self):
        """returns the text for the arts file implementation of the method"""
        if len(self.args) == 0:
            return self.name + "\n"

        strlist = []
        for arg in self.args:
            if isinstance(arg, list) and all(isinstance(s, basestring) for s in arg):
                strlist.append(general.arts_repr_list(arg))
            else:
                strlist.append(str(arg))
        return "%s(%s)\n" % (self.name, ', '.join(strlist))

class AgendaSet(ArtsCode):
    """Generates an agenda from a sequence of WSMs.

    Example:
    
    >>> print AgendaSet("TestAgenda", [WSM("One"), WSM("Two"), WSM("Three", quotify("with"), quotify("arguments"))])
    AgendaSet(TestAgenda){
        One
        Two
        Three("with", "arguments")
    }
    <BLANKLINE>


    """

    def __init__(self, name, wsm_list=None):
        self.agenda = name
        if wsm_list is not None:
            self.wsm_list = wsm_list   # list of WSM objects
        else:
            self.wsm_list = []

    def add(self, wsm):
        if not isinstance(wsm, ArtsCode):
            raise general.PyARTSError("Not ArtsCode: %s" % wsm)
        self.wsm_list.append(wsm)

    def __str__(self):
        out = StringIO()
        out.write("AgendaSet(%s){\n" % self.agenda)
        for wsm in self.wsm_list:
            out.write("    " + str(wsm))
        out.write("}\n")
        return out.getvalue()

class Include(ArtsCode):
    """Include another ARTS-file.
    
    >>> print Include("general.arts")
    INCLUDE "general.arts"
    <BLANKLINE>

    """

    def __init__(self, name):
        self.name = name

    def __str__(self):
        return 'INCLUDE "%s"\n' % self.name

class Comment(ArtsCode):
    """Include a comment.
    
    >>> print Comment("Don't forget, this is automatically generated...")
    # Don't forget, this is automatically generated...
    <BLANKLINE>

    """

    def __init__(self, val):
        self.val = val

    def __str__(self):
        if len(self.val) > 0 and self.val[0].isalnum():
            return "# %s\n" % self.val
        elif self.val.strip().startswith("#"):
            return self.val + "\n"
        else:
            return "#%s\n" % self.val

#Repeatedly used WSMs are conveniently implemented as child classes
class NamedWSM(WSM):
    def __init__(self, *args):
        super(NamedWSM, self).__init__(self.__class__.__name__, *args)

class IndexSet(NamedWSM):
    """Base class for any WSM where the name is inferred from the class.

    >>> print IndexSet("Counter", 10)
    IndexSet(Counter, 10)
    <BLANKLINE>

    """
    pass

class NumericSet(NamedWSM):
    pass

class ArraySet(NamedWSM):
    """Base class for arrays of numbers
    """
    def __init__(self, key, val):
        val = general.array2arts(val)
        super(ArraySet, self).__init__(key, str(val))

class VectorSet(ArraySet):
    pass

class ArrayOfIndexSet(ArraySet):
    pass

class Ignore(NamedWSM):
    pass

class VectorNLinSpace(NamedWSM):
    pass

class VectorNLogSpace(NamedWSM):
    pass

class Copy(NamedWSM):
    pass

class Matrix1RowFromVector(NamedWSM):
    pass

class ReadXML(NamedWSM):
    def __init__(self, var, filename='""'):
        super(ReadXML, self).__init__(var, quotify(filename))

class WriteXML(NamedWSM):
    """Special treatment for WriteXML, because it has changed compared to ARTS-1.
    """

    def __init__(self, var, filename=None):
        if filename is None:
            super(WriteXML, self).__init__("output_file_format", var)
        else:
            super(WriteXML, self).__init__("output_file_format", var, quotify(filename))

class Print(NamedWSM):
    def __init__(self, var, level=1):
        super(Print, self).__init__(var, level)

def commenter(meth):
    """Decorator adding comment objects at start and end of running method

    """
    @functools.wraps(meth)
    def newmeth(self, *args, **kwargs):
        if self.docomment > 0:
            self.add(Comment("##### START %s" % meth.func_name))
        if meth.func_doc is not None and self.docomment > 1:
            self.add(Comment("### Documentation:"))
            for line in meth.func_doc.split('\n'):
                self.add(Comment("## %s" % line))
        rv = meth(self, *args, **kwargs)
        if self.docomment > 0:
            self.add(Comment("##### END %s" % meth.func_name))
        return rv
    return newmeth

def runonce(meth):
    """Decorator verifying method is ran only once.

    """

    @functools.wraps(meth)
    def newmeth(self, *args, **kwargs):
        if meth.func_name in self.didrun:
            raise general.PyARTSError("Already executed %s!" % meth.func_name)
        self.didrun.add(meth.func_name)
        return meth(self, *args, **kwargs)
    return newmeth

def verbose(meth):
    """Decorator turning verbosity to the highest for here
    """

    @functools.wraps(meth)
    def newmeth(self, *args, **kwargs):
        self.add(WSM("verbositySet", "verbosity", 3, 3, 3))
        rv = meth(self, *args, **kwargs)
        self.add(WSM("verbosityInit"))
    return newmeth

class ArtsCodeGenerator(object):
    """Container class for generating ARTS code.

    Most methods create one or more new WSM or AgendaSet objects and
    append those to the .commands member, and return whatever was newly
    added.
    
    Settings passed to constructor may be required and/or used by zero or
    more methods (for details, see the methods' docstrings). Some of those
    are:

        dim
        freq
        stokes_dim
        p_grid
        lat_grid
        lon_grid
            either of:
                - mappings with keys start, n, stop
                - strings to files containing grids
                - vectors/arrays
        z_field
        p_field
        vmr_field
        atm_basename
        z_surface
        r_geoid
        abs_lookup
        abs_species
            arts_types.AbsSpecies or convertable thereto
        scat_data_mono
        scat_data_file
        pnd_field_raw
        pnd_field
        cloudbox
        sensor_pos
            ...
        sensor_los
            ...
        antenna
        tan_z
        z_surface
        full_spectrum
            write full spectrum on each batchrun

        (list incomplete)

        """

    # for automatic generation: which one to use?
    grids = {
        "p_grid": VectorNLogSpace,
        "lat_grid": VectorNLinSpace,
        "lon_grid": VectorNLinSpace,
        }

    # in batch?
    current_batch_agenda = None

    opts = ("freq sensor antenna WGS84 dim freq stokes_dim p_grid lat_grid "
            "lon_grid z_field p_field vmr_field atm_basename z_surface r_geoid "
            "abs_lookup abs_species scat_data_mono scat_data_file pnd_field_raw "
            "pnd_field cloudbox sensor_pos sensor_los antenna tan_z z_surface "
            "full_spectrum aa_fwhm abs_from_sensor abs_lines "
            "abs_lines_format abs_lookup abs_species antenna atm_basename "
            "atm_fields_compact batch batch_selection batch_write channels "
            "cheval_file cheval_selection cloud cloud_box continua dim "
            "doit_limits extra_fields f_grid_spacing freq full_spectrum "
            "infrared lat_grid limb line_range lmax lon_grid make_lookup "
            "massdensity_threshhold max_iter max_time mc_max_iter "
            "mc_max_time mc_seed mc_std_err N_aa_grid nelem NumericSet "
            "N_za_grid part_species p_grid pnd_field pnd_field_raw "
            "ppath_lmax reference r_geoid rte_los rte_pos satellite "
            "scat_data_and_meta_files scat_data_file_list "
            "scat_data_meta_array scat_data_meta_extra scat_data_mono "
            "scat_data_raw scat_data_raw_extra sensor size_range std_err "
            "stokes_dim StringSet surface_reflectivities "
            "surface_reflectivity surface_scalar_reflectivity tan_z t_field "
            "VectorSet views vmr_field write_lookup y ybatch_index y_unit "
            "za_fwhm za_grid_opt_file z_field z_surface").split()


    def __init__(self, generic=True, **kwargs):
        self.docomment = False
        self.didrun = set()
        self.settings = kwargs
        for k in self.settings.keys():
            if k not in self.opts:
                raise TypeError("Unknown keyword argument: %s" % k)
        st = self.settings
        self.commands = []
        self.out = []
        self.write_vars = {}
##        for k, v in defaults.items():
##            if k not in self.settings:
##                self.settings[k] = v
        if generic:
            # do this always
            self.setup_general()

            self.ybatch_calc_agenda = AgendaSet("ybatch_calc_agenda")

            self.batchcase = bool(self.settings.get("batch"))
        else:
            self.batchcase = False

        # do some checks
        self.makes_sense()
        self.i = 0 # for tempvar counter

        if "limb" in self.settings:
            self.islimb = self.settings["limb"]
        elif ("sensor_pos" in st and "sensor_los" in st and
              not isinstance(st["sensor_pos"], basestring)):
            self.islimb = arts_geometry.islimb(st["sensor_pos"][0],
                                               st["sensor_los"][0])


    @property
    def tempvar(self):
        t = "temp%d" % self.i
        self.i += 1
        return t

    def makes_sense(self):
        """Checks that all options make sense

        Returns nothing, but may raise exception.
        May also convert some settings.
        """

        st = self.settings
        if "dim" in st and st["dim"] not in (1, 2, 3):
            raise general.PyARTSError("Invalid dim. Got: %s. " \
                                      "Expected: (1, 2, 3)" % self.settings[dim])

        if st.get("abs_lookup") and st.get("abs_lines"):
            raise general.PyARTSError("Both abs_lookup and abs_lines set. " \
                    "I don't know what to do.")
        if "abs_species" in st:
            if not isinstance(st["abs_species"], AbsSpecies):
                st["abs_species"] = AbsSpecies(st["abs_species"])

    def writevar(self, v):
        """Write var to file, not suitable as literal.

        Actual writing only done when control-file is written.
        """
        self.write_vars[v] = self.settings[v]

    def add(self, wsm):
        """Adds a single WSM or AgendaSet to the commandlist

        Added to Batch if we are working in a batch at the moment.
        """

        if not isinstance(wsm, ArtsCode):
            raise general.PyARTSError("Not ArtsCode: %s" % wsm)

        if self.current_batch_agenda is None:
            self.commands.append(wsm)
        else:
            self.current_batch_agenda.add(wsm)

    def addMulti(self, wsms):
        """Add many WSMs or AgendaSets to the commandlist
        """

        for wsm in wsms:
            self.add(wsm)
    
    def toString(self):
        s = StringIO()
        self.toFile(s)
        return s.getvalue()

    def toFile(self, out):
        """Writes ARTS2-controlfile
        """
##        out.write("#DEFINITIONS:  -*-sh-*-\n")
        out.write("# This file was generated by PyARTS at %s\n" % \
                    datetime.datetime.now())
        if self.docomment > 1:
            out.write("# Settings used for generation:\n")
            out.writelines("#" + line + "\n" for line in pprint.pformat(self.settings).split("\n"))
        out.write("Arts2{\n")
        for command in self.commands:
            out.write(str(command))
        out.write("}")
        out.flush()

    def toControlFile(self, filename, settings_file=None):
        """Writes list of WSM and AgendaSet objects to ARTS control file.

        Input:
            - sequence of WSM and AgendaSet objects
            - filename (string-like)
            - settings: if set, pickle settings to this file.
              Special value 'True' means name based on filename.
        """
        with open(filename, 'w') as out:
            self.toFile(out)
        # write vars
        dirname = os.path.dirname(filename)
        basename = os.path.basename(filename)
        base_ext = os.path.splitext(basename)[0]
        for k, v in self.write_vars.items():
            fn = os.path.join(dirname, ".".join([base_ext, k, "xml"]))
            print now(), "writing", fn
            artsXML.save(v, fn)
        if settings_file is not None:
            if settings_file is True: # special value, autocalculate name
                settings_file = os.path.splitext(filename)[0] + ".pickle"
            with open(settings_file, 'w') as fp:
                cPickle.dump(self.settings, fp, protocol=cPickle.HIGHEST_PROTOCOL)

    def missing(self, *args):
        """Called when required setting is missing.

        Raises an exception.
        """

        exc = general.PyARTSError
        settings = str(self.settings.keys())
        if len(args) == 0:
            raise exc("Some settings were missing")
        elif len(args) == 1:
            raise exc("Missing setting: %s. Settings found: %s" % \
                (args[0], settings))
        else:
            raise exc("At least one of these settings must be present: %s. " \
                      "Settings found: %s" % (args, settings))
                

    def write_outputs(self, *allvars):
        for var in allvars:
            self.add(WriteXML(var))
        self.out.extend(allvars)

    @contextlib.contextmanager
    def batch(self, agenda="ybatch_calc_agenda"):
        if self.current_batch_agenda is not None:
            raise general.PyARTSError("current_batch_agenda already set " \
                    " to: %s" % self.current_batch_agenda)
        self.current_batch_agenda = AgendaSet("ybatch_calc_agenda")
        yield
        curbat = self.current_batch_agenda
        self.current_batch_agenda = None
        self.add(curbat)

    ##### Outside atmosphere-specific part #####

    @commenter
    @runonce
    def setup_general(self):
        """Generates a list of arts general settings commands.
        
        Must have:
            atmosphere_dim
        May have:
            continua (True)
                whether to include continua.arts
            stokes_dim
                number setting the stokes dimension.
            y_unit
                set the y_unit
        """
        # general includes
        self.add(Include("general.arts"))
        if self.settings.get("continua", True):
            self.add(Include("continua.arts"))
        # set stokes dimension
        if "stokes_dim" in self.settings:
            self.add(IndexSet("stokes_dim", self.settings["stokes_dim"]))
        self.add(WSM("AtmosphereSet%dD" % self.settings["dim"]))

        if "y_unit" in self.settings:
            self.add(WSM("StringSet", "y_unit", quotify(self.settings["y_unit"])))
        self.y_unit = self.settings.get("y_unit")
        self.y_unit_applied = self.y_unit

        if self.settings.get("full_spectrum"):
            ag = AgendaSet("iy_clearsky_agenda")
            ag.add(WSM("iyEmissionStandardClearsky"))
            ag.add(WSM("WriteXMLIndexed", "output_file_format",
                       "ybatch_index", "iy",
                       quotify(self.settings["full_spectrum"])))
            self.add(ag)

    @commenter
    def setup_geoid(self):
        """Sets up the geoid.

        Can be run more than once because of the MC 1D->3D trick.
        """
        if self.settings.get("WGS84"):
            self.add(WSM("r_geoidWGS84"))
        else:
            self.add(WSM("r_geoidSpherical", "r_geoid", "atmosphere_dim",
                         "lat_grid", "lon_grid", EARTH_RADIUS))

    @commenter
    @runonce
    def setup_ground_behaviour(self):
        """Generates a list of commands to set up ground properties

        Must have:

        May have:
            WGS84
                If this evaluates to true, use WGS84.
                Otherwise, use spherical
            surface_reflectivity
                scalar or vector (with length of no. batch cases)

        """

        self.setup_geoid()
        if "surface_reflectivity" in self.settings:
            # set constant
            surf = AgendaSet("surface_prop_agenda")
            surf.add(Ignore("rte_pos"))
            surf.add(Ignore("rte_gp_p"))
            surf.add(WSM("InterpAtmFieldToRteGps",
                         "surface_skin_t", "atmosphere_dim", "rte_gp_p",
                         "rte_gp_lat", "rte_gp_lon", "t_field"))
            surf.add(WSM("surfaceFlatReflectivity"))
            if isinstance(self.settings["surface_reflectivity"], numbers.Number):
                #self.add(WSM("IndexCreate", "surface_scalar_reflectivity"))
                self.add(WSM("VectorSet", "surface_scalar_reflectivity", [self.settings["surface_reflectivity"]]))
                self.many_reflectivities = False
            else: # should be a vector of length of no. batch cases
                self.add(WSM("NumericCreate", "local_surface_reflectivity"))
                self.add(WSM("VectorCreate", "surface_reflectivities"))
                self.settings["surface_reflectivities"] = self.settings["surface_reflectivity"]
                self.writevar("surface_reflectivities")
                self.add(WSM("ReadXML", "surface_reflectivities"))
                #self.add(WSM("VectorSet", "surface_reflectivities", self.settings["surface_reflectivity"].tolist()))
                self.many_reflectivities = True
            self.add(surf)
        # else a blackbody, already set in general.arts

    @commenter
    @runonce
    def setup_gas_abs(self):
        """Sets up the gaseous absorption coefficient data.

        Possibilities:
        - line-by-line calculation (needs: line-by-line file)
        - absorption lookup table from file (needs: lookup table file)
        - calculated absorption lookup table (needs: line-by-line file)
        - already defined in sensor include file (needs: nothing)

        Options:
            abs_from_sensor
                Don't set abs_species, don't read a lookup table, don't
                read lines; this is done in the sensor-include file.
            abs_species
                Specifies what gas absorption species to consider.
                Must always be set.
            abs_lookup
                Indicates precalculated lookup table to read from.
            abs_lines
                Indicates file to read absorption lines from.
                Not compatible with abs_lookup.
            abs_lines_format
                format for file to read lines from: e.g. Arts, Hitran,
                Jpl, etc.
            line_range
                frequency range to read from hitran
            lookup
                boolean: create lookup table or not
            batch:
                if lookup, have batch cases or not

        Note: some include files, like the ones for setting up amsu or
        mhs, already include part of what's done here.
        """
        lookup = False # set to True in two situations
        if not self.settings.get("abs_from_sensor"):
            species_str = str(self.settings["abs_species"])
            self.add(WSM("SpeciesSet", "abs_species", species_str))
            self.add(WSM("abs_lines_per_speciesCreateFromLines"))
        if self.settings.get("abs_lookup"):
            # read lookup-table from file
            lookup = True
            self.add(WSM("abs_lookupInit"))
            self.add(ReadXML("abs_lookup", self.settings["abs_lookup"]))
            self.add(WSM("abs_lookupAdapt"))
        elif self.settings.get("abs_lines") and not self.settings.get("abs_from_sensor"):
            # read line info
            fn = self.settings["abs_lines"]
            form = self.settings["abs_lines_format"]
            lo, hi = self.settings["line_range"]
            self.add(WSM("abs_linesReadFrom%s" % form,
                         "abs_lines", quotify(fn), "%d"%lo, "%d"%hi))
            self.add(WSM("abs_lines_per_speciesCreateFromLines"))

        if self.settings.get("make_lookup"):
            lookup = True
            self.add(WSM("abs_lookupSetupBatch" if self.batchcase else "abs_lookupSetup"))
            self.add(WSM("abs_lookupCreate"))
            if self.settings.get("write_lookup"):
                self.add(WSM("WriteXML", "output_file_format", "abs_lookup",
                         quotify(self.settings["write_lookup"])))

        ag = AgendaSet("abs_scalar_gas_agenda")
        ag.add(WSM("Ignore", "rte_doppler"))
        if lookup:
            ag.add(WSM("abs_scalar_gasExtractFromLookup"))
        else:
            ag.add(WSM("abs_scalar_gasCalcLBL"))
        self.add(ag)

    @commenter
    @runonce
    def setup_sensor_ir_common(self, sensor, mode):
        self.setup_ir_init()
        if "abs_lookup" in self.settings:
            self.add(Include("hirs/hirs_general.arts"%vars()))
            self.add(Include("hirs/hirs_spectroscopy.arts"%vars()))
            self.add(Include("%(sensor)s/%(sensor)s_sensor_common.arts"%vars()))
            self.add(Include("%(sensor)s/%(sensor)s_sensor_%(mode)s.arts"%vars()))
        else:
            self.add(Include(constants.sensor_includes["%(sensor)s_%(mode)s"%vars()]))

    @commenter
    @runonce
    def setup_sensor_avhrr_fast(self):
        self.setup_sensor_ir_common(sensor="avhrr", mode="fast")

    @commenter
    @runonce
    def setup_sensor_avhrr_reference(self):
        self.setup_sensor_ir_common(sensor="avhrr", mode="reference")

    @commenter
    @runonce
    def setup_sensor_hirs_fast(self):
        self.setup_sensor_ir_common(sensor="hirs", mode="fast")

    @commenter
    @runonce
    def setup_sensor_hirs_reference(self):
        self.setup_sensor_ir_common(sensor="hirs", mode="reference")

    @commenter
    @runonce
    def setup_sensor_include_only(self, inc):
        self.add(Include(constants.sensor_includes[inc]))

    # catchall sensor getattr
    def __getattr__(self, at):
        if at.startswith("setup_sensor_"):
            return functools.partial(self.setup_sensor_include_only,
                                     at[13:])
        else:
            raise AttributeError("No such attribute: %s" % at)
#    setup_sensor_mhs_fast = functools.partial(setup_sensor_include_only,
#                                              "mhs_fast")
#    setup_sensor_mhs_reference = functools.partial(setup_sensor_include_only,
#                                                   "mhs_reference")
#    setup_sensor_amsub_fast = functools.partial(setup_sensor_include_only,
#                                                "amsub_fast")
#    setup_sensor_amsub_reference = functools.partial(setup_sensor_include_only,
#                                                "amsub_reference")
    @commenter
    @runonce
    def setup_sensor(self):
        """Sets up sensor.

        Monochromatic pencil-beam or full sensor setup...

        MUST HAVE
            Either:
                freq
                    frequency grid, can be vector or hash with keys
                    "start", "stop", "nelem" to be passed on to
                    VectorNLinSpace
                sensor
                    One of constants.sensor_includes, setup derived from
                    sensor settings, or a (list of) include-files.
        """

        # set frequency or frequencies
        if "freq" in self.settings:
            freq = self.settings["freq"]
            try:
                self.add(VectorNLinSpace("f_grid", freq["nelem"], freq["start"], freq["stop"]))
            except (TypeError, ValueError): # not a mapping, assume number-like
                self.add(VectorSet("f_grid", freq))
            self.add(WSM("sensorOff"))
        elif "sensor" in self.settings:
            sens = self.settings["sensor"]
            meth = "setup_sensor_%s" % sens
            if isinstance(sens, basestring) and hasattr(self, meth) and callable(getattr(self, meth)):
                getattr(self, meth)()
            else:
                # consider as (list of) include-file(s)
                if isinstance(sens, basestring) or not hasattr(sens, '__iter__'):
                    sens = [sens]
                for incf in sens:
                    self.add(Include(incf))
        else:
            self.missing("freq", "sensor")
        # antenna
        if "antenna" in self.settings:
            if self.settings["antenna"] is not None:
                # FIXME: implement this
                raise NotImplementedError("PyARTS only implements pencil-beam")
        else:
            self.add(WSM("AntennaOff"))
        
        placement_vars = "sensor_pos", "sensor_los"
        for k in placement_vars:
            if k in self.settings:
                v = self.settings[k]
                if isinstance(v, basestring): # interpret as xml
                    self.add(ReadXML(k, quotify(v)))
                else:
                    self.add(WSM("MatrixSet", k, general.array2arts(v)))

    
    #@verbose
    @commenter
    @runonce
    def setup_scattering_properties(self):
        """Read scattering data 

        Old-style:
            Must have:
                scat_data_raw
                    OR
                scat_data_file_list
        New-style:
            Must have:
                scat_data_and_meta_files 
                    OR
                old-style + scat_data_meta_array
            May have:
                part_species (see Daniels docs)
                    defaults as in example control-file
                size_range
                    minimum, maximum size in µm
        """
#        if not self.settings.get("cloud"):
#            raise general.PyARTSError("To set scattering props, 'cloud' must be True")
        meta = False
        # only scattering data (no meta)
        if "scat_data_mono" in self.settings:
            # read mono directly
            raise NotImplementedError("Reading mono directly no longer implemented")
        elif "scat_data_raw" in self.settings:
            # read raw directly
            self.add(ReadXML("scat_data_raw", self.settings["scat_data_raw"]))
        elif "scat_data_file_list" in self.settings:
            # use ParticleTypeAdd
            self.add(WSM("ParticleTypeInit"))
            for scat_data_file in self.settings["scat_data_file_list"]:
                self.add(
                    WSM("ParticleTypeAdd", "scat_data_raw",
                    "pnd_field_raw",
                    "atmosphere_dim", "f_grid", "p_grid", "lat_grid",
                    "lon_grid",
                    "cloudbox_limits", quotify(scat_data_file), '""'))
        elif "scat_data_and_meta_files" in self.settings:
            # use ScatteringParticleTypeAndMetaRead
            self.add(WSM("ScatteringParticleTypeAndMetaRead",
                         "scat_data_raw", "scat_data_meta_array",
                         "f_grid",
                         quotify(self.settings["scat_data_and_meta_files"][0]),
                         quotify(self.settings["scat_data_and_meta_files"][1])))
            meta = True
        else:
            self.missing("scat_data_raw", "scat_data_file_list", "scat_data_and_meta_files")

        # extra
        if "scat_data_raw_extra" in self.settings:
            prep = ""
            sdre = self.settings["scat_data_raw_extra"]
            t = "scat_data_raw_extra"
            if isinstance(sdre, ArtsType):
                mode = "internal"
            else:
                mode = "external"

            if isinstance(sdre, ArrayOf) or mode == "external":
                prep = "ArrayOf"

            if mode == "internal":
                self.writevar(t)

            self.add(WSM(prep+"SingleScatteringDataCreate", t))
            if mode == "internal":
                self.add(WSM("ReadXML", t))
            else:
                self.add(WSM("ReadXML", t, quotify(sdre)))
            self.add(WSM("Append", "scat_data_raw", t))
            self.add(WSM("Delete", t))
        # scattering and meta

        if "scat_data_meta_array" in self.settings:
            if meta:
                raise general.PyARTSError(
                    "Both scat_data_and_meta_files " 
                    "and scat_data_meta_array set, ambiguous.")
            # read meta directly
            self.add(ReadXML("scat_data_meta_array",
                     self.settings["scat_data_meta_array"]))
            meta = True
            # extra
            if "scat_data_meta_extra" in self.settings:
                sdme = self.settings["scat_data_meta_extra"]
                t = "scat_data_meta_extra"
                if mode == "internal":
                    self.writevar(t)
                    
                self.add(WSM(prep+"ScatteringMetaDataCreate", t))
                if mode == "internal":
                    self.add(WSM("ReadXML", t))
                else:
                    self.add(WSM("ReadXML", t, quotify(sdme)))
                self.add(WSM("Append", "scat_data_meta_array", t))
                self.add(WSM("Delete", t))

        if meta: # must also do selection
            size_min, size_max = self.settings.get("size_range", (0.1, 2000))
            sz_str = str(size_min) + "-" + str(size_max)
            if "part_species" in self.settings:
                if str(self.settings["part_species"]).startswith('["'):
                    p_sp = self.settings["part_species"]
                else:
                    p_sp = [quotify(self.settings["part_species"])]
            else:
                ice = "IWC-MH97-" + sz_str
                liq = "LWC-liquid-" + sz_str
                p_sp = [ice, liq]
            self.add(WSM("ParticleSpeciesSet", "part_species", p_sp))
            self.add(WSM("ScatteringParticlesSelect"))

    # this one may be different per atmosphere, but if explicitly set,
    # this is the same for all

    @runonce
    @commenter
    def setup_grids(self):
        """Generates commands to set up the grids.

        Settings used:
            p_grid, lat_grid, lon_grid. Can be:
                string (file to read grid from
                vector (vector to set grid to)
                mapping (keys 'nelem', 'start', 'stop' used with
                         linspace/logspace)

        """

        for grid in ("p_grid", "lat_grid", "lon_grid")[:self.settings["dim"]]:
            val = self.settings[grid]
            if isinstance(val, basestring):
                self.add(ReadXML(grid, val))
            elif isinstance(val, numpy.ndarray):
                self.add(VectorSet(grid, val))
            else:
                nelem, start, stop = \
                    val["nelem"], val["start"], val["stop"]
                self.add(IndexSet("nelem", nelem))
                self.add(self.grids[grid](grid, nelem, start, stop))

    @runonce
    @commenter
    def setup_ir_init(self):
        """Prepare for HIRS/AVHRR setup.

        Sets some strings and numbers that are needed to include HIRS/AVHRR

        Must have:
            satellite
            abs_lines or abs_lookup
        May have:
            channels
            views
            f_grid_spacing (only used for hirs_reference)
        """

        sat = "satellite"
        chans = "channels"
        views = "views"
        hitran = "hitran_file"
        self.add(WSM("StringCreate", sat))
        self.add(WSM("StringSet", sat, quotify(self.settings["satellite"].upper())))
        self.add(WSM("ArrayOfIndexCreate", chans))
        self.add(ArrayOfIndexSet(chans, self.settings.get("channels", numpy.arange(12))))
        self.add(WSM("ArrayOfIndexCreate", views))
        self.add(ArrayOfIndexSet(views, self.settings.get("views", numpy.arange(28))))
        if "abs_lines" in self.settings:
            self.add(WSM("StringCreate", hitran))
            self.add(WSM("StringSet", hitran, quotify(self.settings["abs_lines"])))
            if not self.settings["abs_lines_format"] == "Hitran2004":
                raise general.PyARTSError("HIRS/AVHRR only works with format=Hitran2004. Found format=%s" % self.settings["abs_lines_format"])
        elif not "abs_lookup" in self.settings:
            self.missing("abs_lines", "abs_lookup")
        if self.settings["sensor"].lower().endswith("reference") or self.settings.get("infrared"):
            grid = "f_grid_spacing" # only needed for hirs_reference
            self.add(WSM("NumericCreate", grid))
            self.add(WSM("NumericSet", grid, self.settings.get("f_grid_spacing", 5e8)))
        # make sure unit will be 1
        self.y_unit_applied = self.y_unit
        self.y_unit = "1"

    ##### Inside atmosphere-specific part #####

##    @commenter
##    @runonce
##    def setup_clear_atmosphere(self):
##        """Sets up commands for the clear atmosphere:
##
##        - Sets up grids (with setup_gridnD())
##        - Sets up gas species (with setup_gas_species())
##        
##        Settings used:
##            dim
##                dimensionality: 1 or 3
##            p_grid
##                mapping with keys start, nelem, stop or string with file
##                containing pressure grid
##            lat_grid, lon_grid
##                (only if dim==3)
##                either mappings with keys start, nelem, stop or strings to files
##                containing grids
##            3Dfields, boolean (optional, only applicable if dim==3)
##                if true: call AtmFieldsCalc
##                otherwise: call AtmFieldsCalcExpand1D
##        EITHER
##            z_field, t_field, vmr_field
##                paths to files containing fields
##        OR
##            atm_basename
##                basename to pass to AtmRawRead to find fields
##
##        """
##
##        # set up grid
##        if dim == 1:
##            self.setup_grid1D()
##        elif dim == 3:
##            self.setup_grid3D()
##        else:
##            raise general.PyARTSError("Dimension should be 1 or 3")
##
##        self.setup_gas_species()

    @commenter
    @runonce
    def setup_single_gas_species(self):
        """Sets up gas species. Only applicable for single-calculations.

        From settings, has paths to either:
            - atm_fields_compact
            - z_field, t_field, vmr_field
            - atm_basename

        - Sets species strings
        - Reads z_field, t_field, vmr_field
        - Calculates those from raw fields if necessary

        """

        if self.batchcase:
            raise general.PyARTSError("Wrong function for batch")

        if "atm_fields_compact" in self.settings:
            self.add(ReadXML("atm_fields_compact",
                     self.settings["atm_fields_compact"]))
            self.add(WSM("AtmFieldsFromCompact"))
        elif "z_field" in self.settings and "t_field" in self.settings and \
           "vmr_field" in self.settings:
            self.add(ReadXML("z_field", self.settings["z_field"]))
            self.add(ReadXML("t_field", self.settings["t_field"]))
            self.add(ReadXML("vmr_field", self.settings["vmr_field"]))
        elif "atm_basename" in self.settings:
            self.add(WSM("AtmRawRead","t_field_raw", "z_field_raw",
                         "vmr_field_raw", "abs_species",
                         quotify(self.settings["atm_basename"])))
            if self.settings["dim"] == 3:
                if self.settings.get("3Dfields"):
                    self.add(WSM("AtmFieldsCalc"))
                else:
                    self.add(WSM("AtmFieldsCalcExpand1D"))
            else:
                self.add(WSM("AtmFieldsCalc"))

    @commenter
    @runonce
    def setup_cloud(self):
        """Do a single cloud

        Sets up the cloud:
            - sets the cloud box (may be "auto")
            - sets the pnd fields

        To be documented!
                
        """
        if self.settings.get("cloud"):
            self.add(WSM("cloudboxSetAutomatically"))
            self.add(WSM("pnd_fieldSetup"))
        else:
            self.add(WSM("cloudboxSetManually", "cloudbox_on", "cloudbox_limits",
                         "atmosphere_dim", "p_grid", "lat_grid", "lon_grid",
                         *[self.settings["cloud_box"][s] for s in ("p1", "p2", "lat1",
                         "lat2", "lon1", "lon2")]))
            # establish scattering property settings
            #    MOVED to setup_scattering!
            # establish particle number density settings
            if "pnd_field_raw" in self.settings:
                self.add(ReadXML("pnd_field_raw",
                                 quotify(self.settings["pnd_field_raw"])))
                self.add(WSM("pnd_fieldCalc"))
            elif "pnd_field" in self.settings:
                pnd_field = self.settings["pnd_field"]

                if isinstance(pnd_field, basestring):
                    self.add(ReadXML("pnd_field", quotify(pnd_field)))
                else:
                    self.add(WSM("Tensor4SetConstant", "pnd_field",
                                 len(scat_data_file_list),
                                 self.settings["p_grid"]["nelem"],
                                 self.settings["lat_grid"]["nelem"],
                                 self.settings["lon_grid"]["nelem"],
                                 pnd_field))
            else:
                self.missing("pnd_field_raw", "pnd_field")

    @commenter
    @runonce
    def setup_ground_properties(self):
        """Sets the ground properties (z, t)

        May have:
            z_surface
                if defined as a string, read from file
                if defined as a number, set constant
                if not defined, extract from z_field
        """
        # surface
        if self.settings.get("z_surface"):
                
            if isinstance(z_surface, basestring):
                self.add(ReadXML("z_surface", z_surface))
            elif str(z_surface).isdigit():
                self.add(WSM("nrowsGet", "nrows", "r_geoid"))
                self.add(WSM("ncolsGet", "ncols", "r_geoid"))
                self.add(WSM("MatrixSetConstant", "z_surface", "nrows",
                             "ncols", z_surface))
            else:
                raise NotImplementedError("z_surface must be a string " \
                                          "(path to xml-file) or constant.  " \
                                          "The rest is not implemented.")
        else:
            self.add(WSM("Extract", "z_surface", "z_field", 0))
        self.add(WSM("Extract", "t_surface", "t_field", 0))
        if getattr(self, "many_reflectivities"): # hopefully in batch
            self.add(WSM("Extract", "local_surface_reflectivity",
                         "surface_reflectivities", "ybatch_index"))
            # FIXME: write to file instead
            self.add(WSM("VectorSetConstant", "surface_scalar_reflectivity", 1,
                         "local_surface_reflectivity"))


    @commenter
    @runonce
    def setup_batch_atmosphere_from_chevalier(self):
        """Get batch_atm_fields_compact from Chevalier-ArrayOfMatrix.

        This will probably change in the future.

        Must have:
            cheval_file
        May have:
            cheval_selection
                calculate only for those atmospheres
           
        """
        if isinstance(self.settings["cheval_file"], basestring):
            chev_files = (self.settings["cheval_file"],)
            chev_sel = (self.settings.get("cheval_selection", None),)
        else:
            chev_files = self.settings["cheval_file"]
            chev_sel = self.settings.get("cheval_selection", (None,)*len(chev_files))

        tmpvar_total = self.tempvar
        self.add(WSM("ArrayOfMatrixCreate", tmpvar_total))
        for (i, chev_file) in enumerate(chev_files):
            tmpvar_here = self.tempvar
            self.add(WSM("ArrayOfMatrixCreate", tmpvar_here))
            self.add(ReadXML(tmpvar_here, quotify(chev_file)))
            spec = io.get_species_from_arrayofmatrixfile(chev_file)
                
            if chev_sel[i] is not None:
                self.add(WSM("ArrayOfIndexCreate", "batch_selection"))
                self.add(WSM("ReadXML", "batch_selection"))
                self.settings["batch_selection"] = chev_sel[i].tolist()
                self.writevar("batch_selection")
                self.add(WSM("Select", tmpvar_here, tmpvar_here,
                         "batch_selection"))
            self.add(WSM("Append", tmpvar_total, tmpvar_here))
            self.add(WSM("Delete", tmpvar_here))

        # TODO/FIXME: this will be unified soon
        if self.settings.get("cloud"):
            self.add(WSM("batch_atm_fields_compactFromArrayOfMatrixChevalAll",
                         "batch_atm_fields_compact",
                         "batch_atm_fields_compact_all",
                         "atmosphere_dim",
                         tmpvar_total,
                         ["T", "z", "LWC", "IWC", "Rain", "Snow"] + spec))
        else:
            self.add(WSM("batch_atm_fields_compactFromArrayOfMatrix",
                         "batch_atm_fields_compact",
                         "atmosphere_dim", 
                         tmpvar_total,
                         ["T", "z"] + spec))
        self.add(WSM("Delete", tmpvar_total))
        self.add(WSM("nelemGet", "ybatch_n", "batch_atm_fields_compact"))
        self.add(Print("ybatch_n"))
        self.__fields = set(spec)

    @commenter
    @runonce
    def setup_batch_atmosphere_extra(self):
        """Adds extra atmospheric fields, e.g. when main is from Chevalier.

        Must be called after setup_batch_atmosphere_from_*.

        By default, this adds N2 and O2 according to
        constants.well_mixed_species.
        
        May have:
            extra_fields
                dict with additional fields. Values are either:
                    numbers, interpreted as constant
                    strings, interpreted as path to GriddedField3
                    GriddedFields; to be implemented!
                The default values are as for constants.well_mixed_species.
        """
        tmpvar = self.tempvar
        extra = constants.well_mixed_species.copy()
        extra.update(self.settings.get("extra_fields", {}))
        extra_ordered = general.OrderedDict()
        if "abs_species" in self.settings:
            musthave = self.settings["abs_species"].species()
        elif self.settings.get("sensor") in constants.sensor_species:
            musthave = constants.sensor_species[self.settings["sensor"]]
        else:
            raise general.PyARTSError(\
                "Cannot determine required atmospheric species. " \
                "I need either abs_species or a sensor defining it.")
        # need to add extras in the correct order
        for species in musthave:
            if species not in self.__fields:
                # next line will fail if required species not specified
                extra_ordered[species] = extra.pop(species)
        if extra: # not empty
            raise general.PyARTSError(\
                "extra species dict contains species not in abs_species. " \
                "Unexpected species: %s. abs_species: %s." % \
                    (extra.keys(), musthave))
        for k in extra_ordered:
            v = extra_ordered[k]

            if isinstance(v, numbers.Real):
                self.add(WSM("batch_atm_fields_compactAddConstant",
                             "batch_atm_fields_compact",
                             quotify(k), v))
                if self.settings.get("cloud"):
                    self.add(WSM("batch_atm_fields_compactAddConstant",
                                 "batch_atm_fields_compact_all",
                                 quotify(k), v))
            elif isinstance(v, basestring):
                # interpret as path to file containing GriddedField3
                # FIXME: should not be needed to have new name in each # iteration?
                tmpvar = self.tempvar
                self.add(WSM("GriddedField3Create", tmpvar))
                self.add(ReadXML(tmpvar, v))
                self.add(WSM("batch_atm_fields_compactAddSpecies",
                         "batch_atm_fields_compact",
                         quotify(k), tmpvar))
                if self.settings.get("cloud"):
                    # FIXME/TODO: this shan't be needed much longer
                    self.add(WSM("batch_atm_fields_compactAddSpecies",
                             "batch_atm_fields_compact_all",
                             quotify(k), tmpvar))
                self.add(WSM("Delete", tmpvar))
            else:
                raise general.PyARTSError(\
                    "I don't know how to add %s:%s" % (k, v))

    @commenter
    @runonce
    def atmosphere_from_batch(self):
        """Extract atmosphere from batch of atmospheres

        Options: massdensity_threshhold (default: 1e-14)
        """

        # TODO/FIXME: this may be unified in the future
        if self.settings.get("cloud"):
            self.add(WSM("Extract", "atm_fields_compact_all",
                         "batch_atm_fields_compact_all", "ybatch_index"))
            self.add(WSM("AtmFieldsFromCompactChevalAll"))
            self.add(WSM("Massdensity_cleanup", "massdensity_field", 
                repr(self.settings.get("massdensity_threshhold", 1e-14))))
        else:
            self.add(WSM("Extract", "atm_fields_compact",
                         "batch_atm_fields_compact", "ybatch_index"))
            self.add(WSM("AtmFieldsFromCompact"))

    ##### Method-specific finalisations ###

    @commenter
    @runonce
    def setup_doit(self):
        """Prepares for DOIT calculations

        May have:
            doit_limits: passed to doit_conv_flagAbsBT
            N_aa_grid:   passed to DoitAngularGridsSet
            N_za_grid:   passed to DoitAngularGridsSet
        """
        if self.settings["dim"] != 1:
            raise general.PyARTSError(
                    "Don't use DOIT for multi-dimensional " \
                    "calculations, use MonteCarlo.")

        ## add some agendas
        dma = AgendaSet("doit_mono_agenda")
        dma.add(WSM("DoitScatteringDataPrepare"))
        dma.add(WSM("doit_i_fieldSetClearsky"))
        dma.add(WSM("doit_i_fieldIterate"))
        dma.add(WSM("DoitCloudboxFieldPut"))
        # FIXME: do something with it?
        self.add(dma)

        pmsa = AgendaSet("pha_mat_spt_agenda")
        pmsa.add(WSM("pha_mat_sptFromDataDOITOpt"))
        self.add(pmsa)

        dsfa = AgendaSet("doit_scat_field_agenda")
        dsfa.add(WSM("doit_scat_fieldCalcLimb" if self.islimb \
                    else "doit_scat_fieldCalc"))
        self.add(dsfa)

        dra = AgendaSet("doit_rte_agenda")
        dra.add(WSM("doit_i_fieldUpdateSeq1D"))
        self.add(dra)

        sca = AgendaSet("spt_calc_agenda")
        sca.add(WSM("opt_prop_sptFromMonoData"))
        self.add(sca)

        oppa = AgendaSet("opt_prop_part_agenda")
        oppa.add(WSM("ext_matInit"))
        oppa.add(WSM("abs_vecInit"))
        oppa.add(WSM("ext_matAddPart"))
        oppa.add(WSM("abs_vecAddPart"))
        self.add(oppa)

        limits = self.settings.get("doit_limits", [0.1, 0.01, 0.01, 0.01][:self.settings["stokes_dim"]])
        dcta = AgendaSet("doit_conv_test_agenda")
        dcta.add(WSM("doit_conv_flagAbsBT", "doit_conv_flag",
                     "doit_iteration_counter", "doit_i_field",
                     "doit_i_field_old", "f_grid", "f_index",
                     limits))
        self.add(dcta)

        ica = AgendaSet("iy_cloudbox_agenda")
        ica.add(Ignore("ppath"))
        ica.add(Ignore("rte_pos"))
        ica.add(Ignore("rte_los"))
        ica.add(WSM("iyInterpCloudboxField"))
        self.add(ica)

        ## other preparation
        self.add(WSM("doit_za_interpSet", "doit_za_interp",
                     "atmosphere_dim", quotify("linear")))
        if self.islimb:
            if "za_grid_opt_file" in self.settings:
                za_file = self.settings["za_grid_opt_file"]
            else:
                raise NotImplementedError(
                    "For DOIT limb calculations, must calculate " \
                    "optimized grid. PyARTS cannot do this yet. " \
                    "To override, set za_grid_opt_file in settings.")
        else:
            za_file = ""

                                    
        self.add(WSM("DoitAngularGridsSet", "doit_za_grid_size",
                     "scat_aa_grid", "scat_za_grid",
                     self.settings.get("N_za_grid", 19),
                     self.settings.get("N_aa_grid", 37),
                     quotify(za_file)))


    @commenter
    @runonce
    def setup_DISORT(self):
        """Prepares for DISORT calculations

        """
        raise NotImplementedError("DISORT not implemented yet")

    @commenter
    @runonce
    def setup_montecarlo(self):
        """Prepares for MonteCarlo simulations

        Can use std_err, max_time, max_iter
        """
        #self.add(IndexSet('nelem', self.settings['stokes_dim']))
        #self.add(VectorSet('mc_error', 0))
        #self.add(IndexSet('mc_iteration_count', 0))
        #self.add(IndexSet('ncols', self.settings['stokes_dim']))
        #self.add(IndexSet('nrows', self.settings['stokes_dim']))
        #self.add(WSM('StringSet', "y_unit", quotify('RJBT')))
        if "mc_seed" in self.settings:
            self.add(IndexSet('mc_seed', self.settings["mc_seed"]))
        else:
            self.add(WSM('MCSetSeedFromTime'))
        #self.add(WriteXML("mc_seed"))
        self.add(NumericSet("mc_std_err", self.settings.get("std_err", -1)))
        self.add(IndexSet("mc_max_time", self.settings.get("max_time", -1)))
        self.add(IndexSet("mc_max_iter", self.settings.get("max_iter", -1)))
#        self.add(IndexSet("mc_z_field_is_1D",
#                          self.settings.get("z_field_is_1D",
#                                            1 if self.settings["dim"] == 1 else 0-1)))
        if "antenna" in self.settings:
            self.add(WSM('mc_antennaSetGaussianByFWHM',
                     "mc_antenna",
                     self.settings["antenna"]["za_fwhm"],
                     self.settings["antenna"]["aa_fwhm"]))
        else:
            self.add(WSM('mc_antennaSetPencilBeam'))
        if self.y_unit != "1":
            self.y_unit_applied = self.y_unit
            self.y_unit = "1"

    @commenter
    @runonce
    def setup_mc_ycalc(self):
        """Prepares for MonteCarlo simulations with sensor.

        Also calls setup_montecarlo.
        """
        self.setup_montecarlo()
        ag = AgendaSet("iy_clearsky_agenda")
        ag.add(Ignore("iy_error"))
        ag.add(Ignore("iy_error_type"))
        ag.add(WSM("iyMC"))
        self.add(ag)

    @commenter
    def expand_matrix(self, mat, n):
        """Expands matrix (e.g. sensor_pos, sensor_los) to n cols.

        First column is copied, rest are set to 0.
        This can be useful to expand sensor_pos or sensor_los to 3D.
        """

        nrows = "nrows_" + self.tempvar
        mat2vec = "mat2vec_" + self.tempvar
        zeroes = "zeroes_" + self.tempvar
        self.add(WSM("IndexCreate", nrows))
        self.add(WSM("nrowsGet", nrows, mat))
        self.add(WSM("VectorCreate", zeroes))
        self.add(WSM("VectorSetConstant", zeroes, nrows, 0))
        self.add(WSM("VectorCreate", mat2vec))
        self.add(WSM("VectorExtractFromMatrix", mat2vec, mat, 0, quotify("column")))
        if n==2:
            self.add(WSM("Matrix2ColFromVectors", mat, mat2vec, zeroes))
        elif n==3:
            self.add(WSM("Matrix3ColFromVectors", mat, mat2vec, zeroes, zeroes))
        else:
            raise ValueError("vector can only be expanded to 2 or 3 cols, "
                            "found %d cols" % cols)
        for v in (nrows, mat2vec, zeroes):
            self.add(WSM("Delete", v))

        
    @commenter
    @runonce
    def setup_hack_1D3D_mcir(self):
        """Sets up some MC-IR hacks to use for 1D

        Sets up dummy values z_dummy, t_dummy, sub_lat_grid, sub_lon_grid.
        This is needed later for the 1D/3D MC/IR trick.
        """
        self.add(WSM("IndexCreate", "n_p"))
        for w in "tz":
            self.add(WSM("Tensor3Create", w+"_dummy"))
        for w in ("sub_lat_grid", "sub_lon_grid"):
            self.add(WSM("VectorCreate", w))
        self.expand_matrix("sensor_pos", 3)
        self.expand_matrix("sensor_los", 2)


    ##### Higher-level stuff #####

    @commenter
    @runonce
    def setup_common(self):
        """Sets up gas_abs, ground, sensor
        """

        # sensor first; lookup table needs f_grid
        self.setup_sensor()
        self.setup_gas_abs()
        self.setup_ground_behaviour()

    # Finally, simulation commands

    @commenter
    @runonce
    def simulate_clearsky(self):
        """Simulate with clearsky setup
        """
        self.add(WSM('jacobianOff'))
        if self.y_unit != self.y_unit_applied:
            self.add(WSM('StringSet','y_unit', quotify(self.y_unit)))
        self.add(WSM("basics_checkedCalc"))
        self.add(WSM("cloudbox_checkedCalc"))
        self.add(WSM("yCalc"))
        if self.y_unit != self.y_unit_applied:
            self.add(WSM("StringSet", "y_unit", quotify(self.y_unit_applied)))
            self.add(WSM("y_unitApply"))
##        self.write_outputs("y")


    @commenter
    @runonce
    def simulate_montecarlo(self):
        """Simulate with MonteCarlo (MCGeneral)
        """

        self.add(WSM("basics_checkedCalc"))
        self.add(WSM("cloudbox_checkedCalc"))
        self.add(WSM("MCGeneral"))
##        self.write_outputs("y", "mc_error", "mc_iteration_count")

    @commenter
    @runonce
    def simulate_mc_ycalc_hack(self):
        """Hack to use MonteCarlo for 1D.

        pnd_fieldSetup wants 1D, pnd fields needs to be expanded to edges
        minus 20 degrees, rest need to be expanded to edges
        """
        cb_set = WSM("cloudboxSetManually", "cloudbox_on", "cloudbox_limits",
                     "atmosphere_dim", "p_grid", "lat_grid", "lon_grid",
                     2000000, 0, -20, 20, -20, 20)

        self.add(WSM("nelemGet", "n_p", "p_grid"))
        for w in "tz":
            self.add(WSM("Tensor3SetConstant", w+"_dummy", "n_p", 1, 1, 0))
        for w in ("sub_lat_grid", "sub_lon_grid"):
            self.add(WSM("VectorLinSpace", w, -20, 20, 1))


        self.add(WSM("AtmosphereSet1D"))
        self.add(cb_set)
        self.add(WSM("pnd_fieldSetup"))
        self.add(WSM("AtmosphereSet3D"))
        for w in ("lat_grid", "lon_grid"):
            self.add(WSM("VectorLinSpace", w, -40, 40, 1))
        self.add(cb_set)
        self.add(WSM("AtmFieldsExpand1D", "t_field", "z_field",
                     "vmr_field", "p_grid", "lat_grid", "lon_grid",
                     "atmosphere_dim"))
        self.add(WSM("AtmFieldsExpand1D", "t_dummy", "z_dummy",
                     "pnd_field", "p_grid", "sub_lat_grid",
                     "sub_lon_grid", "atmosphere_dim"))
        self.setup_geoid() # do this again

    @commenter
    @runonce
    def simulate_mc_ycalc(self):
        """Simulate with MonteCarlo with sensor (iyMC etc.)

        Since MC-specific stuff is setup in the iy_clearsky_agenda, this
        is actually the same as simulate_clearsky!
        """

        self.simulate_clearsky()

    @commenter
    @runonce
    def simulate_doit(self):
        """Simulate with DOIT

        Can be inside batch-calculation
        """

        if self.y_unit != self.y_unit_applied:
            self.add(WSM('StringSet','y_unit', quotify(self.y_unit)))
        self.add(WSM("basics_checkedCalc"))
        self.add(WSM("cloudbox_checkedCalc"))
        self.add(WSM("CloudboxGetIncoming"))
        self.add(WSM("DoitInit"))
        self.add(WSM("ScatteringDoit"))
        self.add(WSM("yCalc"))
        if self.y_unit != self.y_unit_applied:
            self.add(WSM("StringSet", "y_unit", quotify(self.y_unit_applied)))
            self.add(WSM("y_unitApply"))
##        self.write_outputs("y")

    ##################################
    ##### Alternate constructors #####
    ##################################
    #
    # Naming rule: do_NAME

    @classmethod
    def do_batch_doit(cls, filename="PyARTS_batch_doit.arts", **settings):
        settings["batch"] = True
        settings["cloud"] = True
        settings.setdefault("dim", 1)
        settings.setdefault("make_lookup", True)
        code = cls(**settings)
        # setup atmos before common, lookup table creation needs it
        code.setup_batch_atmosphere_from_chevalier()
        code.setup_batch_atmosphere_extra()
        code.setup_common()
        code.setup_scattering_properties()
        code.setup_doit()
        with code.batch():
            code.atmosphere_from_batch()
            code.setup_ground_properties()
            code.setup_cloud()
            code.simulate_doit()
            if settings.get("batch_write"):
                code.add(WSM("WriteXMLIndexed", "output_file_format",
                "ybatch_index", "y", general.quotify(settings["batch_write"])))
        code.add(WSM("ybatchCalc"))
        code.write_outputs("ybatch")
        code.toControlFile(filename, True)
        return code

    @classmethod
    def do_batch_montecarlo(cls, filename="PyARTS_batch_mc.arts", **settings):
        settings["batch"] = True
        settings["cloud"] = True
        settings.setdefault("make_lookup", True)
        code = cls(**settings)
        code.setup_batch_atmosphere_from_chevalier()
        code.setup_batch_atmosphere_extra()
        code.setup_common()
        code.setup_scattering_properties()
        #code.setup_montecarlo()
        code.setup_mc_ycalc()
        # is always 1D
        code.setup_hack_1D3D_mcir()
        with code.batch():
            code.atmosphere_from_batch()
            code.simulate_mc_ycalc_hack()
            code.setup_ground_properties()
            code.simulate_mc_ycalc()
            if settings.get("batch_write"):
                code.add(WSM("WriteXMLIndexed", "output_file_format",
                "ybatch_index", "y", general.quotify(settings["batch_write"])))
        code.add(WSM("IndexCreate", "no_freq"))
        #code.add(WSM("IndexCreate", "no_photon"))
        code.add(WSM("nelemGet", "no_freq", "f_grid"))
        #code.add(WSM("nelemGet", "no_photon", "mc_max_iter"))
        code.add(WSM("Print", "no_freq"))
        code.add(WSM("Print", "mc_max_iter"))
        code.add(WSM("ybatchCalc"))
        code.write_outputs("ybatch")
        code.toControlFile(filename, True)
        return code


    @classmethod
    def do_batch_clear(cls, filename="PyARTS_batch_clear.arts", **settings):
        settings["batch"] = True
        settings.setdefault("make_lookup", True)
        code = cls(**settings)
        code.setup_batch_atmosphere_from_chevalier()
        code.setup_batch_atmosphere_extra()
        code.setup_common()
        with code.batch():
            code.atmosphere_from_batch()
            code.setup_ground_properties()
            code.simulate_clearsky()
            if settings.get("batch_write"):
                code.add(WSM("WriteXMLIndexed", "output_file_format",
                "ybatch_index", "y", general.quotify(settings["batch_write"])))
        code.add(WSM("ybatchCalc"))
        code.write_outputs("ybatch")
        code.toControlFile(filename, True)
        return code


    @classmethod
    def do_montecarlo(cls, filename="PyARTS_Montecarlo.arts", **kwargs):
        """Generate Arts Montecarlo file.

        Returns ArtsCodeGenerator object.
        """
        if "dim" in kwargs and kwargs["dim"] != 3:
            warn("MonteCarlo: changing dim from %d to 3" % kwargs["dim"],
                 general.PyARTSWarning)
            kwargs["dim"] = 3
        code = cls(**kwargs)
        code.setup_grids()
        code.setup_sensor()
        code.setup_gas_abs()
        code.setup_single_gas_species()
        code.setup_ground_behaviour()
        code.setup_ground_properties()
        code.setup_cloud()
        code.setup_montecarlo()
        code.simulate_montecarlo()
        code.toControlFile(filename, True)
        return code

    @classmethod
    def do_doit(cls, filename="PyARTS_DOIT.arts", **kwargs):
        if "dim" in kwargs and kwargs["dim"] != 1:
            raise general.PyARTSError("DOIT dim must be 1, got: %d" % \
                    kwargs["dim"])
        else:
            kwargs["dim"] = 1

        code = cls(**kwargs)
        code.setup_common()
        code.setup_cloud()
        code.setup_doit()
        code.simulate_doit()
        code.toControlFile(filename, True)
        return code

    @classmethod
    def do_clear(cls, filename="PyARTS_Clear.arts", **kwargs):
        """Generates arts 1D clear sky control file for given keyword arguments
        """

        code = cls(**kwargs)
        code.setup_common()
        code.setup_grids()
        code.simulate_clearsky()

        code.toControlFile(filename, True)
        return code

    @classmethod
    def do_iwp_tau(cls, filename="PyARTS_iwp_tau.arts", **kwargs):
        """Generates an arts control file for the calcualtion of FOV averaged ice
        water path and cloud optical path
        """

        kwargs["dim"] = 3
        code = cls(**kwargs)

        code.setup_sensor()
        code.setup_grids()
        code.setup_gas_abs()
        code.setup_single_gas_species()
        code.setup_ground_behaviour()
        code.setup_ground_properties()
        #code.setup_sensor()
        code.setup_cloud()
        code.setup_scattering_properties()
        code.add(WSM("scat_data_monoCalc"))
        code.setup_montecarlo()
        code.addMulti(
                    [
                     ReadXML('particle_masses', code.settings['particle_masses']),
                     Print('mc_seed'),
                     IndexSet("mc_max_iter", code.settings["max_iter"]),
##                     WSM("IndexCreate", "max_iter"),
##                     IndexSet("max_iter", code.settings["max_iter"]),
                     WSM("mc_IWP_cloud_opt_pathCalc", "mc_IWP",
                         "mc_cloud_opt_path", "mc_IWP_error",
                         "mc_cloud_opt_path_error", "mc_iteration_count",
                         "mc_antenna", "sensor_pos", "sensor_los",
                         "ppath_step_agenda", "p_grid", "lat_grid",
                         "lon_grid", "r_geoid", "z_surface", "z_field",
                         "t_field", "vmr_field", "cloudbox_limits",
                         "pnd_field", "scat_data_mono", "particle_masses",
                         "mc_seed", 1000),
                     WriteXML('mc_IWP'),
                     WriteXML('mc_cloud_opt_path'),
                     WriteXML('mc_IWP_error'),
                     WriteXML('mc_cloud_opt_path_error'),
                     WriteXML('mc_iteration_count')])

        code.toControlFile(filename, True)
        return code

    @classmethod
    def do_ppath(cls, filename="PyARTS_ppathCalc3D.arts", **kwargs):
        """Performs a clear sky ppath calculation (useful for plotting ppaths)
        """
        kwargs["stokes_dim"] = 1
        kwargs["dim"] = 3
        code = cls(**kwargs)
        code.setup_common()
        code.setup_sensor()
        code.add(NumericSet("ppath_lmax", code.settings["lmax"]))
        # FIXME: GH 2011-03-04 WSM "ppath" does not exist, removed
        code.add(AgendaSet("ppath_step_agenda",
                           [WSM("ppath_stepGeometric"),
                            WSM("ppathCalc")]))
##                            WSM("ppath")]))
        code.toControlFile(filename, True)
        return code

    def tan_z2sensor_poslos(self):
        """If the tan_z argument
        (meaning tangent height) is present, then rte_pos and rte_pos is set
        accordingly.  Currently there is no need to account for azimuthal
        coordinates, so for now the line of sight is considered to be in the
        lon=0 plane, with the tangent point at (0.0 N,0.0 E)"""
        if "tan_z" in self.settings:
            tan_z = self.settings["tan_z"]
            if verbosity:
                print "Setting rte_los and rte_pos from tan_z argument"
            r_geoid = self.settings["r_geoid"]
            if r_geoid<0:
                r_geoid=6.378e6
            z_max=get_z_max()

            rte_los={}
            rte_pos={}
            rte_los["za"] = 180/numpy.pi*arccos((tan_z+r_geoid)/(z_max+r_geoid))+90
            rte_pos["lat"] = rte_los["za"]-90
            rte_pos["r_or_z"] = z_max+0.1#work around ppath.cc:5844
            rte_pos["corrected"] = True
            rte_pos["lon"] = 0
            rte_los["aa"] = 180
            self.settings["rte_los"] = rte_los
            self.settings["rte_pos"] = rte_pos
        else:
            if verbosity:
                print "tan_z argument not present, using rte_los and rte_pos"

    def tan_z2sensor_poslos_thru_cloud(self):
        """If tan_z is present in the settings, rte_los
        and rte_pos are set so that the line of sight passes through the
        centre of the cloudbox (which for now must be at (0N,0E))"""

        if "tan_z" in self.settings:
            tan_z = self.settings["tan_z"]
            r_geoid = self.settings["r_geoid"]
            if r_geoid < 0:
                r_geoid = 6.378e6
            z_max = self.get_z_max()
            cloudbox = self.settings["cloud_box"]
            z_mid = 8e3*log10(101325**2/(cloudbox["p1"]*cloudbox["p2"]))
            z_cloudboxtop = 16e3*log10(101325/cloudbox["p2"])

            if not tan_z <= z_cloudboxtop:
                raise general.PyARTSError("It looks like the line of sight " \
                                  "doesn't pass through the cloud box")

            if tan_z < z_mid:
                if verbosity:
                    print "Setting line of sight to pass through the centre of the" \
                          " cloud box and the given tangent height"

                r_mid = r_geoid + z_mid
                r_sensor = r_geoid + z_max + 0.1
                r_tp = r_geoid + tan_z

                a = arccos(r_tp/r_mid)
                bplusa = arccos(r_tp/r_sensor)

                rte_los = {}
                rte_pos = {}

                rte_los["za"] = bplusa*180/numpy.pi+90
                rte_los["aa"] = 180

                if not rte_los["za"] > 90:
                    raise general.PyARTSError("zenith angle should be greater than 90")

                rte_pos["lat"] = (bplusa-a)*180/numpy.pi
                rte_pos["lon"] = 0
                rte_pos["r_or_z"] = z_max+0.1#work-around ppath.cc:5844
                rte_pos["corrected"] = True
                self.settings["rte_los"]=rte_los
                self.settings["rte_pos"]=rte_pos
            else:
                if verbosity:
                    print "The given tangent height is above the mid-point of the "\
                          "cloud box.  The tangent point is being set to "\
                          "(0.0N, 0.0E, tan_z)."
                self.tan_z2sensor_poslos()
        else:
            if verbosity:
                print "tan_z argument not present, using rte_los and rte_pos"

    def get_z_max(self):
        """Returns the largest z value of the modelled atmosphere"""
        from arts_types import GriddedField3
        if "z_field" in self.settings:
            z_file=self.settings["z_field"]
            z=artsXML.load(z_file)["Tensor3"]

        else:
            z_fileself.settings["atm_basename"]+".z.xml"
            z=GriddedField3.load(z_file).data["data"]
        nz=z.shape[0]
        z_max=z[nz-1,0,0]
        return z_max

def safe_latlon_limits(self, cloud_box,z_field,r_geoid=EARTH_RADIUS):
    """Returns safe lat and lon limits for given cloudbox parameters and z_field.
    This can be needed for scattering simulations so that no incoming rays can enter
    the modelled atmosphere at a longitudinal or latitudinal face.
    Input cloud_box: dictionary with keys ['p1','p2','lat1','lat2','lon1','lon2']
    z_field: arts_types.GriddedField3 object
    Output: lat_min,lat_max,lon_min,lon_max"""

    r_max=r_geoid+ z_field.data.max()

    z_cloud=z_field.subset(p_range=(cloud_box['p1'],cloud_box['p2']),
                           lat_range=(cloud_box['lat1'],cloud_box['lat2']),
                           lon_range=(cloud_box['lon1'],cloud_box['lon2']))

    r_cbtop=r_geoid+az_cloud.data.max()

    alpha=(numpy.arccos(r_geoid/r_cbtop)+arccos(r_geoid/r_max))
    lat_safetymargin=alpha*180/numpy.pi

    lat_rad=abs(numpy.array([cloud_box["lat1"],cloud_box["lat2"]])).max()*pi/180
    lon_safetymargin=180/numpy.pi*numpy.arccos(numpy.sqrt(numpy.cos(alpha)**2-numpy.sin(lat_rad)**2)/numpy.cos(lat_rad))
    lat_min=numpy.floor(cloud_box["lat1"]-lat_safetymargin)
    lat_max=numpy.ceil(cloud_box["lat2"]+lat_safetymargin)
    lon_min=numpy.floor(cloud_box["lon1"]-lon_safetymargin)
    lon_max=numpy.ceil(cloud_box["lon2"]+lon_safetymargin)
    return lat_min,lat_max,lon_min,lon_max


def arts_batch_from_template(*args, **kwargs):
    raise general.PyARTSError("Support for arts_batch_from_template dropped " \
                      "per 2011-03-04 (blame Gerrit)")
#    text=open(template_file).read()
#    for field in argdict.keys():
#        text=text.replace('$PyARTS_INPUT_%s' % field, str(argdict[field]))
#    if '$PyARTS_INPUT' in text:
#        print "Error: Some template fields not set."
#        start_i=0
#        missing=[]
#        while text.find('$PyARTS_INPUT',start_i)>=0:
#            field_start=text.find('$PyARTS_INPUT',start_i)
#            start_i=text.find('}',field_start)
#            missing.append(text[field_start:start_i])
#        print missing
#        raise PyARTSError("template fields not set.")
#    open(arts_file,'w').write(text)
