#!/usr/bin/env python
"""Various functions dealing with 3D spherical geometry."""

import numpy as np
import general

from .constants import EARTH_RADIUS, DEG2RAD

def rot_mat_y(beta):
    """returns the rotation matrix for a rotation of cartesion coordinates
    about the y axis by angle beta (degrees)"""
    R=c_[len(beta)*[np.identity(3,float)]]
    beta*=DEG2RAD
    c=np.cos(beta)
    s=np.sin(beta)
    R[:,0,0]=c
    R[:,2,2]=c
    R[:,0,2]=-s
    R[:,2,0]=s
    return R

def sph2cart(pos1):
    """returns the cartesian coordinates given a position as an array
    with columns: radius in m, 'lat', 'lon'.  The x axis correponds
    to lon=0.  Returns array([xyz]) """
    pos = np.atleast_2d(pos1.copy())
    pos[:,1:]*=DEG2RAD
    xyz=zeros(pos.shape,float)
    xyz[:,0]=pos[:,0]*np.cos(pos[:,1])*np.cos(pos[:,2])
    xyz[:,1]=pos[:,0]*np.cos(pos[:,1])*np.sin(pos[:,2])
    xyz[:,2]=pos[:,0]*np.sin(pos[:,1])
    return xys.squeeze()

def cart2sph(xyz1):
    """returns the (r,lat,lon) coordinates given a position as an array
    with columns: x, y, z.  The x axis correponds
    to lon=0.  Returns array([xyz]) """
    xyz=np.atleast_2d(np.copy(xyz1))
    pos=np.zeros(xyz.shape,float)
    pos[:,0] = np.sqrt(sum(xyz**2,axis=-1))
    pos[:,1] = np.arcsin( xyz[:,2] / pos[:,0] )/DEG2RAD
    pos[:,2] = np.arctan2(xyz[:,1] , xyz[:,0] )/DEG2RAD
    return pos.squeeze()

def two_points2LOS(sensor_pos, other_pos):
    """given sensor_pos and other_pos, which are arrays with columns
    (radius (m), 'lat', 'lon'), return the line of sight pointing
    at other_pos from sensor_pos.  This is returned as an array with
    columnss 'za' and 'aa'"""

    op = other_pos.copy()
    sp = sensor_pos.copy()

    assert((op.shape==sp.shape).all()),str((op.shape,sp.shape))

    #Rotate about z axis so that sensor_pos lies on the x-z plane (lon=0)
    op[:,2]-=sp[:,2]

    #put other_pos into cartesian coordinates
    xp1 = sph2cart(op)

    #rotate about new y axis so that sensor_pos lies on z axis
    R = rot_mat_y(90-sp[:,1])

    xp = op   #we are finished with op, just using its memory
    for i in range(3):
        xp[:,i]=(R[:,i,:]*xp1).sum(axis=-1)

    del R,xp1

    #azimuth is now given by the new xp and yp values
    #zenith angle given by the difference in the new z's and xp, yp

    los = np.zeros((sp.shape[0],2), np.float64)

    los[:,0] = np.arctan(np.sqrt(xp[:,0]**2+xp[:,1]**2)/(xp[:,2]-sp[:,0]))/DEG2RAD
    los[:,1] = np.arctan(-xp[:,1]/xp[:,0])/DEG2RAD

    #adjustments for limited range of arctan returns.
    los[:,0] = np.where(los[:,0]<0, los[:,0]+180, los[:,0])
    los[:,1] = np.select([(los[:,1]<0)*(xp[:,1]>0), (los[:,1]>0)*(xp[:,1]<0)],
                         [los[:,1]+180, los[:,1]-180],
                         default=los[:,1])

    return los

def earth_fov(r):
    """Calculates angular diameter of Earth at distance r[m] from Earth

    Note: this calculation is approximate and considers the Earth to be
    perfectly spherical and flat.

    IN

        r   number-like     distance in meters from Earth centre

    OUT

        angle in degrees, size of FOV
    """

    h = r - EARTH_RADIUS
    if h < 0:
        raise ValueError("Invalid radius %d, less than Earth radius %d" % \
            (r, EARTH_RADIUS))
    return 2 * np.arctan(EARTH_RADIUS/h)/DEG2RAD

def islimb(pos, ang):
    """Determines whether viewing geometry is limb or not.

    IN

        pos     number-like     distance in meters from Earth centre
        z_ang   number-like     zenith angle in degrees

    OUT

        boolean, true if Earth is not visible (e.g. limb geometry)
    """

    earth_diam = earth_fov(pos)
    nad_ang = abs(ang - 180)
    return nad_ang > earth_diam/2

def sphdist(lat1, lon1, lat2, lon2):
    """Distance between points on sphere
    """

    # ported from Matlabs sphdist

    ga = (lon1-lon2) * DEG2RAD
    a = (90 - lat2) * DEG2RAD
    b = (90 - lat1) * DEG2RAD
    c = np.arccos(np.cos(a) * np.cos(b) +
                np.sin(a) * np.sin(b) * np.cos(ga))
    return EARTH_RADIUS * c

