#!/usr/bin/env python

"""Describe batches of data: atmospheric db, scattering db, etc.

This module contains a class with functionality to read atmospheric data
from a variety of formats and write it to a variety of formats.
"""

from __future__ import (print_function, unicode_literals)

# $Id: db.py 8140 2013-01-24 15:47:30Z gerrit $

import numpy
import datetime
now = datetime.datetime.now

class AtmosphericDatabase:

    props = {"data", "instrument"}

    def __init__(self, **kwargs):
        for k in self.props:
            setattr(self, k, None)

        for (k, v) in kwargs.iteritems():
            if not k in self.props:
                raise AttributeError("Invalid attribute: {},"
                    " not in {}".format(k, self.props))
            setattr(self, k, v)

    # class-methods for reading

    @classmethod
    def from_evans(cls, dbfile, instrument=None):
        """Read from evans-style training database
        """

        with open(dbfile, 'r') as fp:

            # skip and interpret header
            fp.readline() # header line
            ncases = int(fp.readline().strip().split()[0])
            fp.readline() # no. of dimensions of independent Gaussian prior space
            fp.readline() # no. of elements in observation vector
            nchans = int(fp.readline().strip().split()[0])
            chan_names = fp.readline().split()
#            print(now(), "compare:", self.instrument.channel_string(), chan_names)
            fp.readline() # no. of elements per channel
            fp.readline() # no. of viewing angles
            fp.readline() # cos for each viewing angle
            fp.readline() # random seed
            fp.readline() # no. retrieved quantities
            fp.readline() # no. integrated cloud params
            fp.readline() # no. retrieved humidity levels
            fp.readline() # height per humidity layer
            fp.readline() # no. retrieved ice cloud layers
            fp.readline() # bottom and top height ice cloud region
            fp.readline() # header line

            # pre-allocate
            n_bt = nchans - sum(["radar" in x.lower() for x in chan_names])
            Z = numpy.zeros(ncases, dtype=[("IWP", numpy.float32),
                                           ("BT", numpy.float32, (n_bt,))])

            # use while, not for-loop, because I read several lines per
            # iteration
            i = 0
            while True:
                line = fp.readline()
                if not line: # empty means EOF
                    break
                (rvcheck, IWP, Dme, Zmed,
                 shape1, shape2, shape3, shape4,
                 meltLWP, cldLWP) = (float(x) for x in line.strip().split())
                line = fp.readline()
                RH = numpy.array([float(x) for x in line.strip().split()])
                line = fp.readline()
                IWC = numpy.array([float(x) for x in line.strip().split()])
                line = fp.readline()
                Dme = numpy.array([float(x) for x in line.strip().split()])
                line = fp.readline()
                measurements = [float(x) for x in line.strip().split()]
                BT = numpy.array(measurements[:n_bt])
        #        radar_integrated = numpy.array(measurements[9])
        #        radar_prof = numpy.array(measurements[10:])
                #print "measurements:", BT
                #print "IWP:", IWP
                #print "IWC:", IWC
                Z["IWP"][i] = IWP
                Z["BT"][i] = BT
                i += 1
                if i%(ncases//5) == 0:
                    print(now(), "done", "{}/{}".format(i, ncases))

        return cls(data=Z, instrument=instrument)

    # writing out the results in different ways

    def write_evans_obs(self, fn, sel=None):
        """Write as evans-obs file

        Input:
            fn    file to write to
            sel  selection (list of indices), or None for all (default)
        """

    # from iceprofret.txt in the Evans distribution:
    #
    # Six header lines:
    #    Number of pixels
    #    Number of channels, total number of elements
    #    Channel IDs or names (matches channel IDs in retrieval database)
    #    number of elements in each channel
    #    additive uncertainties for each element
    #    multiplicative uncertainties for each element
    # For each line:
    #    Time(hours)  Cosine viewing zenith angle   Measurement for each element


        if sel is None:
            sel = numpy.arange(self.data.size)

        with open(fn, 'wt') as fp:
            # number of pixels
            fp.write("{}\n".format(sel.size))
            # no channels / no elements; both equal to size of BT
            fp.write("{} {}\n".format(self.data.dtype["BT"].shape[0],
                                      self.data.dtype["BT"].shape[0]))
            # names of channels
            fp.write("{}\n".format(self.instrument.channel_string(pre="im", width=True)))

            # no. element per channel
            fp.write(("1 " * len(self.instrument.channels)).strip())
            fp.write("\n")

            # additive uncertainties
            for chan in self.instrument.channels:
                fp.write(str(chan.noise) + " ")
            fp.write("\n")

            # multiplicative uncertainties (not implemented)
            for chan in self.instrument.channels:
                fp.write("0 ")
            fp.write("\n")

            for elem in sel:
                # time; n/a, so write fake
                fp.write("0.0 ")
                # cos(angle); only nadir implented
                fp.write("1.0 ")
                # write channel BT's
                for chan_bt in self.data[elem]["BT"]:
                    fp.write(str(chan_bt) + " ")
                fp.write("\n")
