#!/usr/bin/env python

"""Higher level functions to help getting started more quickly

"""

# $Id: helpers.py 7200 2011-11-10 13:33:05Z gerrit $

import os
import os.path
import glob
import cPickle
import datetime
now = datetime.datetime.now

import numpy
import ConfigParser

from . import artsXML

from .clouds import Droplet
from .io import (get_chevalier_file, get_scattering_file, get_garand_file,
                 get_nicam_file,
                 read_chevalier, read_chevalier_orig,
                 get_chevalier_orig_file, get_f_grid)
from .general import (get_fascod_for_chevalier_hirs, PyARTSError,
                      get_config, get_fascod_for_nicam_ir)
from .constants import sensor_los_ranges, channels, all_relevant_cheval

simulations_subfile = "%(sensor)s_cloudy%(cloudy)d_mode%(mode)s_photons%(photons)d_shape%(shape)s%(suffix)s"
lookup_subfile = "%(sat)s_%(sensor)s%(channel)s_%(mode)s.xml.gz"

def get_simul_dir(cloudy, mode, photons, shape, sensor, suffix=""):
    """Path to simulation dir for settings
    """
    base = get_config("simulations")
    p = simulations_subfile % vars()
    return os.path.join(base, p)


def get_lookup_table(sat, sensor, mode, channel):
    """Path to lookup table for settings:

    sat : string, e.g. noaa18, noaa19
    sensor : string, e.g. hirs, avhrr
    mode : string e.g. fast, reference
    channel : number, e.g. 11
    """
    base = get_config("lookup")
    p = lookup_subfile % vars()
    return os.path.join(base, p)

def defaults_batch(cloudy, mode, photons,
                   sat="noaa19",
                   scatter="yang_2005",
                   shape="SOL",
                   sensor="hirs",
                   channel=None,
                   atmosphere="cheval",
                   batch_write="/tmp/batch_output",
                   force_lookup=False,
                   include_liquid=True,
                   **kwargs):
    """Returns a dict with sensible defaults for batch-calculations

    This function was originally developed for HIRS MonteCarlo Chevalier.

    Inputs
    ~~~~~~

    cloudy : boolean. If true, set up MonteCarlo
    mode : string, "reference" or "fast"
    photons : number, how many photons per MPBS

    Optional inputs
    ~~~~~~~~~~~~~~~

    scatter : string: if cloudy, scattering database to use; defaults to yang_2005
    shape : string: if cloudy, particle shape to use; defaults to "SOL"
    sensor : string: defaults to "hirs"
    channel : number, defaults to 11
    atmosphere : string, "cheval" or "garand"; if cloudy, must be "cheval"
    batch_write : string, defaults to "/tmp/batch_output"
        if not None, write output on each batch-run

    Other inputs
    ~~~~~~~~~~~~

    All other inputs are directly put into the dictionary at the end.
    """

    if channel is None:
        channel = 11 if sensor=="hirs" else 5
    basis = {}
    basis["stokes_dim"] = 1
    basis["dim"] = 1
    basis["limb"] = False
    if batch_write is not None:
        basis["batch_write"] = batch_write
    basis["surface_reflectivity"] = 0.2

    # cloudy or clearsky
    if atmosphere == "cheval":
        basis["cheval_file"] = get_chevalier_file("all" if cloudy else "clear",
                                                  "ccol")
        basis["cheval_selection"] = numpy.r_[0:5000:100]

        basis["extra_fields"] = get_fascod_for_chevalier_hirs("tropical")
    elif atmosphere == "nicam":
        basis["cheval_file"] = get_nicam_file()
        basis["extra_fields"] = get_fascod_for_nicam_ir("tropical")
    elif atmosphere == "garand":
        if cloudy:
            raise PyARTSError("garand has no cloud info")
        basis["cheval_file"] = get_garand_file()
    basis["y_unit"] = "PlanckBT"

    if cloudy:
        ssd, smd = get_scattering_file(scatter, shape)
        basis["scat_data_raw"] = ssd
        basis["scat_data_meta_array"] = smd
        basis["max_iter"] = photons
        if include_liquid:
            drop = Droplet()
            drop.scat_calc(f_grid = get_f_grid(sat, sensor, channel, mode))
            # this one SSD object actually represents multiple particle
            # sizes
            basis["scat_data_raw_extra"] = drop.aossd
            basis["scat_data_meta_extra"] =  drop.aosmd

    # sensor
    basis["sensor"] = sensor + "_" + mode
    basis["abs_from_sensor"] = True
    basis["satellite"] = sat.upper()
    basis["channels"] = numpy.array([channels[sensor][channel]])
    basis["views"] = numpy.array([sensor_los_ranges[sensor][-1]]) # only nadir
    lookup_table = get_lookup_table(sat, sensor, mode, channel)
    if os.path.exists(lookup_table) and not force_lookup:
        basis["abs_lookup"] = lookup_table
        basis["make_lookup"] = False
    else:
        basis["abs_lines"] = get_config("hitran2004")
        basis["abs_lines_format"] = "Hitran2004"
        basis["make_lookup"] = True
        basis["write_lookup"] = lookup_table

    basis.update(kwargs)
    return basis

def generate_simulation_filename(cloudy, mode, photons, shape="SOL",
                                 sensor="hirs", suffix=""):
    d = get_simul_dir(cloudy, mode, photons, shape, sensor, suffix)
    if not os.path.exists(d):
        os.makedirs(d)
    for i in xrange(100):
        f = os.path.join(d, "%s_run%02d.arts" % (simulations_subfile, i))
        f = f % vars()
        if not os.path.exists(f):
            break
    else:
        raise RuntimeError("100 runs already?")
    return f

def get_chev_selections_segments(lo, hi, step):
    """Returns selections and associated strings
    """
    selections = [numpy.r_[x:x+step] for x in xrange(lo, hi, step)]
    suf = ["%d:%d:%d" % (sel.min(), sel.max()+1, numpy.diff(sel).mean()) for sel in selections]
    return (selections, suf)

def get_chev_selections_tropical_all_suffix(types):
    return "_" + "_".join(all_relevant_cheval) + "_tropical"
    

def get_chev_selections_tropical_all():
    """Returns Chev. selections
    """
    try:
        cache_dir = get_config("cache")
        cache_file = os.path.join(cache_dir, "cheval_selections.dat")
        with open(cache_file, 'r') as fp:
            return cPickle.load(fp)
    except (IOError, EOFError):
        pass # reread
    chev_files = []
    chev_selections = {}
    for chev in all_relevant_cheval:
        chev_arts = get_chevalier_file("all", chev)
        chev_files.append(chev_arts)
        #print now(), "figuring out tropicals for", chev
        orig = read_chevalier_orig(get_chevalier_orig_file(chev))
        chev_selections[chev] = (numpy.abs(orig["lat"]) < 30).nonzero()[0]

    with open(cache_file, 'w') as fp:
        cPickle.dump(chev_selections, fp)

    return chev_selections

# WARNING: maybe a bug if passed in the wrong order...
def read_chev_selections_tropical_all(types=all_relevant_cheval):
    """Reads chev selections, tropical for all types.

    If retsels=True, also return selections made (can be useful to select
    subset of simulated BT's)
    """
    sels = get_chev_selections_tropical_all()

    allsubsets = []
    for t in types:
        chev = read_chevalier(t)
        sub = chev[sels[t]]
        allsubsets.append(sub)
    return numpy.concatenate(allsubsets)


def get_result_file(cloudy, mode, photons, shape, sensor, suffix="", no=-1):
    """Get path to result file; no indicates number, -1 is latest
    """
    d = get_simul_dir(cloudy, mode, photons, shape, sensor, suffix)
    sub = simulations_subfile % vars()
#    gstr = "%s*.%s" % (os.path.join(d, sub), ext)
#    # FIXME: ugly hack to account for prior bug in
#    # generate_simulation_filename
#    res = glob.glob(gstr)
#    if len(res) >= 1:
#        return sorted(res)[-1]
#    else:
#        raise PyARTSError("No result file found!")
    f = os.path.join(d, sub) + "_run%02d.ybatch.xml.gz"
    if no == -1:
        no = 0
        while os.path.exists((f%no).split(".")[0]+".arts"):
            no += 1
        no -= 1
    if not os.path.exists(f%no):
        raise PyARTSError("File not found: %s" % (f%no))
    return f%no

def read_result_file_single(cloudy, mode, photons, shape, sensor, suffix, no):
    """Gets (result, selection) for settings
    """
    f = get_result_file(cloudy, mode, photons, shape, sensor, suffix, no)
    pickfile = f.split(".")[0] + ".pickle"
    with open(pickfile) as fp:
        d = cPickle.load(fp)
    print now(), "reading", f
    res = artsXML.load(f).squeeze()
    sel = d["cheval_selection"]
    return (res, sel)

def read_result_file_multi(cloudy, mode, photons, shape, sensor, suffix, n, no=-1):
    """Returns all results and corresponding selection.

    'n' is either int (highest no.) or tuple (passed to xrange).
    
    Suffix contains %d
    """
    res = []
    allsels = []
    try:
        r = xrange(n)
    except TypeError:
        r = xrange(*n)

    for i in r:
        this_res, this_sel = read_result_file_single(
            cloudy, mode, photons, shape, sensor, suffix%i, no)
        res.append(this_res)
        allsels.append(this_sel)
    return (numpy.concatenate(res), numpy.concatenate(allsels))

