#!/usr/bin/env python3.2

"""Classes to represent radiometers
"""

# $Id: instruments.py 8140 2013-01-24 15:47:30Z gerrit $

import errno
import datetime
now = datetime.datetime.now

import os.path

import numpy
import decimal

from . import physics

class Channel(object):
    """Represents a (rectangular) radiometer channel.

    The centre frequency is a Decimal.

    Freq, noise, width, sideband (default=0)
    """

    def __init__(self, freq, noise, width, sideband=0):
        self.centre_frequency = decimal.Decimal(freq)
        self.noise = noise
        self.width = decimal.Decimal(width)
        self.sideband = decimal.Decimal(sideband)
        self.shape = "rectangular"

    def srf(self, step=99e6):
        """Returns SRF (x, y) with stepsize `step`
        """
        if self.sideband == 0:
            centres = [self.centre_frequency]
        else:
            centres = [self.centre_frequency-self.sideband,
                       self.centre_frequency+self.sideband]
        if self.shape == "rectangular":
            edges = [(float(c-self.width), float(c+self.width)) for c in centres]
            pp = [numpy.arange(e[0], e[1]+step/2, step) for e in edges]
            f_grid = numpy.concatenate(pp)
            y = numpy.ones(f_grid.size)/f_grid.size
            return (f_grid, y)
        else:
            raise NotImplementedError("Unknown shape: {}".format(rectangular))

    def write_srf(self, to, quantity="frequency", factor=1):
        """Write SRF to file.

        To can be:
            - open file
            - filename
            - directory (fn will be auto-calculated)

        Quantity can be:
            - frequency (default)
            - wavelength
            - wavenumber

        Factor is a multiplication factor, e.g. to do wavenumber in cm^-1
        instead of m^-1, use factor=0.01
        """
        try:
            to.write
        except AttributeError: # not an open file
            try:
                to = open(to, 'wb')
            except IOError as exc:
                if exc.errno == errno.EISDIR:
                    to = open(os.path.join(to, "specresp_" +
                                               self.get_chanstr(full=True) +
                                               ".dat"),
                              'wb')
                else:
                    raise
                
        (x, y) = self.srf()
        if quantity == "frequency":
            pass
        elif quantity == "wavelength":
            x = physics.frequency2wavelength(x)
        elif quantity == "wavenumber":
            x = physics.frequency2wavenumber(x)
        else:
            raise ValueError("Unknown quantity: {}".format(quantity))

        print now(), "Writing to", to.name
        numpy.savetxt(to, numpy.c_[x*factor, y])


            
    def get_chanstr(self, full=False):
        """Get the shortest string representation of frequency in GHz
        """

        # if whole number, strip off decimal part
        # see also http://stackoverflow.com/q/11227620/974555
        chanstr = str(self.centre_frequency/decimal.Decimal("1e9"))
        if '.' in chanstr:
            chanstr = chanstr.rstrip('0').rstrip('.')

        if full:
            widthstr = "-" + str(self.width/decimal.Decimal("1e9"))
            if self.sideband == 0:
                sbstr = ""
            else:
                sbstr = u"±" + str(self.sideband/decimal.Decimal("1e9"))
            return chanstr + sbstr + widthstr
        else:
            return chanstr

    def __repr__(self):
        return "<Channel:{} GHz>".format(self.get_chanstr())

class Radiometer(object):
    """Represents a radiometer.

    Initialise with list of Channel-objects.
    """

    name = None

    def __init__(self, channels):
        self.channels = channels

    def channel_string(self, pre="", full=False):
        """Get string with channel descriptions.

        Channel descriptions are compact string specifications of channel
        frequencies in GHz. Those are concatenated together separated by
        whitespace.
        """

        return " ".join(pre+c.get_chanstr(full=full) for c in self.channels)

    def write_all_srf(self, d, *args):
        """Write all SRF to directory d.

        Remaining arguments passed on to Channel.write_srf
        """

        for chan in self.channels:
            chan.write_srf(d, *args)

    def __repr__(self):
        if self.name is not None:
            return "<Radiometer {} {} chans>".format(
                        self.name, len(self.channels))
        else:
            return "<Radiometer {} chans>".format(len(self.channels))

# FIXME: read from some external definition file

# icemusic all single band
icemusic = Radiometer([
    Channel("89E9", 0.1, "10E9"), # string to keep precision for Decimal
    Channel("150E9", 0.1, "10E9"),
    Channel("183.13E9", 0.1, "1.8E9"),
    Channel("186.31E9", 0.1, "1.8E9"),
    Channel("190.31E9", 0.1, "1.8E9"),
    Channel("205E9", 0.1, "10E9"),
    Channel("312.5E9", 0.1, "10E9"),
    Channel("321.5E9", 0.1, "3.2E9"),
    Channel("505E9", 0.1, "10E9"),
    Channel("530E9", 0.1, "10E9"),
    Channel("875E9", 0.1, "20E9"),
    Channel("1500E9", 0.1, "20E9")])

# source buehler07:_ciwsir_qjrms
ciwsir = Radiometer([
    Channel("183.31e9", 0.6, "1.4e9", "1.5e9"),
    Channel("183.31e9", 0.5, "2.0e9", "3.5e9"),
    Channel("183.31e9", 0.4, "3.0e9", "7.0e9"),
    Channel("243.2e9", 0.5, "3.0e9", "0.5e9"),
    Channel("325.15e9", 1.0, "1.6e9", "1.5e9"),
    Channel("325.15e9", 0.8, "2.4e9", "3.5e9"),
    Channel("325.15e9", 0.7, "3.0e9", "9.5e9"),
    Channel("448.0e9", 1.9, "1.2e9", "1.4e9"),
    Channel("448.0e9", 1.4, "2.0e9", "3.0e9"),
    Channel("448.0e9", 1.2, "3.0e9", "7.2e9"),
    Channel("664.0e9", 1.5, "5.0e9", "4.2e9")])
