#!/usr/bin/env python
"""Some physical constants, Rayleigh Jeans brightness temperature function,
and psychrometric functions."""

# -*- coding: utf-8

import numpy
import scipy
from .constants import (h, k, c, R_d, R_v)


def rjbt(radiance,frequency):
    """Returns the Rayleigh Jeans brightness temperature for a given
    radiances and frequency(Hz)"""
    return radiance*c**2/(2*frequency**2*k)

def planckbt(radiance, frequency):
    """Returns the Planck brightness temperature for a given radiance and
    frequency (Hz)
    """

    #planck = constants('PLANCK_CONST');
    #boltzmann = constants('BOLTZMANN_CONST');
    #speed_light = constants('SPEED_OF_LIGHT');

    a = h/k
    b = 2*h/(c**2)

    return a * frequency / numpy.log((b*frequency**3)/radiance + 1 );


def vapour_P(T):
    """ Pvs = vapour_P(T) Calculates the saturated vapour pressure (Pa)
    of water using Hyland-Wexler eqns (ASHRAE Handbook) T in Kelvin!"""

    #%%%%%%%%%%%%%%%%%%%%%%%%%%
    #%CONSTANTS
    A = -5.8002206e3
    B = 1.3914993
    C = -4.8640239e-2
    D = 4.1764768e-5
    E = -1.4452093e-8
    F = 6.5459673
    #%%%%%%%%%%%%%%%;%%%%%%%%%%%


    Pvs = numpy.exp(A/T + B + C*T + D*T**2 + E*T**3 + F*numpy.log(T))
    return Pvs

def vapour_P_ice(T):
    """ Pvs = vapour_P_ice(T) Calculates the saturated vapour pressure (Pa)
    of water using the relation of List (1951). See JPL Clouds ATBD for
    reference."""
    Pvs=10**(-9.097*(273.16/T-1)-3.5665*numpy.log10(273.16/T)+0.8768*(1-T/273.16)+2.786)
    return Pvs

def psychrometer(tdb,twb,P=1.01325e5):
    """returns the relative humidity (wrt liquid water, between 0 and 1) given
    the dry bulb and wet-bulb temperature (in K)"""
    P = 1.01325e5;
    Wsdash = 0.62198*vapour_P(twb)/(P-vapour_P(twb));
    W = ((2501 - 2.381*(twb-273.15))*Wsdash - 1.006*(tdb-twb))/\
        (2502+1.805*(tdb-273.15)-4.186*(twb-273.15));
    Ws = 0.62198*vapour_P(tdb)/(P-vapour_P(tdb));
    mu = W/Ws;

    relhum = mu/(1-(1-mu)*(vapour_P(tdb)/P));
    return relhum

def wavelength2frequency(wavelength):
    """Converts wavelength (in meters) to frequency (in Hertz)
    """

    return c/wavelength

def wavenumber2frequency(wavenumber):
    """Converts wavenumber (in m^-1) to frequency (in Hz)
    """

    return c*wavenumber

def frequency2wavelength(frequency):
    return c/frequency

def frequency2wavenumber(frequency):
    return frequency/c

def mixingratio2density(mixingratio, p, T):
    """Converts mixing ratio (e.g. kg/kg) to density (kg/m^3) for dry air

    Needs pressure [Pa], temperature [K].
    """

    # from ideal gas law, mass of 1 m³ of air:
    #
    # ρ = p/(R*T)

    R_d = 287.0 # J K^-1 kg^-1
    m_air = p/(R_d*T)
    return mixingratio * m_air

def mixingratio2rh(w, p, T):
    """For water on Earth, convert mixing-ratio to relative humidity

    Inputs:

    w: water vapour mixing ratio [1]
    p: pressure [Pa]
    T: temperature [K]

    Returns:

    relative humidity [1]
    """

    eps = R_d/R_v # Wallace and Hobbs, 3.14
    e = w/(w+eps)*p # Wallace and Hobbs, 3.59
    e_s = vapour_P(T)
    return e/e_s # Wallace and Hobbs, 3.64

def specific2mixingratio(q):
    """Convert specific humidity [kg/kg] to volume mixing ratio
    """

    # Source: extract_arts_1.f90

    eps = R_d/R_v
    return q / ( q + eps*(1-q) )

def planck_wl(wl, T):
    """Calculate Planck's law for wavelength, temperature

    Returns spectral irradiance in W.m^-3.sr^-1
    """

    return 2*h*c**2/(wl**5*(numpy.exp((h*c)/(wl*k*T))-1))
