"""This is an example of using the ArtsRunCollection class to
do multiple runs with different parameters. We take a CIWSIR-like instrument, and a boxcloud with different values of IWC"""
from scipy import *
from PyARTS import *
from pylab import *
import tempfile
import os
import types


IWClist=[0.0,0.01,0.02,0.04,0.06,0.08,0.1]
freq=334.65e9


#First generate the cloud stuff
#here are the cloud dimensions
lat1=-2.0
lat2=2.0
lon1=-2.0
lon2=2.0
ztopkm=14.0
zbottomkm=13.0
#and the number of grid points in the cloudbox
cb_size={'np':10,'nlat':10,'nlon':10} 

#location of the temperature and altitude profiles
zfile='../data/tropical.z.xml'
tfile='../data/tropical.t.xml'

#generate a cloud object with IWC = 0.01
a_cloud=clouds.boxcloud(ztopkm,zbottomkm,lat1,lat2,
                        lon1,lon2,cb_size,zfile,
                        tfile,IWC=0.01)

#consider a gamma distribution of spherical particles, with an
#effective radius of 50 microns
gamma_dist=clouds.Gamma(r_eff=50,#the effective radius
                        g=2, #the gamma dist. shape parameter
                        T_grid=[215,272.15],
                        npoints=5,#number of gauss-laguerre quadrature points
                        ptype=20,#random orientation
                        NP=-1, #spheroidal shape
                        aspect_ratio=1.000001 # a sphere!
                        )

a_cloud.addHydrometeor(gamma_dist)
a_cloud.pnd_field_gen('pnd_field_raw.xml')
artsXML.save(a_cloud.particle_masses(),'particle_masses.xml')


#generate scattering data files (using 2 processes)
scat_data_mono=clouds.scat_data_raw_calc([gamma_dist],[freq],num_proc=2)
scat_data_mono.save('scat_data_mono.xml')#there is currently a bug in arts
#that will not allow SingleScatteringData objects to be loaded in binary


grids={}
#define the pressure latitude and longitude grids
for s in ['p','lat','lon']:
    #merge the finer cloudbox grids with the coarse grids for the
    #modelled atmosphere
    old_grid=artsXML.load('../data/%s_grid.xml' % s)
    if s=='p':
        #chop of levels with p<1333.52 so that it will work with 
        #the pre-supplied absoprtion lookup table
        keep_i=nonzero(old_grid>1333.52)
        #Annoying catch for newer numpy versions ...
        if type(keep_i)==types.TupleType:
            keep_i=keep_i[0]
        old_grid=take(old_grid,keep_i)
    grids[s]=eval('arts_math.gridmerge(old_grid,a_cloud.%s_grid[1:-2])' % s)
    artsXML.save(grids[s],'new_%s_grid.xml' %s)
    #if you are doing multiple calculations with big fields it makes
    #sense to convert them into the binary xml format.
    arts.xml_ascii_to_binary('new_%s_grid.xml' % s,
                             '%s_grid' % s,
                             'new_%s_grid.xml.bin' % s)

#calculate the pnd_field on the new grids
arts.pnd_fieldCalc(pnd_field_raw_file='pnd_field_raw.xml',
                   cloudbox=a_cloud.cloudbox,
                   p_file='new_p_grid.xml.bin',
                   lat_file='new_lat_grid.xml.bin',
                   lon_file='new_lon_grid.xml.bin',
                   pnd_field_file='pnd_field.xml',
                   cloudbox_limits_file='')

#load this field, which is for IWC=0.01, into python,
#because we are going to use it to generate the pnd fields for
#the higher IWCs
pnd_field_01=artsXML.load('pnd_field.xml')
pnd_files=[]
for IWC in IWClist:
    pnd_field=pnd_field_01*IWC/0.01
    ignore,ascii_file=tempfile.mkstemp(suffix='.xml')
    artsXML.save(pnd_field,ascii_file)
    pnd_files.append('pnd_field_%02d.xml.bin' % (IWC*100.0))
    arts.xml_ascii_to_binary(ascii_file,'pnd_field',
                             pnd_files[-1])
    os.remove(ascii_file)

#create 3D temperature, altitude and volume mixing ratio fields
#from 1-D profiles

params={'p_grid':'new_p_grid.xml.bin',
        'atm_basename':'../data/tropical',
        'abs_species':['H2O','N2','O2'],
        "dim": 1}

fields={}
fields['t'],fields['z'],fields['vmr']=arts.artsGetAtmFields(params)
#create binary files for each of these
for s in ['t','z','vmr']:
    ascii_file=tempfile.mkstemp(suffix='.xml')[1]
    #print fields[s].shape
    new_shape=list(fields[s].shape)
    new_shape[-2]=len(grids['lat'])
    new_shape[-1]=len(grids['lon'])
    fields[s]=reshape(outer(ravel(fields[s]),ones((new_shape[-2],new_shape[-1]))),
                      new_shape)
    artsXML.save(fields[s],ascii_file)
    arts.xml_ascii_to_binary(ascii_file,'%s_field' % s,
                             '%s_field.xml.bin' % s)
    os.remove(ascii_file)
    #set arts run parameters
    params['%s_field' % s]='%s_field.xml.bin' % s
    

#calculate a sensible CIWSIR viewing position that looks through
#the centre of the cloud

rg=arts_geometry.EARTH_RADIUS
va=53
zs=833e3
za=180-arcsin(rg*sin((90+va)*pi/180)/(rg+zs))*180/pi
lat=za-90-va
sensor_los=reshape(array((za,180)),(1,2))
sensor_pos=reshape(array((zs+rg,lat,0)),(1,3))
artsXML.save(sensor_los,'sensor_los.xml')
artsXML.save(sensor_pos,'sensor_pos.xml')

#antenna characteristics




#Now specify the rest of the parameters required for an arts RT simulation
#some have already been specified above!

params.update({
    "cloud_box":a_cloud.cloudbox,
    'lat_grid':'new_lat_grid.xml.bin',
    'lon_grid':'new_lon_grid.xml.bin',
    'atmosphere_dim':3,
    'freq':freq,
    "abs_lookup":"../data/ciwsir_gas_abs_lookup.xml.bin",
    "max_time":10, #only let rt calculations run for 10 seconds (inaccurate!!!)
    "std_err":0.5,
    "sensor_pos":'sensor_pos.xml',
    "sensor_los":'sensor_los.xml',
    "scat_data_mono":'scat_data_mono.xml',
    "stokes_dim":4,
    'z_field_is_1D':1,
    'particle_masses':'particle_masses.xml'
    })


#Create an ArtsRunCollection instance and include one run for every
#IWC value (the first will be a clear sky calculation)

# GH 2011-05-18: This class no longer exists (ARTS is multi-threaded)
#arts_runs=arts.ArtsRunCollection()
#for pf in pnd_files:
#    params['pnd_field']=pf
#    arts_runs.append(arts.ArtsRun(params,
#                                  'mcgeneral',
#                                  tempfile.mkstemp(suffix='.arts')[1]))
#
##lets also calculate the FOV integrated IWP in each case
##for this we need to specify the number of Monte Carlo iterations
#params['max_iter']=1000
#
#for pf in pnd_files:
#    params['pnd_field']=pf
#    arts_runs.append(arts.ArtsRun(params,
#                          run_type='iwp_tau',
#                          filename=tempfile.mkstemp(suffix='.arts')[1]))                      
#    assert arts_runs[-1].run_type=='iwp_tau'
#
##now perform all of the calculations across four cores        
#arts_runs.run(num_proc=4)
#
#
###Now collect all of the results
###this could be put into an ArtsRunCollection method e.g. collate_results
#y=[]
#y_error=[]
#for i in range(len(IWClist)):
#    y.append(arts_runs[i].output['y'])
#    y_error.append(arts_runs[i].output['mc_error'])
#
##calculate the cloud induced radiance and its error
#dI=array(y)[1:,0]-y[0][0]
#dIerror=sqrt(array(y_error)[1:,0]**2+y_error[0][0]**2)
#
#iwp=[]
###Collect iwp
#for i in range(len(IWClist),2*len(IWClist)):
#    iwp.append(arts_runs[i].output['mc_IWP'])
#
#iwp=array(iwp[1:])
#
###Now plot the results
#errorbar(x=iwp,y=dI,yerr=dIerror)
#title('arts_run_collection.py')
#xlabel('Ice Water Path')
#ylabel('$\Delta I$')
#savefig('dIvIWP.png')
#
