% ODIN_ANTENNA   Odin-SMR antenna response in ARTS format
%
%   The function returns the antenna response for given frequency and
%   integration time. The antenna pattern is 1D. The effect of the vertical
%   scanning during the integration is included. The response is scaled to
%   the given frequency. The data match ARTS' *antenna_response*.
%
%    If stored to file:
%      xmlStore( filename, R, 'GriddedField4' );
%
%   A matching *mblock_za_grid* is also returned. This grid is created in a
%   simple manner, and shall not be treated as "optimal" but should be OK
%   for most applications.
%
% OUT   R             The response, as a GriddedField4.
%       mblock_grid   Suggested mblock_za_grid.
% IN    f             Frequency, a scalar value.
%       tint          Integration time, 0.875, 1.875 or 3.875 s.
%       lbgrid        Length of *mblock_grid*.

% 2012-11-17 Patrick Eriksson


function [R,mblock_grid] = odin_antenna(f,tint,lbgrid)
%
rqre_datatype( f, {@istensor0} );
rqre_in_range( f, 116e9, 585e9 );
if f>121e9 & f<470e9 
  error( 'Argument *f* must be inside [116,121] or [470 585] GHz.' );  
end
rqre_datatype( tint, {@istensor0} );
rqre_in_range( tint, 0.8, 4 );
rqre_alltypes( lbgrid, {@istensor0,@iswhole} );
rqre_in_range( lbgrid, 5 );


% Obtain raw "lab data"
%
[f535,R535,R119] = lab_data;
%  
if f < 300e9
  f0 = 119e9;
  za = R119(:,1);
  r  = R119(:,2);
else
  [u,i] = min( abs( f-f535 ) );
  f0     = f535(i);
  za     = R535(:,1); 
  r      = R535(:,i+1); 
end
  

% Scale to selected frequency
%
za = (f0/f) * za;


% Include scanning
%
[za,r] = broaden_apattern( za, r, tint );


% Set up gridded
R.name      = 'Antenna response function';
R.gridnames = { 'Polarisation', 'Frequency', 'Zenith angle', 'Azimuth angle' };
R.grids     = { {'1'}, f, za, 0 };
R.dataname  = 'Response';
R.data(1,1,:,1) = r;


% Create mblock_grid
%
c           = cumsum( r.^0.3 );  % r.^0.3 found purely emperical, to give
c           = c - c(1);          % a result that looks reasonable
c           = c / c(end);
%
mblock_grid = interp1( c, za, linspace( 0, 1, lbgrid ) );

return


function [za,r] = broaden_apattern(za,r,t)
  
  %- Scanning speed is 750 m/s. Convert to an approximate angle.
  %
  R  = constants( 'EARTH_RADIUS' );
  zp = 600e3;
  %
  v = geomztan2za( R, zp, 20e3 ) - geomztan2za( R, zp, 20e3+750 ); 
  
  %- Select n "middle points" over relative zenith angle scanning range
  %
  n = 11;
  %
  dza = (v*t/2) * linspace( -1+1/n, 1-1/n, n ); 
  
  %- Include scanning by simply adding antenna pattern for each dza
  %
  p = 0;
  %
  for i = 1:n
    p = p + interp1( za+dza(i), r, za );
  end

  %- The interpolation generates some NaNs
  %- Normalise to max value
  %
  ind = find( ~isnan(p) );
  %
  za = za(ind);
  r  = p(ind)/max(p(ind));  
return


function [f535,R535,R119] = lab_data
  
f535 = 1e9*[490 540 580]';

R119=[
-1.000000 1.369193e-004
-0.987500 1.332597e-004
-0.975000 1.303986e-004
-0.962500 1.307330e-004
-0.950000 1.363269e-004
-0.937500 1.477263e-004
-0.925000 1.634959e-004
-0.912500 1.805233e-004
-0.900000 1.949541e-004
-0.887500 2.035011e-004
-0.875000 2.046731e-004
-0.862500 1.995728e-004
-0.850000 1.914477e-004
-0.837500 1.849256e-004
-0.825000 1.840386e-004
-0.812500 1.909653e-004
-0.800000 2.047295e-004
-0.787500 2.214491e-004
-0.775000 2.355676e-004
-0.762500 2.416907e-004
-0.750000 2.368281e-004
-0.737500 2.221003e-004
-0.725000 2.027421e-004
-0.712500 1.868798e-004
-0.700000 1.831784e-004
-0.687500 1.976395e-004
-0.675000 2.313102e-004
-0.662500 2.793987e-004
-0.650000 3.319757e-004
-0.637500 3.773412e-004
-0.625000 4.056365e-004
-0.612500 4.125187e-004
-0.600000 4.010189e-004
-0.587500 3.812764e-004
-0.575000 3.671078e-004
-0.562500 3.709316e-004
-0.550000 3.992695e-004
-0.537500 4.486943e-004
-0.525000 5.061456e-004
-0.512500 5.527575e-004
-0.500000 5.699922e-004
-0.487500 5.481984e-004
-0.475000 4.919138e-004
-0.462500 4.215455e-004
-0.450000 3.695812e-004
-0.437500 3.712480e-004
-0.425000 4.521556e-004
-0.412500 6.175114e-004
-0.400000 8.460028e-004
-0.387500 1.093371e-003
-0.375000 1.302362e-003
-0.362500 1.422141e-003
-0.350000 1.427117e-003
-0.337500 1.328077e-003
-0.325000 1.173384e-003
-0.312500 1.033769e-003
-0.300000 9.739302e-004
-0.287500 1.021905e-003
-0.275000 1.146611e-003
-0.262500 1.265374e-003
-0.250000 1.287027e-003
-0.237500 1.204771e-003
-0.225000 1.226309e-003
-0.212500 1.922853e-003
-0.200000 4.364389e-003
-0.187500 1.020128e-002
-0.175000 2.165331e-002
-0.162500 4.135176e-002
-0.150000 7.207190e-002
-0.137500 1.164337e-001
-0.125000 1.760968e-001
-0.112500 2.518202e-001
-0.100000 3.425569e-001
-0.087500 4.455588e-001
-0.075000 5.560953e-001
-0.062500 6.680553e-001
-0.050000 7.742382e-001
-0.037500 8.670787e-001
-0.025000 9.389571e-001
-0.012500 9.844606e-001
0.000000 1.000000e+000  
0.012500 9.831819e-001  
0.025000 9.367128e-001  
0.037500 8.637301e-001  
0.050000 7.700745e-001  
0.062500 6.627030e-001  
0.075000 5.498918e-001  
0.087500 4.387254e-001  
0.100000 3.356056e-001  
0.112500 2.449186e-001  
0.125000 1.695518e-001  
0.137500 1.105303e-001  
0.150000 6.710448e-002  
0.162500 3.738385e-002  
0.175000 1.873781e-002  
0.187500 8.302588e-003  
0.200000 3.342861e-003  
0.212500 1.595814e-003  
0.225000 1.379759e-003  
0.237500 1.627289e-003  
0.250000 1.794715e-003  
0.262500 1.721534e-003  
0.275000 1.467457e-003  
0.287500 1.177520e-003  
0.300000 9.778339e-004  
0.312500 9.273029e-004  
0.325000 1.010691e-003  
0.337500 1.161625e-003  
0.350000 1.297305e-003  
0.362500 1.351893e-003  
0.375000 1.295963e-003  
0.387500 1.140750e-003  
0.400000 9.278217e-004  
0.412500 7.106982e-004  
0.425000 5.359477e-004  
0.437500 4.304928e-004  
0.450000 3.963455e-004  
0.462500 4.148693e-004  
0.475000 4.567677e-004  
0.487500 4.933954e-004  
0.500000 5.060061e-004  
0.512500 4.902084e-004  
0.525000 4.547418e-004  
0.537500 4.157299e-004  
0.550000 3.896316e-004  
0.562500 3.865056e-004  
0.575000 4.071936e-004  
0.587500 4.426009e-004  
0.600000 4.782520e-004  
0.612500 4.992919e-004  
0.625000 4.956072e-004  
0.637500 4.646287e-004  
0.650000 4.124231e-004  
0.662500 3.508994e-004  
0.675000 2.935150e-004  
0.687500 2.514370e-004  
0.700000 2.300357e-004  
0.712500 2.278233e-004  
0.725000 2.377562e-004  
0.737500 2.498892e-004  
0.750000 2.550705e-004  
0.762500 2.478475e-004  
0.775000 2.278661e-004  
0.787500 1.997298e-004  
0.800000 1.710054e-004  
0.812500 1.494429e-004  
0.825000 1.405051e-004  
0.837500 1.455985e-004  
0.850000 1.618512e-004  
0.862500 1.831405e-004  
0.875000 2.023274e-004  
0.887500 2.133116e-004  
0.900000 2.129236e-004  
0.912500 2.014692e-004  
0.925000 1.825236e-004  
0.937500 1.612812e-004  
0.950000 1.429993e-004  
0.962500 1.313900e-004  
0.975000 1.278981e-004  
0.987500 1.315365e-004  
1.000000 1.398636e-004  
];

R535=[
-0.200000 4.258031e-004 3.097056e-004  3.553543e-004
-0.197500 4.390343e-004 3.314790e-004  3.402546e-004
-0.195000 4.477900e-004 3.579949e-004  3.280175e-004
-0.192500 4.503376e-004 3.869272e-004  3.217517e-004
-0.190000 4.447494e-004 4.164781e-004  3.238953e-004
-0.187500 4.294888e-004 4.448169e-004  3.356782e-004
-0.185000 4.037498e-004 4.707415e-004  3.565206e-004
-0.182500 3.683268e-004 4.932235e-004  3.846667e-004
-0.180000 3.257208e-004 5.106157e-004  4.173105e-004
-0.177500 2.799466e-004 5.212989e-004  4.517443e-004
-0.175000 2.361238e-004 5.231069e-004  4.852793e-004
-0.172500 2.000600e-004 5.136209e-004  5.160157e-004
-0.170000 1.770156e-004 4.910389e-004  5.424219e-004
-0.167500 1.707056e-004 4.550771e-004  5.620833e-004
-0.165000 1.828604e-004 4.073554e-004  5.732849e-004
-0.162500 2.123507e-004 3.520314e-004  5.728577e-004
-0.160000 2.558481e-004 2.954167e-004  5.582502e-004
-0.157500 3.081415e-004 2.452006e-004  5.277618e-004
-0.155000 3.635587e-004 2.091096e-004  4.818100e-004
-0.152500 4.169964e-004 1.932925e-004  4.236396e-004
-0.150000 4.653283e-004 2.009485e-004  3.597455e-004
-0.147500 5.079114e-004 2.317118e-004  2.990939e-004
-0.145000 5.470137e-004 2.814853e-004  2.516779e-004
-0.142500 5.867420e-004 3.436990e-004  2.261059e-004
-0.140000 6.319640e-004 4.108478e-004  2.277994e-004
-0.137500 6.866334e-004 4.767668e-004  2.574577e-004
-0.135000 7.520564e-004 5.382210e-004  3.109478e-004
-0.132500 8.253502e-004 5.955723e-004  3.806677e-004
-0.130000 8.998048e-004 6.527462e-004  4.577419e-004
-0.127500 9.649649e-004 7.151128e-004  5.346517e-004
-0.125000 1.009221e-003 7.873655e-004  6.080920e-004
-0.122500 1.022368e-003 8.712254e-004  6.790438e-004
-0.120000 9.977739e-004 9.619473e-004  7.524683e-004
-0.117500 9.361207e-004 1.049873e-003  8.343577e-004
-0.115000 8.465898e-004 1.120394e-003  9.282608e-004
-0.112500 7.467585e-004 1.157831e-003  1.031004e-003
-0.110000 6.600285e-004 1.150280e-003  1.132573e-003
-0.107500 6.119860e-004 1.093837e-003  1.216388e-003
-0.105000 6.260366e-004 9.968763e-004  1.262472e-003
-0.102500 7.172951e-004 8.800701e-004  1.256266e-003
-0.100000 8.885699e-004 7.741951e-004  1.192268e-003
-0.097500 1.129382e-003 7.149084e-004  1.081168e-003
-0.095000 1.414360e-003 7.340043e-004  9.509327e-004
-0.092500 1.709420e-003 8.518337e-004  8.420094e-004
-0.090000 1.975497e-003 1.070128e-003  7.990526e-004
-0.087500 2.178668e-003 1.368433e-003  8.595089e-004
-0.085000 2.295331e-003 1.707825e-003  1.041289e-003
-0.082500 2.319957e-003 2.037653e-003  1.335098e-003
-0.080000 2.264470e-003 2.307125e-003  1.702340e-003
-0.077500 2.158425e-003 2.477363e-003  2.085469e-003
-0.075000 2.038369e-003 2.533324e-003  2.417387e-003
-0.072500 1.942483e-003 2.486931e-003  2.643319e-003
-0.070000 1.895415e-003 2.374605e-003  2.736309e-003
-0.067500 1.906290e-003 2.246754e-003  2.703860e-003
-0.065000 1.966933e-003 2.153078e-003  2.590078e-003
-0.062500 2.064876e-003 2.124428e-003  2.456649e-003
-0.060000 2.207118e-003 2.168751e-003  2.364449e-003
-0.057500 2.457378e-003 2.276156e-003  2.352175e-003
-0.055000 2.986195e-003 2.449455e-003  2.427571e-003
-0.052500 4.119928e-003 2.756186e-003  2.587683e-003
-0.050000 6.392081e-003 3.398511e-003  2.867889e-003
-0.047500 1.057824e-002 4.796589e-003  3.427289e-003
-0.045000 1.769566e-002 7.669481e-003  4.660447e-003
-0.042500 2.899176e-002 1.308556e-002  7.311826e-003
-0.040000 4.589044e-002 2.247129e-002  1.256866e-002
-0.037500 6.988120e-002 3.757062e-002  2.208622e-002
-0.035000 1.023499e-001 6.030569e-002  3.796797e-002
-0.032500 1.444581e-001 9.262691e-002  6.258979e-002
-0.030000 1.969669e-001 1.362032e-001  9.827344e-002
-0.027500 2.599500e-001 1.920884e-001  1.470365e-001
-0.025000 3.329223e-001 2.608962e-001  2.103565e-001
-0.022500 4.144830e-001 3.415492e-001  2.879773e-001
-0.020000 5.022632e-001 4.322944e-001  3.788357e-001
-0.017500 5.935155e-001 5.300621e-001  4.800532e-001
-0.015000 6.841840e-001 6.305940e-001  5.871893e-001
-0.012500 7.705897e-001 7.287768e-001  6.941033e-001
-0.010000 8.482151e-001 8.189864e-001  7.939973e-001
-0.007500 9.129510e-001 8.958479e-001  8.806449e-001
-0.005000 9.617524e-001 9.539544e-001  9.469328e-001
-0.002500 9.909168e-001 9.895298e-001  9.879749e-001
0.000000 1.000000e+000 1.000000e+000  1.000000e+000 
0.002500 9.874149e-001 9.845785e-001  9.822607e-001 
0.005000 9.535897e-001 9.438464e-001  9.357510e-001 
0.007500 9.012596e-001 8.812307e-001  8.639439e-001 
0.010000 8.327376e-001 8.003661e-001  7.725376e-001 
0.012500 7.514845e-001 7.058366e-001  6.679549e-001 
0.015000 6.619457e-001 6.041509e-001  5.570302e-001 
0.017500 5.683681e-001 5.010046e-001  4.475494e-001 
0.020000 4.753717e-001 4.014317e-001  3.449477e-001 
0.022500 3.862257e-001 3.098674e-001  2.540511e-001 
0.025000 3.042539e-001 2.295213e-001  1.777849e-001 
0.027500 2.316708e-001 1.624329e-001  1.174476e-001 
0.030000 1.698607e-001 1.091552e-001  7.258596e-002 
0.032500 1.194127e-001 6.913415e-002  4.160326e-002 
0.035000 8.003957e-002 4.092576e-002  2.206192e-002 
0.037500 5.079671e-002 2.256885e-002  1.115980e-002 
0.040000 3.034225e-002 1.179140e-002  6.130804e-003 
0.042500 1.704696e-002 6.382956e-003  4.589250e-003 
0.045000 9.210735e-003 4.344807e-003  4.682207e-003 
0.047500 5.229753e-003 4.094936e-003  5.180890e-003 
0.050000 3.704768e-003 4.501548e-003  5.417706e-003 
0.052500 3.516065e-003 4.874975e-003  5.175956e-003 
0.055000 3.854022e-003 4.899761e-003  4.531847e-003 
0.057500 4.206479e-003 4.533022e-003  3.699668e-003 
0.060000 4.311802e-003 3.894525e-003  2.908798e-003 
0.062500 4.105978e-003 3.163797e-003  2.321317e-003 
0.065000 3.646030e-003 2.511033e-003  2.001744e-003 
0.067500 3.055075e-003 2.045612e-003  1.922059e-003 
0.070000 2.461540e-003 1.803201e-003  1.994013e-003 
0.072500 1.970822e-003 1.754939e-003  2.108660e-003 
0.075000 1.641432e-003 1.829451e-003  2.174246e-003 
0.077500 1.485024e-003 1.941640e-003  2.138016e-003 
0.080000 1.472896e-003 2.017656e-003  1.991508e-003 
0.082500 1.550904e-003 2.010541e-003  1.763542e-003 
0.085000 1.657908e-003 1.907751e-003  1.503498e-003 
0.087500 1.739684e-003 1.726025e-003  1.260484e-003 
0.090000 1.759468e-003 1.500109e-003  1.071935e-003 
0.092500 1.703426e-003 1.271898e-003  9.526830e-004 
0.095000 1.577093e-003 1.075612e-003  8.982701e-004 
0.097500 1.402408e-003 9.332336e-004  8.909360e-004 
0.100000 1.209211e-003 8.501888e-004  9.073295e-004 
0.102500 1.026285e-003 8.178163e-004  9.285687e-004 
0.105000 8.764269e-004 8.196124e-004  9.446138e-004 
0.107500 7.722454e-004 8.365234e-004  9.545874e-004 
0.110000 7.154170e-004 8.546137e-004  9.642132e-004 
0.112500 6.983761e-004 8.661153e-004  9.802812e-004 
0.115000 7.081507e-004 8.709079e-004  1.005808e-003 
0.117500 7.303323e-004 8.744258e-004  1.037488e-003 
0.120000 7.526979e-004 8.828011e-004  1.066012e-003 
0.122500 7.682833e-004 9.007689e-004  1.078159e-003 
0.125000 7.750983e-004 9.278780e-004  1.062616e-003 
0.127500 7.758007e-004 9.581994e-004  1.012561e-003 
0.130000 7.760483e-004 9.823342e-004  9.307701e-004 
0.132500 7.815777e-004 9.892418e-004  8.271140e-004 
0.135000 7.962426e-004 9.695934e-004  7.187842e-004 
0.137500 8.198067e-004 9.200142e-004  6.235857e-004 
0.140000 8.485662e-004 8.431026e-004  5.562473e-004 
0.142500 8.757956e-004 7.474637e-004  5.243781e-004 
0.145000 8.924241e-004 6.469187e-004  5.262479e-004 
0.147500 8.914871e-004 5.559090e-004  5.520388e-004 
0.150000 8.670676e-004 4.869645e-004  5.862673e-004 
0.152500 8.182593e-004 4.471714e-004  6.124814e-004 
0.155000 7.486934e-004 4.374475e-004  6.179490e-004 
0.157500 6.653231e-004 4.520422e-004  5.963067e-004 
0.160000 5.785471e-004 4.802075e-004  5.494773e-004 
0.162500 4.986130e-004 5.090677e-004  4.859981e-004 
0.165000 4.347138e-004 5.269285e-004  4.187396e-004 
0.167500 3.928109e-004 5.258704e-004  3.604713e-004 
0.170000 3.746451e-004 5.029534e-004  3.213110e-004 
0.172500 3.774097e-004 4.613087e-004  3.053453e-004 
0.175000 3.947306e-004 4.082493e-004  3.106112e-004 
0.177500 4.180915e-004 3.536869e-004  3.301818e-004 
0.180000 4.386254e-004 3.070815e-004  3.543113e-004 
0.182500 4.488968e-004 2.757878e-004  3.733421e-004 
0.185000 4.442517e-004 2.628810e-004  3.810116e-004 
0.187500 4.235659e-004 2.672255e-004  3.752222e-004 
0.190000 3.894744e-004 2.837950e-004  3.583549e-004 
0.192500 3.470896e-004 3.053341e-004  3.366181e-004 
0.195000 3.032165e-004 3.246441e-004  3.177813e-004 
0.197500 2.648215e-004 3.356436e-004  3.091365e-004 
0.200000 2.371859e-004 3.354535e-004  3.156141e-004 
];
return