function v = colloc_constants(c)

% colloc_constants Defines (pseudo)-constants for collocations subsystem
%
% This m-file defines values required by functions in the collocations subsystem.
% It differs from collocs_config in the manner that the values here should
% not normally be changed. If called without arguments, returns
% the entire structure containing all properties and values.
%
% FORMAT
%
%   value = colloc_constants(c)
%
% IN
%
%   c   string      the property asked for (optional)
%
% OUT
%
%   v   (any)       the value of property c or the entire struct
%
% $Id: colloc_constants.m 7175 2011-10-10 09:06:45Z gerrit $

%% define what processing functions to use for what sensor pair

A.process_cpr_mhs = @colloc_process_cpr_mhs;
A.process_data_cpr_mhs = @colloc_process_data_cpr_mhs;
A.process_meandata_cpr_mhs = @colloc_process_meandata_cpr_mhs;

A.no_cpr_mhs = 3;

% set each combination of amsub and mhs to the same function handle
for s1 = {'amsub', 'mhs'}
    for s2 = {'amsub', 'mhs'}
        A.(sprintf('process_%s_%s', s1{1}, s2{1})) = ...
            partial_x(s1{1}, s2{1});
        A.(sprintf('process_data_%s_%s', s1{1}, s2{1})) = ...
            partial_data_x(s1{1}, s2{1});
        A.(sprintf('no_%s_%s', s1{1}, s2{1})) = 2;
    end
end

A.process_cpr_avhrr = @colloc_process_cpr_avhrr;
A.process_data_cpr_avhrr = @colloc_process_data_cpr_avhrr;

A.no_cpr_avhrr = 2;

A.process_hirscs_mhs = @colloc_process_hirscs_poes;
A.process_data_hirscs_mhs = @colloc_process_data_hirscs_poes;

A.no_hirscs_mhs = 2;

A.process_amsua_amsua = partial_x('amsua', 'amsua');
A.process_data_amsua_amsua = partial_data_x('amsua', 'amsua');

A.no_amsua_amsua = 2;

A.process_hirs_hirs = partial_x('hirs', 'hirs');
A.process_data_hirs_hirs = partial_data_x('hirs', 'hirs');

A.no_hirs_hirs = 2;

A.process_collocation_cpr_mhs_avhrr = @colloc_process_collocation_cpr_poes_avhrr;
A.process_data_collocation_cpr_mhs_avhrr = @colloc_process_data_collocation_cpr_poes_avhrr;
A.process_meandata_collocation_cpr_mhs_avhrr = @colloc_process_meandata_collocation_cpr_poes_avhrr;

A.no_collocation_cpr_mhs_avhrr = 3;

%% column definitions

%% helper: anything x/x

x_x_overlap = struct(...
    'LAT1', 1, ...
    'LONG1', 2, ...
    'START1', 3, ...
    'TIME1', 4, ...
    'I1', 5, ...
    'C1', 6, ...
    'LZA1', 7, ...
    'LAA1', 8, ...
    'SZA1', 9, ...
    'SAA1', 10, ...
    'LAT2', 11, ...
    'LONG2', 12, ...
    'START2', 13, ...
    'TIME2', 14, ...
    'I2', 15, ...
    'C2', 16, ...
    'LZA2', 17, ...
    'LAA2', 18, ...
    'SZA2', 19, ...
    'SAA2', 20, ...
    'DIST', 21, ...
    'INT', 22, ...
    'NCOLS', 22);

%% for cpr/mhs
A.cols_cpr_mhs.overlap = struct(...
    'C_LONG',    1, ...
	'C_LAT',     2, ...
	'C_START',   3, ...
	'C_TIME',    4, ...
	'C_I',       5, ...
	'A_LONG',    6, ...
	'A_LAT',     7, ...
	'A_START',   8, ...
	'A_TIME',    9, ...
	'A_I',       10, ...
	'A_C',       11, ...
	'A_DIST',    12, ...
	'A_INT',     13, ...
	'B_LONG',    14, ...
	'B_LAT',     15, ...
	'B_START',   16, ...
	'B_TIME',    17, ...
	'B_I',       18, ...
	'B_C',       19, ...
	'B_DIST',    20, ...
	'B_INT',     21, ...
    'B_LZA',     22, ...
    'B_LAA',     23, ...
	'H_LONG',    24, ...
	'H_LAT',     25, ...
	'H_START',   26, ...
	'H_TIME',    27, ...
	'H_I',       28, ...
	'H_C',       29, ...
	'H_DIST',    30, ...
	'H_INT',     31, ...
    'SZA',       32, ...
    'SAA',       33, ...
    'filter_double', {{{}, {'B_START', 'B_I'}}}, ... % for tackling double scanlines
	'NCOLS',     33);

A.cols_cpr_mhs.data = struct(...
	'ROIWP',     1, ...
	'dROIWP',    2, ...
	'IOROIWP',   3, ...
	'dIOROIWP',  4, ...
	'AMSU', 5:24, ...
	'AMSU_A', 5:19, ...
	'AMSU_B', 20:24, ...
	'MHS', 20:24, ...
	'HIRS', 25:44, ...
	'MSPPS_IWP', 45, ...
	'NCOLS', 45);

% mean files
A.cols_cpr_mhs.meandata = struct(...
	'FIRST', 1, ... % meandata should always start with first/last
	'LAST', 2, ...
	'NO', 3, ...
	'MEAN', 4, ...
	'SD', 5, ...
	'CV', 6, ...
	'FRAC', 7, ...
	'INHIRS', 8, ...
    'NCOLS', 8);

%% for POES/POES

A.cols_mhs_mhs.overlap = x_x_overlap;
A.cols_mhs_mhs.overlap.filter_double = {{{'START1', 'I1'}, {'START2', 'I2'}}}; % for tackling double scanlines

A.cols_mhs_mhs.data = struct(...
	'AMSU_B1', 1:5, ...
	'MHS1', 1:5, ...
    'AMSU_B2', 6:10, ...
    'MHS2', 6:10, ...
    'NCOLS', 10);

%% for AMSUA/AMSUA

A.cols_amsua_amsua.overlap = x_x_overlap;

A.cols_amsua_amsua.data = struct(...
    'AMSU_A1', 1:15, ...
    'AMSU_A2', 16:30, ...
    'NCOLS', 30);

%% for HIRS/HIRS

A.cols_hirs_hirs.overlap = x_x_overlap;

A.cols_hirs_hirs.data = struct(...
    'HIRS1', 1:20, ...
    'HIRS2', 21:40, ...
    'NCOLS', 40);

%% for CPR/AVHRR

A.cols_cpr_avhrr.overlap = struct(...
    'C_LAT', 1, ...
    'C_LONG', 2, ...
    'C_START', 3, ...
    'C_TIME', 4, ...
    'C_I', 5, ...
    'AVHRR_LAT', 6, ...
    'AVHRR_LONG', 7, ...
    'AVHRR_START', 8, ...
    'AVHRR_TIME', 9, ...
    'AVHRR_ROW', 10, ...
    'AVHRR_COL', 11, ...
    'AVHRR_RAA', 12, ...
    'SZA', 13, ...
    'AVHRR_LZA', 14, ...
    'DIST', 15, ...
    'INT', 16, ...
    'filter_double', {{{}, {'AVHRR_START', 'AVHRR_ROW'}}}, ... % for tackling double scanlines
    'NCOLS', 16);

A.cols_cpr_avhrr.data = struct(...
    'ROIWP', 1, ...
    'dROIWP', 2, ...
    'IOROIWP', 3, ...
    'dIOROIWP', 4, ...
    'AVHRR', 5:10, ...
    'CLOUDFLAG', 11, ...
    'NCOLS', 11);

A.cols_cpr_avhrr.aliases = struct();

%% for HIRSCS/MHS

A.cols_hirscs_mhs.overlap = struct(...
    'H_LAT', 1, ...
    'H_LONG', 2, ...
    'HCS_I', 3, ...
    'H_START', 4, ...
    'H_TIME', 5, ...
    'H_SZA', 6, ...
    'H_ALT', 7, ...
    'H_I', 8, ...
    'H_C', 9, ...
    'MHS_LZA', 10, ...
    'MHS_LAA', 11, ...
    'MHS_LAT', 12, ...
    'MHS_LONG', 13, ...
    'MHS_ROW', 14, ...
    'MHS_COL', 15, ...
    'MHS_START', 16, ...
    'MHS_TIME', 17, ...
    'MHS_SZA', 18, ...
    'MHS_SAA', 19, ...
    'DIST', 20, ...
    'INT', 21, ...
    'filter_double', {{{}, {'MHS_START', 'MHS_ROW'}}}, ... % TODO/FIXME: HIRS Not implemented yet
    'NCOLS', 21);

A.cols_hirscs_mhs.data = struct(...
    'VIS', 1, ...
    'BT', 2:20, ...
    'LC', 21:39, ...
    'MHS', 40:44, ...
    'NCOLS', 44);

%% for collocation cpr poes/avhrr

A.cols_collocation_cpr_mhs_avhrr.overlap = struct(...
    'CPR_MHS_START', 1, ...
    'CPR_MHS_ROW', 2, ...
    'CPR_MHS_MEAN_ROW', 3, ...
    'AVHRR_LAT', 4, ...
    'AVHRR_LONG', 5, ...
    'AVHRR_START', 6, ...
    'AVHRR_TIME', 7, ...
    'AVHRR_ROW', 8, ...
    'AVHRR_COL', 9, ...
    'AVHRR_RAA', 10, ...
    'AVHRR_SZA', 11, ...
    'AVHRR_LZA', 12, ...
    'DIST_MHS_AVHRR', 13, ...
    'DIST_CPR_AVHRR', 14, ...
    'INT_MHS_AVHRR', 15, ...
     'filter_double', {{{}, {'AVHRR_START', 'AVHRR_ROW'}}}, ... % for tackling double scanlines
    'NCOLS', 14);

A.cols_collocation_cpr_mhs_avhrr.data = struct(...
    'AVHRR', 1:6, ...
    'CLOUDFLAG', 7, ...
    'NCOLS', 7);

A.cols_collocation_cpr_mhs_avhrr.meandata = struct(...
    'FIRST', 1, ... % meandata should always start with first/last
    'LAST', 2, ...
    'NO', 3:8, ...
    'MEAN', 9:14, ...
    'SD', 15:20, ...
    'NCOLS', 20);

A.cols_collocation_cpr_mhs_avhrr.links = struct(...
    'start', 'CPR_POES_START', ...
    'lineno', 'CPR_POES_LINENO', ...
    'sat1', 'cloudsat', ...
    'sensor1', 'cpr', ...
    'sensor2', 'mhs');

%% stored style, from 2010-09-22

% for storing in NetCDF, take a structure according to
% A.cols.cpr_poes.stored.C_LONG.type = 'double'; % type is mandatory
% additional atts are optional
% A.cols.cpr_poes.stored.C_LONG.atts.long_name = 'Cloudsat Longitude';
% A.cols.cpr_poes.stored.C_LONG.atts.units = 'degrees_east';
% ... etc., see below

%% shared

SZA = struct(...
        'type', 'float', ...
        'atts', struct(...
            'long_name', 'Solar zenith angle', ...
            'units', 'degrees', ...
            'valid_range', [-180 180]));
SAA = struct(...
        'type', 'float', ...
        'atts', struct(...
            'long_name', 'Solar azimuth angle', ...
            'units', 'degrees', ...
            'valid_range', [-180 180]));
LZA = struct(...
        'type', 'float', ...
        'atts', struct(...
            'long_name', 'Local zenith angle', ...
            'units', 'degrees', ...
            'valid_range', [-180 180]));
LAA = struct(...
        'type', 'float', ...
        'atts', struct(...
            'long_name', 'Local azimuth angle', ...
            'units', 'degrees', ...
            'valid_range', [-180 180]));
RAA = struct(...
        'type', 'float', ...
        'atts', struct(...
            'long_name', 'Relative azimuth angle', ...
            'units', 'degrees', ...
            'valid_range', [-180 180]));

hirs = struct(...
    'H_LAT', struct(...
        'type', 'float', ...
        'atts', struct(...
            'long_name', 'HIRS Latitude', ...
            'units', 'degrees_north', ...
            'valid_range', [-90 90])), ...
    'H_LONG', struct(...
        'type', 'float', ...
        'atts', struct(...
            'long_name', 'HIRS Longitude', ...
            'units', 'degrees_east', ...
            'valid_range', [-180 180])), ...
    'H_I', struct(...
        'type', 'short', ...
        'atts', struct(...
            'long_name', 'HIRS scanline number')), ...
    'H_C', struct(...
        'type', 'byte', ...
        'atts', struct(...
            'long_name', 'HIRS scanline position', ...
            'valid_range', [1 56])), ...
    'H_START', struct(...
        'type', 'int', ...
        'atts', struct(...
            'long_name', 'HIRS granule start time', ...
            'units', 'seconds since 1970-01-01 00:00:00')), ...
    'H_TIME', struct(...
        'type', 'int', ...
        'atts', struct(...
            'long_name', 'HIRS measurement time', ...
            'units', 'seconds since 1970-01-01 00:00:00')), ...
    'HIRS', struct(...
        'type', 'float', ...
        'dims', {{'HIRS_CHANS', 20}}, ...
        'atts', struct(...
            'long_name', 'HIRS Brightness Temperature', ...
            'units', 'Kelvin')));

amsu_a = struct(...
    'A_LONG', struct(...
        'type', 'float', ...
        'atts', struct(...
            'long_name', 'AMSU-A Longitude', ...
            'units', 'degrees_east', ...
            'valid_range', [-180 180])), ...
    'A_LAT', struct(...
        'type', 'float', ...
        'atts', struct(...
            'long_name', 'AMSU-A Latitude', ...
            'units', 'degrees_north', ...
            'valid_range', [-90 90])), ...
    'A_START', struct(...
        'type', 'int', ...
        'atts', struct(...
            'long_name', 'AMSU-A granule start time', ...
            'units', 'seconds since 1970-01-01 00:00:00')), ...
    'A_TIME', struct(...
        'type', 'int', ...
        'atts', struct(...
            'long_name', 'AMSU-A measurement time', ...
            'units', 'seconds since 1970-01-01 00:00:00')), ...
    'A_I', struct(...
        'type', 'short', ...
        'atts', struct(...
            'long_name', 'AMSU-A scanline number')), ...
    'A_C', struct(...
        'type', 'byte', ...
        'atts', struct(...
            'long_name', 'AMSU-A scanline position', ...
            'valid_range', [1 30])));

amsu_b = struct(...
    'B_LONG', struct(...
        'type', 'float', ...
        'atts', struct(...
            'long_name', 'AMSU-B/MHS Longitude', ...
            'units', 'degrees_east', ...
            'valid_range', [-180 180])), ...
    'B_LAT', struct(...
        'type', 'float', ...
        'atts', struct(...
            'long_name', 'AMSU-B/MHS Latitude', ...
            'units', 'degrees_north', ...
            'valid_range', [-90 90])), ...
    'B_START', struct(...
        'type', 'int', ...
        'atts', struct(...
            'long_name', 'AMSU-B/MHS granule start time', ...
            'units', 'seconds since 1970-01-01 00:00:00')), ...
    'B_TIME', struct(...
        'type', 'int', ...
        'atts', struct(...
            'long_name', 'AMSU-B/MHS measurement time', ...
            'units', 'seconds since 1970-01-01 00:00:00')), ...
    'B_I', struct(...
        'type', 'short', ...
        'atts', struct(...
            'long_name', 'AMSU-B/MHS scanline number')), ...
    'B_C', struct(...
        'type', 'byte', ...
        'atts', struct(...
            'long_name', 'AMSU-B/MHS scanline position', ...
            'valid_range', [1 90])), ...
    'B_LZA', LZA, ...
    'B_LAA', LAA);

mhs = struct(...
    'MHS_LAT', struct(...
        'type', 'double', ...
        'atts', struct(...
            'long_name', 'AMSU-B/MHS Latitude', ...
            'units', 'degrees_north', ...
            'valid_range', [-90 90])), ...
    'MHS_LONG', struct(...
        'type', 'double', ...
        'atts', struct(...
            'long_name', 'AMSU-B/MHS Longitude', ...
            'units', 'degrees_east', ...
            'valid_range', [-180 180])), ...
    'MHS_ROW', struct(...
        'type', 'short', ...
        'atts', struct(...
            'long_name', 'AMSU-B/MHS scanline number')), ...
    'MHS_COL', struct(...
        'type', 'byte', ...
        'atts', struct(...
            'long_name', 'AMSU-B/MHS scanline position', ...
            'valid_range', [1 90])), ...
    'MHS_START', struct(...
        'type', 'int', ...
        'atts', struct(...
            'long_name', 'AMSU-B/MHS granule start time', ...
            'units', 'seconds since 1970-01-01 00:00:00')), ...
    'MHS_TIME', struct(...
        'type', 'int', ...
        'atts', struct(...
            'long_name', 'AMSU-B/MHS measurement time', ...
            'units', 'seconds since 1970-01-01 00:00:00')), ...
    'MHS_LZA', LZA, ...
    'MHS_LAA', LAA, ...
    'MHS_SZA', SZA, ...
    'MHS_SAA', SAA, ...
    'MHS', struct(...
        'type', 'float', ...
        'dims', {{'MHS_CHANS', 5}}, ...
        'atts', struct(...
            'long_name', 'MHS brightness temperature', ...
            'units', 'Kelvin')));
 
cloudsat = struct(...
    'C_LONG', struct(...
        'type', 'double', ...
        'atts', struct(...
            'long_name', 'Cloudsat Longitude', ...
            'units', 'degrees_east', ...
            'valid_range', [-180 180])), ...
    'C_LAT', struct(...
        'type', 'double', ...
        'atts', struct(...
            'long_name', 'Cloudsat Latitude', ...
            'units', 'degrees_north', ...
            'valid_range', [-90 90])), ...
    'C_START', struct(...
        'type', 'int', ...
        'atts', struct(...
            'long_name', 'Cloudsat granule start time', ...
            'units', 'seconds since 1970-01-01 00:00:00')), ...
    'C_TIME', struct(...
        'type', 'int', ...
        'atts', struct(...
            'long_name', 'Cloudsat measurement time', ...
            'units', 'seconds since 1970-01-01 00:00:00')), ...
    'C_I', struct(...
        'type', 'int', ... % would like to have ushort here, but no NetCDF-4 support yet
        'atts', struct(...
            'long_name', 'Cloudsat measurement number')), ...
    'ROIWP', struct(...
        'type', 'float', ...
        'atts', struct(...
            'long_name', 'CloudSat Radar-Only CPR IWP', ...
            'units', 'g/m^2')), ...
    'dROIWP', struct(...
        'type', 'float', ...
        'atts', struct(...
            'long_name', 'Cloudsat Radar-Only CPR IWP uncertainty', ...
            'units', 'g/m^2')), ...
    'IOROIWP', struct(...
        'type', 'float', ...
        'atts', struct(...
            'long_name', 'Cloudsat Ice-Only Radar-Only CPR IWP', ...
            'units', 'g/m^2')), ...
    'dIOROIWP', struct(...
        'type', 'float', ...
        'atts', struct(...
            'long_name', 'Cloudsat Ice-Only Radar-Only CPR IWP uncertainty', ...
            'units', 'g/m^2')));

avhrr = struct(...
    'AVHRR_LAT', struct(...
        'type', 'double', ...
        'atts', struct(...
            'long_name', 'AVHRR Latitude', ...
            'units', 'degrees_north', ...
            'valid_range', [-90 90])), ...
    'AVHRR_LONG', struct(...
        'type', 'double', ...
        'atts', struct(...
            'long_name', 'AVHRR Longitude', ...
            'units', 'degrees_east', ...
            'valid_range', [-180 180])), ...
    'AVHRR_START', struct(...
        'type', 'int', ...
        'atts', struct(...
            'long_name', 'AVHRR granule start time', ...
            'units', 'seconds since 1970-01-01 00:00:00')), ...
    'AVHRR_TIME', struct(...
        'type', 'int', ...
        'atts', struct(...
            'long_name', 'AVHRR measurement time', ...
            'units', 'seconds since 1970-01-01 00:00:00')), ...
    'AVHRR_ROW', struct(...
        'type', 'int', ...
        'atts', struct(...
            'long_name', 'AVHRR scanline number')), ...
    'AVHRR_COL', struct(...
        'type', 'int', ...
        'atts', struct(...
            'long_name', 'AVHRR scanline position')), ...
    'AVHRR_RAA', RAA, ...
    'AVHRR_SZA', SZA, ...
    'AVHRR_LZA', LZA, ...
    'AVHRR', struct(...
        'type', 'float', ...
        'dims', {{'AVHRR_CHANS', 6}}, ... # 3A/3B
        'atts', struct(...
            'long_name', 'AVHRR radiance', ...
            'units', '% or K', ...
            'note', 'ch 1, 2, 3a, 3b, 4, 5 (6 channels)')), ...
    'CLOUDFLAG', struct(...
        'type', 'byte', ...
        'atts', struct(...
            'long_name', 'Cloud flag', ...
            'note', 'See AVHRR documentation')));


primary = struct(...
    'LAT1', struct(...
        'type', 'double', ...
        'atts', struct(...
            'long_name', 'Primary Latitude', ...
            'units', 'degrees_north', ...
            'valid_range', [-90 90])), ...
    'LONG1', struct(...
        'type', 'double', ...
        'atts', struct(...
            'long_name', 'Primary Longitude', ...
            'units', 'degrees_east', ...
            'valid_range', [-180 180])), ...
    'START1', struct(...
        'type', 'int', ...
        'atts', struct(...
            'long_name', 'Primary granule starting time', ...
            'units', 'seconds since 1970-01-01 00:00:00')), ...
    'TIME1', struct(...
        'type', 'int', ...
        'atts', struct(...
            'long_name', 'Primary measurement time', ...
            'units', 'seconds since 1970-01-01 00:00:00')), ...
    'I1', struct(...
        'type', 'short', ...
        'atts', struct(...
            'long_name', 'Primary scanline number')), ...
    'C1', struct(...
        'type', 'short', ...
        'atts', struct(...
            'long_name', 'Primary scanline position', ...
            'valid_range', [1 90])), ...
    'LZA1', LZA, ...
    'LAA1', LAA, ...
    'SZA1', SZA, ...
    'SAA1', SAA);

secondary = struct(...
    'LAT2', struct(...
        'type', 'double', ...
        'atts', struct(...
            'long_name', 'Secondary Latitude', ...
            'units', 'degrees_north', ...
            'valid_range', [-90 90])), ...
    'LONG2', struct(...
        'type', 'double', ...
        'atts', struct(...
            'long_name', 'Secondary Longitude', ...
            'units', 'degrees_east', ...
            'valid_range', [-180 180])), ...
    'START2', struct(...
        'type', 'int', ...
        'atts', struct(...
            'long_name', 'Secondary granule starting time', ...
            'units', 'seconds since 1970-01-01 00:00:00')), ...
    'TIME2', struct(...
        'type', 'int', ...
        'atts', struct(...
            'long_name', 'Secondary measurement time', ...
            'units', 'seconds since 1970-01-01 00:00:00')), ...
    'I2', struct(...
        'type', 'short', ...
        'atts', struct(...
            'long_name', 'Secondary scanline number')), ...
    'C2', struct(...
        'type', 'short', ...
        'atts', struct(...
            'long_name', 'Secondary scanline position', ...
            'valid_range', [1 90])), ...
    'LZA2', LZA, ...
    'LAA2', LAA, ...
    'SZA2', SZA, ...
    'SAA2', SAA);

x_with_x = catstruct(...
    primary, ...
    secondary, ...
    struct(...
        'DIST', struct(...
            'type', 'float', ...
            'atts', struct(...
                'long_name', 'Distance', ...
                'units', 'km')), ...
        'INT', struct(...
            'type', 'float', ...
            'atts', struct(...
                'long_name', 'Secondary measurement time - primary measurement time', ...
                'units', 'seconds'))));

%% for cpr/mhs

A.cols_cpr_mhs.stored = catstruct(...
    cloudsat, ...
    amsu_a, ...
    amsu_b, ...
    hirs, ...
    struct( ... % specifics
        'SZA', SZA, ...
        'SAA', SAA, ...
        'A_DIST', struct(...
            'type', 'float', ...
            'atts', struct(...
                'long_name', 'Distance AMSU-A to Cloudsat', ...
                'units', 'km')), ...
        'A_INT', struct(...
            'type', 'float', ...
            'atts', struct(...
                'long_name', 'AMSU-A measurement time minus Cloudsat measurement time', ...
                'units', 'seconds')), ...
        'B_DIST', struct(...
            'type', 'float', ...
            'atts', struct(...
                'long_name', 'Distance AMSU-B/MHS to CloudSat', ...
                'units', 'km')), ...
        'B_INT', struct(...
            'type', 'float', ...
            'atts', struct(...
                'long_name', 'AMSU-B/MHS measurement time minus CloudSat measurement time', ...
                'units', 'seconds')), ...
        'H_DIST', struct(...
            'type', 'float', ...
            'atts', struct(...
                'long_name', 'Distance HIRS to Cloudsat', ...
                'units', 'km')), ...
        'H_INT', struct(...
            'type', 'float', ...
            'atts', struct(...
                'long_name', 'HIRS measurement time minus Cloudsat measurement time', ...
                'units', 'seconds')), ...
        'AMSU', struct(...
            'type', 'float', ...
            'dims', {{'AMSU_CHANS', 20}}, ...
            'atts', struct(...
                'long_name', 'AMSU Brightness Temperature', ...
                'units', 'Kelvin')), ...
        'MSPPS_IWP', struct(...
            'type', 'float', ...
            'atts', struct(...
                'long_name', 'NOAA NESDIS MSPPS IWP', ...
                'units', 'g/m^2')), ...
        'FIRST', struct(...
            'type', 'int', ... % would like to use ushort here
            'atts', struct(...
                'long_name', 'First corresponding row in overlap')), ...
        'LAST', struct(...
            'type', 'int', ... % would like to use ushort here
            'atts', struct(...
                'long_name', 'Last corresponding row in overlap')), ...
        'NO', struct(...
            'type', 'byte', ...
            'atts', struct(...
                'long_name', 'Number of CloudSat pixels in AMSU-B/MHS')), ...
        'MEAN', struct(...
            'type', 'float', ...
            'atts', struct(...
                'long_name', 'Mean CloudSat ROIWP', ...
                'units', 'g/m^2')), ...
        'SD', struct(...
            'type', 'float', ...
            'atts', struct(...
                'long_name', 'Standard deviation CloudSat ROIWP', ...
                'units', 'g/m^2')), ...
        'CV', struct(...
            'type', 'float', ...
            'atts', struct(...
                'long_name', 'Coefficient of variation CloudSat ROIWP')), ...
        'FRAC', struct(...
            'type', 'float', ...
            'atts', struct(...
                'long_name', 'Cloud fraction from CloudSat ROIWP in AMSU-B/MHS')), ...
        'INHIRS', struct(...
            'type', 'byte', ...
            'atts', struct(...
                'long_name', 'Number of CloudSat pixels in HIRS'))));

A.cols_cpr_mhs.aliases = struct(...
     'MHS', {{'AMSU', 16:20}}, ... % MHS = AMSU channels 16--20
     'AMSU_B', {{'AMSU', 16:20}}, ...
     'AMSU_A', {{'AMSU', 1:15}}); 

%% for POES/POES

A.cols_mhs_mhs.stored = catstruct(...
    x_with_x, ...
    struct(...
        'AMSU_B1', struct(...
            'type', 'float', ...
            'dims', {{'AMSUB_CHANS', 5}}, ...
            'atts', struct(...
                'long_name', 'Primary brightness temperature', ...
                'units', 'Kelvin')), ...
        'AMSU_B2', struct(...
            'type', 'float', ...
            'dims', {{'AMSUB_CHANS', 5}}, ...
            'atts', struct(...
                'long_name', 'Secondary brightness temperature', ...
                'units', 'Kelvin'))));

A.cols_mhs_mhs.aliases = struct(...
     'MHS1', {{'AMSU_B1', 1:5}}, ...
     'MHS2', {{'AMSU_B2', 1:5}});

%% for HIRS/HIRS

A.cols_hirs_hirs.stored = catstruct(...
    x_with_x, ...
    struct(...
        'HIRS1', struct(...
            'type', 'float', ...
            'dims', {{'HIRS_CHANS', 20}}, ...
            'atts', struct(...
                'long_name', 'Primary brightness temperature', ...
                'units', 'Kelvin')), ...
        'HIRS2', struct(...
            'type', 'float', ...
            'dims', {{'HIRS_CHANS', 20}}, ...
            'atts', struct(...
                'long_name', 'Secondary brightness temperature', ...
                'units', 'Kelvin'))));

A.cols_hirs_hirs.aliases = struct();

%% for AMSUA/AMSUA

A.cols_amsua_amsua.stored = catstruct(...
    x_with_x, ...
    struct(...
        'AMSU_A1', struct(...
            'type', 'float', ...
            'dims', {{'AMSUA_CHANS', 15}}, ...
            'atts', struct(...
                'long_name', 'Primary brightness temperature', ...
                'units', 'Kelvin')), ...
        'AMSU_A2', struct(...
            'type', 'float', ...
            'dims', {{'AMSUA_CHANS', 15}}, ...
            'atts', struct(...
                'long_name', 'Secondary brightness temperature', ...
                'units', 'Kelvin'))));

A.cols_amsua_amsua.aliases = struct();

%% for CPR/AVHRR

A.cols_cpr_avhrr.stored = catstruct(...
    cloudsat, ...
    avhrr, ...
    struct(...
        'DIST', struct(...
            'type', 'float', ...
            'atts', struct(...
                'long_name', 'Distance CPR-AVHRR', ...
                'units', 'km')), ...
        'INT', struct(...
            'type', 'float', ...
            'atts', struct(...
                'long_name', 'AVHRR measurement time minus CPR measurement time', ...
                'units', 'seconds'))));

%% for HIRSCS/MHS

A.cols_hirscs_mhs.stored = catstruct(...
    hirs, ...
    mhs, ...
    struct(...
        'HCS_I', struct(...
            'type', 'int', ...
            'atts', struct(...
                'long_name', 'HIRS-CS index')), ...
        'H_SZA', struct(...
            'type', 'float', ...
            'atts', struct(...
                'long_name', 'solar zenith angle', ...
                'units', 'degrees', ...
                'valid_range', [0 180])), ...
        'H_ALT', struct(...
            'type', 'float', ...
            'atts', struct(...
                'long_name', 'satellite altitude', ...
                'units', 'km')), ...
        'DIST', struct(...
            'type', 'float', ...
            'atts', struct(...
                'long_name', 'Distance AMSU-B/MHS to HIRS', ...
                'units', 'km')), ...
        'INT', struct(...
            'type', 'float', ...
            'atts', struct(...
                'long_name', 'AMSU-B/MHS time minus HIRS time', ...
                'units', 'seconds')), ...
        'VIS', struct(...
            'type', 'float', ...
            'atts', struct(...
                'long_name', 'HIRS channel 20', ...
                'units', '%')), ...
        'BT', struct(...
            'type', 'float', ...
            'dims', {{'HIRS_CHANS', 19}}, ...
            'atts', struct(...
                'long_name', 'HIRS-CS brightness temperature', ...
                'units', 'Kelvin')), ...
        'LC', struct(...
            'type', 'float', ...
            'dims', {{'HIRS_CHANS', 19}}, ...
            'atts', struct(...
                'long_name', 'HIRS-CS limb-corrected brightness temperature', ...
                'units', 'Kelvin'))));
% double: already contained in 'BT' (and 'LC')
A.cols_hirscs_mhs.stored = rmfield(A.cols_hirscs_mhs.stored, 'HIRS');

A.cols_hirscs_mhs.aliases = struct(...
     'MHS', {{'AMSU_B', 1:5}});

%% for collocation cpr poes/avhrr

A.cols_collocation_cpr_mhs_avhrr.stored = catstruct(...
    avhrr, ...
    struct(...
        'CPR_MHS_START', struct(...
            'type', 'int', ...
            'atts', struct(...
                'long_name', 'collocation_cpr_mhs granule start time', ...
                'units', 'seconds since 1970-01-01 00:00:00')), ...
        'CPR_MHS_ROW', struct(...
            'type', 'int', ...
            'atts', struct(...
                'long_name', 'collocation_cpr_mhs collocations line number')), ...
        'CPR_MHS_MEAN_ROW', struct(...
            'type', 'int', ...
            'atts', struct(...
                'long_name', 'collocation_cpr_mhs mean collocations line number')), ...
        'DIST_MHS_AVHRR', struct(...
            'type', 'float', ...
            'atts', struct(...
                'long_name', 'Distance MHS-AVHRR', ...
                'units', 'km')), ...
        'DIST_CPR_AVHRR', struct(...
            'type', 'float', ...
            'atts', struct(...
                'long_name', 'Distance CPR-AVHRR', ...
                'units', 'km')), ...
        'INT_MHS_AVHRR', struct(...
            'type', 'float', ...
            'atts', struct(...
                'long_name', 'AVHRR time minus collocated AMSU-B/MHS time', ...
                'units', 'seconds')), ...
        'FIRST', struct(...
            'type', 'int', ...
            'atts', struct(...
                'long_name', 'First row in overlap corresponding to meanrow in averaged data')), ...
        'LAST', struct(...
            'type', 'int', ...
            'atts', struct(...
                'long_name', 'Last row in overlap corresponding to meanrow in averaged data')), ...
        'NO', struct(...
            'type', 'byte', ...
            'dims', {{'AVHRR_CHANS', 5}}, ...
            'atts', struct(...
                'long_name', 'Number of valid AVHRR pixels inside AMSU-B/MHS collocated with CloudSat CPR')), ...
        'MEAN', struct(...
            'type', 'float', ...
            'dims', {{'AVHRR_CHANS', 5}}, ...
            'atts', struct(...
                'long_name', 'Mean AVHRR brightness temperature for valid AVHRR pixels inside AMSU-B/MHS collocated with CloudSat CPR', ...
                'units', '%/Kelvin', ...
                'note', 'See note for AVHRR')), ...
        'SD', struct(...
            'type', 'float', ...
            'dims', {{'AVHRR_CHANS', 5}}, ...
            'atts', struct(...
                'long_name', 'Standard deviation for AVHRR brightness temperature for valid AVHRR pixels inside AMSU-B/MHS collocated with CloudSat CPR', ...
                'units', '%/Kelvin', ...
                'note', 'See note for AVHRR'))));
           
A.cols_collocation_cpr_mhs_avhrr.aliases = struct();

if exist('c', 'var')
    if isfield(A, c)
        v = A.(c);
    else
        % replace aliases
        c = replace_aliases(c);
        v = A.(c);
    end
else
    v = A;
end

end

function p=partial_x(sens1, sens2)
% partial_x Return partial-applied function for process_..._x_to_x

    function res = wrapper(collocations, ...
        sat1, date1, data1, ...
        sat2, date2, data2)
    res = colloc_process_x_to_x(collocations, sat1, sens1, date1, data1, ...
        sat2, sens2, date2, data2);
    end
p = @wrapper;
end

function p=partial_data_x(sens1, sens2)
% partial_data_x Return partial-applied function for process_data_..._x-to_x_


    function res = wrapper(collocations, M_c, ...
        sat1, date1, data1, ...
        sat2, date2, data2)
    res = colloc_process_data_x_to_x(collocations, M_c, sat1, sens1, date1, ...
        data1, sat2, sens2, date2, data2);
    end
p = @wrapper;
                       
end
