function [files,info] = test_gmt(in)
% TEST_GMT To test various aspecs of gmt_plot
%
% PURPOSE: To test various aspecs of gmt_plot. Especially before commiting
%          changes
%
% USAGE: [info,files] = test_gmt(in)
%
% IN   structure of arguments to be additionally used
%      in the test_gmt call/s. I.e. it will use the default values for each
%      test and the addional options added by the user (which have precedence).
%      -----------
%      in.test: If you only want to run a subset of tests, e.g. in.test =
%      [1,3,4] runs tests 1,3,4. The default behaviour is to run all tests ([1:17])
%
%      -----------
%
%
% OUT: files = cellarray of paths to created figures
%      info  = cellarray of information about each test. The information is
%              also displayed at the end of the run.
%
% created by Salomon Eliasson
% $Id: test_gmt.m 10678 2017-09-29 09:57:06Z seliasson $

if ~exist('in','var'); in=struct([]); end

% SET some defaults
alltests = 1:17;
default.test      = alltests; % which tests to do (default=all)

in = optargs_struct(in,default);
testDataResolution=1;
[gridded,ungridded] = test_generate_data_sin_cos(testDataResolution);

[files,info]           = TEST(in,gridded,ungridded);
if strcmp(files{end}(end-2:end),'.sh')
    logtext(1,'Debugging!!\n')
    return
end
default.display   = true;
default.filename  = 'all_tests' ;
default.pdfcrop   = true;
od = isfield(in,'outdir');
default.outdir    = gmtlab('OUTDIR');
in = optargs_struct(in,default);

% for the final pdfs

if isequal(in.test,alltests) % if all tests
    default.cols = [3 3 3 2 1 3 2];
else
    default.cols = min([length(in.test),2]);
end

in = optargs_struct(in,default);

if ~ischar(in.outdir)
    in.outdir = tempdir;
    in.outdir = in.outdir(1:end-1);
end

warning('off','catstruct:DuplicatesFound') %I'm intentially doing this often

if in.display
    a = system('which pdfinfo');
    
    if a ~=1
        % PUT all figure in one pdf
        files{end+1} = combine_pdfs(files,...
            struct('title','GMT plots: testing its functionality',...
            'fontsize','1cm',...
            'filename',[in.filename,'.pdf'],...
            'cols',in.cols,...
            'valign','b',...
            'outdir',in.outdir,...
            'display',in.display,...
            'pdfcrop',in.pdfcrop));
        
    else
        logtext(1,'Display all PDFs separately since pdfinfo is missing, and hence combine_pdfs wont work.');
        for F = files
            f = F{1};
            % from combine_pdfs
            if ismac
                exec_system_cmd(sprintf('open %s',f));
            elseif isunix
                [~,b1]=system('which gnome-open');
                [~,b2]=system('which kde-open');
                if ~isempty(b1)
                    exec_system_cmd(sprintf('gnome-open %s &',f));
                elseif ~isempty(b2)
                    exec_system_cmd(sprintf('kde-open %s &',f));
                end
            end
        end
    end
end
% DISPLAY information about the tests
disp(' ')
disp(' ')
cellfun(@fprintf,info,'uniformoutput',false)
disp('Look at help gmt_plot for more detailed descriptions of options ')
disp('specified here. I suggest looking at the call for each test in test_gmt')
disp('if you want to see an example on how to create these figures.')
disp(' ')
disp('All the tests where merged onto one pdf using combine_pdfs.m.')

warning('on','gmtcatstruct:DuplicatesFound')

if ~od && ~ischar(gmtlab('OUTDIR'))
    a = sprintf('Files are saved in your tempdir (%s)',in.outdir);
    in.outdir = in.outdir(1:end-1);
    b = 'It is better to set a default directory.';
    c = 'I.e by setting gmtlab(''OUTDIR'',''<Somewhere>''), e.g. in startup.m';
    d = 'Setting this, or inputting a valid directory: in.outdir will suppress this message.';
    fprintf(2,'Warning: %s\n%s\n%s\n%s',a,b,c,d);
end

if nargout==0
    clear info files
end
fprintf('Output file saved at: %s/%s.pdf\n',in.outdir,in.filename)

end
%%%%%%%%%%%%%%%%%
% SUB FUNCTIONS
%    |||||||
%    vvvvvvv

function [files,info]           = TEST(in,gridded,ungridded)

info = cell(1,length(in.test));
files = cell(1,length(in.test));
default.display = false;
in = optargs_struct(in,default);

ind = 1;
for T = sort(in.test)
    switch T
        case 1
            %% TEST: Difference plot, reference=0, nlevels, sidebar
            test.title        = 'gridded: title ^2 * @~D@~r@+-2@~g@~@+, unit';
            test.unit         = 'kg/m@+2@+s';
            test.psbasemap.axes = 'neSW';
            test.legend.tick_spacing = 25;
            
            test.filename = sprintf('Test%g',T);
            testThis = catstruct(gridded,rmfield(in,'test'),test);
            files{ind} = gmt_plot(testThis);
            
            info{ind} = ['Test:' num2str(T) ' demonstrates that you can include some special',...
                'characters in in.title and in.unit. This test uses ',...
                'gridded data, where size(data) = [length(lat),length(lon)]',...
                'Use in.map_title_offset my shift the title wrt the plot',...
                'You can also change the size and position of the legend',...
                '(''x-displacement/y-displacement/height/width''))',...
                'Here the tick-interval is fixed to every 25th level using in.legend.tick_spacing = 25\n\n'];
            
        case 2
            
            test.data = gridded.data-200;  % for difference
            test.lat = gridded.lat;
            test.lon = gridded.lon;
            test.unit='';
            test.xunit='kg/m@+2@+';
            test.ctable.name = 'mypolar';
            test.ctable.reference = 0;
            test.legend.sidebar = 3; % both above and below
            test.psbasemap.axes = 'neSW';
            test.title = 'Diff plot,ref=0,sidebar=3 (above & below)';
            test.legend.tick_spacing = 25;
            test.filename = sprintf('Test%g',T);
            
            testThis = catstruct(rmfield(in,'test'),test);
            files{ind} = gmt_plot(testThis);
            
            info{ind} = ['Test:' num2str(T) ' demonstrates the mypolar-colortable which is similar to polar.cpt',...
                'Using in.ctable=''mypolar'' you can define where the reference value',...
                'should be (here in.reference = 0). The white color is centered around this value.',...
                'in.basemap_axis = ''nSeW'' is tested, and means that the south (S) and west(W)',...
                'axes are labelled. in.sidebar = 3 is tested and means that triangles',...
                'above and below legend-range are shown to indicate the color of data outside the datarange',...
                'defined by in.datarange = [min max]. The title offset is set to -0.5 here (moving it down)',...
                'using in.map_title_offset = -0.5; See help gmt_plot for more details on these options\n\n'];
            
        case 3
            
            test.data = gridded.data-200;  % for difference
            test.lat = gridded.lat;
            test.lon = gridded.lon;
            test.unit='';
            test.xunit='kg/m@+2@+';
            test.ctable.name = 'mypolar';
            test.ctable.nwhite = 6;
            test.ctable.reference = 0;
            test.legend.tick_spacing = 25;
            test.legend.sidebar = 2; % above only
            test.psbasemap.axes = 'Nesw';

            test.fontsize_title = '20p';
            test.title = 'Diff plot,nwhite=6,sidebar=2 (above only),title';
            test.filename = sprintf('Test%g',T);
            
            %this puts standardize_geodata.m to the test
            test.data=test.data'; test.data = test.data(end:-1:1,end:-1:1);
            test.lat = test.lat(end:-1:1);
            test.lon = test.lon(end:-1:1);
            
            testThis = catstruct(rmfield(in,'test'),test);
            files{ind} = gmt_plot(testThis);
            
            info{ind} = ['Test:' num2str(T) ' demonstrates that you can change the number of intervals that are',...
                'colored white around the reference value when using the mypolar-colortable,',...
                'here in.nwhite = 6. You can also change the number of data intervals ',...
                ', change in.fontsize_title (here in.fontsize_title = 20)',...
                'or decide to only have the sidebar for values above the datarange (here in.sidebar=2)\n\n'];
            
        case 4
            
            %% TEST: title, unit (gridded and ungridded)
            test.nearneighbor.resolution = 1; %[deg]
            test.unit            = 'kg/m@+2@+';
            test.psbasemap.axes  = 'nEsW';
            test.title           = sprintf('Font: ''Courier'': ungridded: %g [deg]',test.nearneighbor.resolution);
            test.font_title      = 'Courier';
            test.pscoast.width   = '1p';
            test.pscoast.color   = '255/255/255';
            test.legend.orientation  = 'h';
            test.filename          = sprintf('Test%g',T);
            
            testThis = catstruct(ungridded,rmfield(in,'test'),test);
            files{ind} = gmt_plot(testThis);
            
            info{ind} = ['Test:' num2str(T) ' demonstrates the same data as Test1a except in an ungridded form.',...
                'Test 2a is chosen to show that the user can decide how far to look for the next point',...
                'The search radius, where the values within are used to calculate the grid value,',...
                'is automatically in.nearneighbor * 1.5',...
                'isequal(size(in.data),size(in.lat),size(in.lon))==1. This is e.g. good',...
                'for plotting satellite granule data directly. You can also change the',...
                'color of the coastlines, e.g. in.coast.color = ''255/255/255.''\n\n'];
            
        case 5
            %% test 2b
            test.nearneighbor.resolution = 10;
            test.unit            = 'kg/m@+2@+';
            test.psbasemap.axes    = 'nEsW';
            test.pscoast.width     = '1p';
            test.pscoast.color     = '255/255/255';
            test.legend.orientation  = 'h';
            test.font_title   = 8;
            test.title        = sprintf('Font: 8. ungridded: %.0f [deg]',test.nearneighbor.resolution);
            test.filename        = sprintf('Test%g',T);
            
            testThis = catstruct(ungridded,rmfield(in,'test'),test);
            files{ind} = gmt_plot(testThis);
            
            info{ind} = ['Test:' num2str(T) ' is chosen to show that the user can decide how far to look for the next point',...
                'The search radius, where the values within are used to calculate the grid value,',...
                'is automatically set to in.nearneighbor.resolution * 1.5',...
                'isequal(size(in.data),size(in.lat),size(in.lon))==1. This is e.g. good',...
                'for plotting satellite granule data directly. You can also change the',...
                'color of the coastlines, e.g. in.coast.color = ''255/255/255.''\n\n'];
            
        case 6
            %% test 2c
            test.nearneighbor.resolution = 60;
            test.unit            = 'kg/m@+2@+';
            test.pscoast.width     = '1p';
            test.pscoast.color     = '255/255/255';
            test.legend.orientation  = 'h';
            test.font_title   = 'Times-Roman';
            test.psbasemap.axes    = 'neSW';
            test.title        = sprintf('ungridded: resolution = %g [deg]',test.nearneighbor.resolution);
            test.legend.orientation  = 'v';
            test.filename        = sprintf('Test%g',T);
            testThis = catstruct(ungridded,rmfield(in,'test'),test);
            files{ind} = gmt_plot(testThis);
            
            info{ind} = ['Test:' num2str(T) ' is chosen to show that the user can decide how far to look for the next point',...
                'The search radius, where the values within are used to calculate the grid value,',...
                'is automatically set to in.nearneighbor.resolution * 1.5',...
                'isequal(size(in.data),size(in.lat),size(in.lon))==1. This is e.g. good',...
                'for plotting satellite granule data directly. You can also change the',...
                'color of the coastlines, e.g. in.coast.color = ''255/255/255.''\n\n'];
            
        case 7
            
            %% TEST: sidebar, nan, projection, center
            
            gridded.data(20:40,20:80) = NaN;
            test.datarange = [160,250];
            test.title         = 'upper sidebar, NaN, Hammer projection, center=180';
            test.legend.orientation = 'h';
            test.proj.projection         =  'H';
            test.proj.center             = 180;      % Center map at given center longitude
            test.filename             = sprintf('Test%g',T);
            
            testThis = catstruct(gridded,rmfield(in,'test'),test);
            files{ind} = gmt_plot(testThis);
            
            info{ind} = ['Test:' num2str(T) ' demonstrates the Hammer, Robinsson and sinusoidal projections ',...
                '(using in.projection = ''H''/''R''/''I''), ',...
                'one of many available projections. The test also demonstrates that you can',...
                'easily change the center of the projection to e.g. 180E using in.center = 180.',...
                'The grey region in the map indicates NaN values, and an extra NaN color box',...
                'is automatically added below the vertical legend. You can suppress the extra',...
                'NaN-legend using in.nanlegend = false. The upper sidebar is automatically shown',...
                'if there are data larger than in.datarange(2), given via in.datarange = [v1,v2].',...
                'The color for these values can be changed using',...
                'in.color_foreground (default is ''255/255/255'' (white))',...
                'Test ' num2str(T) ' also shows the use of in.legend.tick_spacing = default,10,15 for a,b,c respectively.\n\n'];
            
        case 8
            
            gridded.data(20:40,20:80) = NaN;
            test.datarange = [160,250];
            test.nlevels        = 18;
            test.legend.orientation = 'h';
            test.title               = 'Robinsson projection, center = 0 ';
            test.proj.projection          = 'N';
            test.proj.center              = 0;
            test.legend.tick_spacing = 10;
            test.extra_legend        = struct('name','nan','type','nan');
            test.filename              = sprintf('Test%g',T);
            
            testThis = catstruct(gridded,rmfield(in,'test'),test);
            files{ind} = gmt_plot(testThis);
            
            info{ind} = ['Test:' num2str(T) ' demonstrates the Hammer, Robinsson and sinusoidal projections ',...
                '(using in.projection = ''H''/''R''/''I''), ',...
                'one of many available projections. The test also demonstrates that you can',...
                'easily change the center of the projection to e.g. 180E using in.center = 180.',...
                'The grey region in the map indicates NaN values, and an extra NaN color box',...
                'is automatically added below the vertical legend. You can suppress the extra',...
                'NaN-legend using in.nanlegend = false. The upper sidebar is automatically shown',...
                'if there are data larger than in.datarange(2), given via in.datarange = [v1,v2].',...
                'The color for these values can be changed using',...
                'in.color_foreground (default is ''255/255/255'' (white))',...
                'Test ' num2str(T) ' also shows the use of in.legend.tick_spacing = default,10,15 for a,b,c respectively.\n\n'];
            
        case 9
            
            gridded.data(20:40,20:80) = NaN;
            test.legend.orientation   = 'h';
            test.datarange            = [160,250];% displayed value range. Sidebar triangles shown for data outside range.
            test.extra_legend         = struct('name','nan','type','nan','position','0i/0i/2c/2ch','fontsize','25p');
            test.legend.position      = '12c/0c/20c/2ch';
            test.legend.fontsize     = '20p';
            test.legend.tick_spacing  = 15;
            test.title                = 'Sinusoidal projection, center = 90';
            test.proj.projection      = 'I';
            test.proj.center          = 90;
            test.filename               = sprintf('Test%g',T);
            
            testThis = catstruct(gridded,rmfield(in,'test'),test);
            files{ind} = gmt_plot(testThis);
            
            info{ind} = ['Test:' num2str(T) ' demonstrates the Hammer, Robinsson and sinusoidal projections ',...
                '(using in.projection = ''H''/''R''/''I''), ',...
                'one of many available projections. The test also demonstrates that you can',...
                'easily change the center of the projection to e.g. 180E using in.center = 180.',...
                'The grey region in the map indicates NaN values, and an extra NaN color box',...
                'is automatically added below the vertical legend. You can suppress the extra',...
                'NaN-legend using in.nanlegend = false. The upper sidebar is automatically shown',...
                'if there are data larger than in.datarange(2), given via in.datarange = [v1,v2].',...
                'The color for these values can be changed using',...
                'in.color_foreground (default is ''255/255/255'' (white))',...
                'Test ' num2str(T) ' also shows the use of in.legend.tick_spacing = default,10,15 for a,b,c respectively.\n\n'];
            
        case 10
            %% TEST: contours, nlevels, annotation format, xunit
            
            test.lat = -90+.5:90-.5;
            test.lon = -180+.5:180-.5;
            for j = test.lat
                for i = test.lon
                    test.data(j+90.5,i+180.5) = ((0.1*sin((j+91)/18*pi))+sin((j+91)/180*pi))*(0.1*cos(i/36*pi)+cos(i/360*pi));
                end
            end
            
            test.title = 'contours, HSV colours, horiz. legend';
            test.proj.projection = 'H';
            test.legend.tick_annotation_format = '%g';
            test.legend.orientation = 'h';
            test.contourline.spacing=0.1;
            test.contourline.range=[0 1];
            test.contourline.linethick=1;
            test.contourline.more='-T1c/0.001c:LH';
            test.datarange = [0, 1];
            test.color_background = '255/0/0';
            test.color_foreground = '255/255/0';
            test.colorrange.colors = {{0,'0/1/1'},{1,'60/1/1'}};
            test.colorrange.color_model = 'HSV';
                        
            % TEST pstext aswell
            test.pstext = struct('text',{'Norrland','Australia'},...
                'lat',{67+51/60,-25},'lon',{20+13/60,130},'angle',{0,-15},...
                'color','0/255/0','fontsize','10p','justify','C');
            
            test.filename = sprintf('Test%g',T);
            
            testThis = catstruct(rmfield(in,'test'),test);
            files{ind} = gmt_plot(testThis);
            
            info{ind} = ['Test:' num2str(T) ' demonstrates contour maps (see contour options in help gmt_plot), ',...
                'where you can optionally choose to show highs and lows in the of the data.',...
                'It also demonstrated that you can have x and y-labels',...
                '(in.xunit=''no unit either'' and in.unit=''no unit'') around the legend',...
                'Using the custom colortable (in.colorrange) you can also use',...
                'in.colorrange.color_model = ''HSV'' instead of the default ''RGB''. I also test'...
                'plotting text on top of the map using pstext\n\n'];
            
        case 11
            
            test.lat        = (-30:30)-.5;
            test.lon        = (152:240)-.5;
            test.data       = gridded.data(...
                gridded.lat>=test.lat(1)&gridded.lat<=test.lat(end),...
                (gridded.lon+180)>=test.lon(1)&(gridded.lon+180)<=test.lon(end));
            test.filename     = sprintf('Test%g',T);
            test.title       = 'Region that spans the dateline';
%            test.legend.xpos= 20.7;
            
            testThis = catstruct(rmfield(in,'test'),test);
            files{ind} = gmt_plot(testThis);
            
            info{ind}       = ['Test:' num2str(T) ' demonstrates data that crosses the dateline, but is not global\n\n'];
            
            
        case 12
            %% TEST: custom color table, fontsize_title, map_width, grid lines, region
            
            % Add some annotations some intervals.
            if ~isfield(in,{'stepsize','nlevels'})
                test.nlevels = 20; %
                
                tmp = repmat({''},1,test.nlevels);
                tmp{1} = 'blue'; tmp{6} = 'light blue'; tmp{10} = 'green'; tmp{15} = 'black'; tmp{20} = 'red';
                test.legend.tick_annotations = tmp;
            end
            test.title = 'custom legend and sub-region';

            % and plot minor lines (e.g 20 xaxis & yaxis)
            test.fontsize_title = '20p';              % (GMT default = 36p)
            test.proj.map_width = '17i';              % (default is 9 inches)
            test.region    = '-130/130/-30/30'; % 'lonmin/lonmax/latmin/latmax'
            test.psbasemap.axes = 'NeSW';
            test.psbasemap.ticks = '30g30/15g15'; % annotations (deg) on major lines (e.g 40 xaxis,20 yaxis)
            test.legend.tick_centering = true;
            
            % CUSTOM COLOR TABLE ---
            % For your own custom colortable use in.colorrange.colors (see below). Assign a
            % color to a relative value, e.g between 0-1, where 0 is for the minimum of the
            % datarange and 1 is for the color of the maximum datarange. For example,
            test.colorrange.colors  = {{0,'0/0/255'},...
                {0.3,'255/255/255'},...
                {0.5,'0/255/0'},...
                {.7,'0/0/0'},...
                {1,'255/0/0'}};
            test.colorrange.color_model = 'RGB';
            % makes a colortable that goes from blue to white to green to black to red in RGB format.
            % ------------------------
            
            test.filename = sprintf('Test%g',T);
            
            testThis = catstruct(gridded, rmfield(in,'test'),test);
            files{ind} = gmt_plot(testThis);
            
            info{ind} = ['Test:' num2str(T) ' demonstrates that you can easily make your own custom colortable using in.colorrange.',...
                'You can also insert your own legend annotations to be used instead of',...
                'data values using in.legend.tick_annotations. You can also change in.mapwidth,',...
                'in.fontsize_title, the tick spacing for the axis using in.ticks,',...
                'and in this test its demonstrated how to confine the figure to certain',...
                'geographical regions using in.region = ''lonmin/lonmax/latmin/latmax''',...
                'The legend is also automatically made horizontal if the base/height > 3',...
                'This can easily be overridden by setting in.orientation.orientation = ''v'' \n\n'];
            
        case 13
            
            [names,lats,lons,textcolor,color,textalign,textsize,shape,sze] = satGroupLocations;
            
            
            for i = 1:length(names)
                test.locations(i).lat = lats(i);
                test.locations(i).lon = lons(i);

                test.locations(i).symbol = shape{i};
                test.locations(i).symbolsize = sze{i};
                test.locations(i).symbolcolor = color{i};
                test.locations(i).text = names{i};
                test.locations(i).color = textcolor{i};
                test.locations(i).fontsize = textsize{i};
                test.locations(i).justify = textalign{i};
            end
            test.psbasemap.axes = 'neSW';
            test.title = 'SatGroup locations';
            test.nodata = true;
            test.region = '-20/160/-30/70';
            test.filename = sprintf('Test%g',T);
            
            testThis = catstruct(rmfield(in,'test'),test);
            files{ind} = gmt_plot(testThis);
            
            info{ind} = ['Test' num2str(T) ' demonstrates that you can plot points and text on a map using',...
                'in.locations (see help gmt_plot), and demonstrates that there are',...
                'several map resolutions available.',...
                'The available file types in.figuretype are ''pdf'',''eps'',''png'', and ''tif''\n\n'];
            
        case 14
            
            [names,lats,lons,textcolor,color,textalign,~,shape,sze] = satGroupLocations;
            
            for i = 1:length(names)
                test.locations(i).lat = lats(i);
                test.locations(i).lon = lons(i);
                test.locations(i).text = names{i};
                test.locations(i).symbol = shape{i};
                test.locations(i).symbolsize = sze{i};
                test.locations(i).symbolcolor = color{i};
                test.locations(i).fontsize = '15p';
                test.locations(i).color= textcolor{i};
                test.locations(i).justify = textalign{i};
            end
            test.psbasemap.axes = 'neSW';
            test.nodata = true;
            test.region     = '-10/30/45/70';
            test.title      = 'SatGroup locations: Europe';
            test.filename = sprintf('Test%g',T);
            
            testThis = catstruct(rmfield(in,'test'),test);
            files{ind} = gmt_plot(testThis);
            
            info{ind} = ['Test' num2str(T) ' demonstrates that you can plot points and text on a map using',...
                'in.locations (see help gmt_plot), and demonstrates that there are',...
                'several map resolutions available.',...
                'The available file types in.figuretype are ''pdf'',''eps'',''png'', and ''tif''\n\n'];
            
        case 15
            
            [names,lats,lons,textcolor,color,textalign,~,shape,sze] = satGroupLocations;
            
            for i = 1:length(names)
                test.locations(i).lat = lats(i);
                test.locations(i).lon = lons(i);
                test.locations(i).text = names{i};
                test.locations(i).symbol = shape{i};
                test.locations(i).symbolsize = sze{i};
                test.locations(i).symbolcolor = color{i};
                test.locations(i).fontsize = '15p';
                test.locations(i).color= textcolor{i};
                test.locations(i).justify = textalign{i};
            end
            test.psbasemap.axes = 'neSW';
            test.nodata = true;
            
            
            % TEST: smallregion
            test.title = 'SatGroup locations: Kiruna';
            test.region = '16/22/66.2/70';
            test.pscoast.rivers = 'a';
            test.filename = sprintf('Test%g',T);
            
            testThis = catstruct(rmfield(in,'test'),test);
            files{ind} = gmt_plot(testThis);
            
            info{ind} = ['Test' num2str(T) ' demonstrates that you can plot points and text on a map using',...
                'in.locations (see help gmt_plot), and demonstrates that there are',...
                'several map resolutions available.',...
                'The available file types in.figuretype are ''pdf'',''eps'',''png'', and ''tif''\n\n'];
            
        case 16
            %% TEST: pspolygon

            test = regionSetUp_for_GMT;
            test.nodata=true;
            test.psbasemap.axes='nwes';
            test.proj.projection = 'Q';
            
            test.filename = sprintf('Test%g',T);
            
            testThis = catstruct(rmfield(in,'test'),test);
            files{ind} = gmt_plot(testThis);
            
            info{ind} = ['Test:' num2str(T) ' demonstrates that you can draw polygons on a map, using in.pspoly.',...
                'E.g. useful if you want to draw regions of interest on a map\n\n'];
            
        case 17
            [~,extra] = getPredefinedRegions({'westerlies','twp','trop_cont','sc_regs','polar'});

            test.lat = extra{1}.mask.lat;
            test.lon = extra{1}.mask.lon;
            test.data = false(length(test.lon),length(test.lat));
            for i = 1:length(extra)
                test.data = test.data | extra{i}.mask.mask;
            end

            test.psbasemap.axes='nWeS';
            test.filename = sprintf('Test%g',T);
            test.legend.tick_centering = true;
            test.legend.box_spacing = 1;
            test.legend.tick_annotations = {'not region','region'};

            testThis = catstruct(rmfield(in,'test'),test);
            files{ind} = gmt_plot(testThis);
            
            info{ind} = ['Test:', num2str(T) ' demonstrates a fancy pscale, which can come in handy'];
        otherwise
            error(['atmlab:' mfilename],'Test %g has no entry',T)
    end
    ind = ind+1;
    clear test
end

end

function [names,lats,lons,textcolor,color,textalign,textsize,shape,sze] = satGroupLocations()
% List of home towns for people from the SatGroup

names    = {'Kiruna','Brisbane','Stuttgart','Bremen','Idukki',...
    'Oberstdorf','Amsterdam','Lauchhammer','Freiburg','Hyderabad'};

% Missing: Eskilstuna, Huskvarna

lats     = [67.85 -27.48 48.83 53.1 9.9 47.3 52.4 51.5 48 17.36];
lons     = [20.216 153 9.2 8 77 10.22 4.77 13.77 7.85 78.48];

textcolor= repmat({'black'},1,length(names)); textcolor{1} = 'red';
color    = repmat({'green'},1,length(names)); color{3} = 'yellow';
textalign= {'LM','RB','LM','LB','CB','LT','RM','LM','RM','RM'};
textsize = {'15p','15p','7p','7p','15p','7p','7p','7p','7p','7p'};
shape    = repmat({'c'},1,length(names));shape{3} = 'd';
sze      = repmat({'.08p'},1,length(names)); sze{3} = '.1p';

end

function [gridded,ungridded] = test_generate_data_sin_cos(sz)
%% test_generate_data_sin_cos

if ~exist('sz','var')
    sz = 1;
end

gridded.lat    = -90+sz/2:sz:90-sz/2;
gridded.lon    = -180+sz/2:sz:180-sz/2;
gridded.data   = zeros(length(gridded.lat),length(gridded.lon));
ungridded.data = zeros(length(gridded.lat)*length(gridded.lon),1);
ungridded.lon  = zeros(size(ungridded.data));
ungridded.lat  = zeros(size(ungridded.data));

ln = length(gridded.lon);
lt = length(gridded.lat);
for j=0:lt-1
    for i=0:ln-1
        gridded.data(j+1,i+1)=sin(((i+lt)/ln)*pi+pi/2)* cos((j/lt)*pi+pi/2)*100+160;
    end
end

if nargout==2
    ungridded.data = gridded.data;
    ungridded.lat = repmat(gridded.lat',1,length(gridded.lon));
    ungridded.lon = repmat(gridded.lon,length(gridded.lat),1);
end

end

function [gmtinput,options] = regionSetUp_for_GMT(options)
%% regionSetUp_for_GMT
%
% Makes the structure to be passed to GMT to plot the regions with or without
% data overlay
%
% INPUT
%      options (structure)
%           regionstr = {'reg1','reg2',etc}        ; mandatory
%           color     = a 3xlength(regionstr) array; opt
%           fontsize  = 10p (size of region labels) ; opt
%           thick     = a 1xlength(regionstr) array; opt 
%
% USAGE: region = regionSetUp_for_GMT(struct('regionstr',{{'trop_cont'}}));
%
% NOTE: If you only want to plot the regions with no data overlay, include
%       in.nodata= true.
%       e.g.
%       file = gmt_plot(catstruct(regionSetUp_for_GMT(struct('regionstr',{{'trop_cont','twp'}})),struct('nodata',true)))
% $Id: test_gmt.m 10678 2017-09-29 09:57:06Z seliasson $

if ~nargin, options = struct; end

% REGIONS FROM eliasson11:assessing_acp
default.regionstr   = {'westerlies','twp','trop_cont','sc_regs','polar'};
options             = optargs_struct(options,default);
default.color       = list_colors('Ncolors',length(options.regionstr));
default.fontsize    = '10p';
options             = optargs_struct(options,default);
default.thick       = repmat({'2p'},1,length(options.regionstr)); % this is textsize
options             = optargs_struct(options,default);
clear default

% I need these to be strings
col = cell(1,length(options.color));
for i = 1:length(options.color)
    col{i} = sprintf('%g/%g/%g',255*options.color{i});
end
options.color = col; clear col

lat=[];lon=[];text=[];textcolor=[];i=1;
for r = 1:length(options.regionstr)
    R = options.regionstr{r};
    tmp = [];
    switch R
            
        case {'westerlies','swl','ant','arc','polar'}
            tmp = [tmp;veryAnnoyingSideeffect(R)];
            [~,extra] = getPredefinedRegions(R);
        otherwise
            [regin,extra] = getPredefinedRegions(R);
            tmp = [tmp;polygoninize_regions(regin)];
    end
    
    lat = [lat,{extra{1}.center.lat}];
    lon = [lon,{extra{1}.center.lon}];
    
    
    for k = 1:length(tmp)
        gmtinput.pspoly.coord{i} = tmp{k};
        gmtinput.pspoly.color{i} = options.color{r};
        gmtinput.pspoly.thick{i} = options.thick{r};
        
        i = i+1;
    end
    text = [text,repmat({R},1,length(extra{1}.center))];
    textcolor = [textcolor,repmat(options.color(r),1,length(extra{1}.center))];
    clear extra
end

gmtinput.pstext= struct('text',text,'lat',lat,'lon',lon,...
    'fontsize',options.fontsize,'color',textcolor);

assert( (length(gmtinput.pspoly.color)==1 || length(gmtinput.pspoly.color)==length(gmtinput.pspoly.coord)),...
    'There are %d destinct reigions and I need as many colors. recieved %d',...
    length(gmtinput.pspoly.coord),length(gmtinput.pspoly.color))

end

function reg = veryAnnoyingSideeffect(R)

% all high latitude regions that go from -180 to 180
reg = {};
switch R
    
    case 'westerlies'
        tmp = getPredefinedRegions('nwl');
        reg = [polygoninize_regions(tmp);veryAnnoyingSideeffect('swl')'];
    case 'swl'
        reg{end+1}= [-180  -60; -180 -45];reg{end+1}= [-180  -45; -80 -45];
        reg{end+1}= [-80  -45; -80 -60];reg{end+1}= [-80  -60; -180 -60];
        
        % splitting it up into 2 horisontal legs
        reg{end+1}= [-50  -60; -50 -45];reg{end+1}= [-50  -45; 60 -45];
        reg{end+1}= [60  -45; 180 -45];reg{end+1}= [180  -45; 180 -60];
        
        reg{end+1}= [180  -60; 60 -60];reg{end+1}= [60  -60; -50 -60];
    
    case 'ant'
        reg{end+1}= [-180  -90; -180 -70];reg{end+1}= [-180  -70; 0 -70];
        reg{end+1}= [0  -70; 180 -70];reg{end+1}= [180  -70; 180 -90];
        reg{end+1}= [180  -90; 0 -90];reg{end+1}= [0 -90; -180,-90]; 
        
    case 'arc'
        reg{end+1}= [-180  90; -180 70];reg{end+1}= [-180  70; 0 70];
        reg{end+1}= [0  70; 180 70];reg{end+1}= [180  70; 180 90];
        reg{end+1}= [180  90; 0 90];reg{end+1}= [0 90; -180,90]; 
        
    case 'polar'
        reg = [veryAnnoyingSideeffect('ant')';veryAnnoyingSideeffect('arc')'];
    otherwise
        disp('Nothing coded for this')
end

end
