
      COMPLEX FUNCTION REFWAT( WL, TEMP)

c       Calculates complex refractive index of pure liquid water for
c       wavelengths between 0.01 microns and 10.0 m.  Temperature
c       dependence is only considered for wavelengths greater
c       than 10.0 microns.

c      I N P U T :  WL   = wavelength (microns)
c                   TEMP = temperature (K) -- for WL.GT.10.0 only

c    O U T P U T :  REFWAT = complex refractive index
c                            ( with positive imag. part )

c    METHOD :

c      for wavelengths less then 10. microns :
c          tabular interpolation assuming real index and 
c          log( imaginary index ) linear in log( wavelength ).

c      for wavelengths 10.0 - 20.0 microns :
c          weighted data correction using Ray's model to account
c          for temperature dependence

c      for wavelengths 20.0 - 1.e7 microns :
c          data correction using Ray's model to account for 
C          temperature dependence

c      for wavelengths greater then 1.e7 microns :
c          Ray's analytic fits based on some theories of Debye

c    REFERENCES :

c      (1) for 0.01 - 1.e7 microns :  Segelstein, D., 1981:
c             "The Complex Refractive Index of Water", M.S. Thesis,
c             University of Missouri--Kansas City

c      (2) for 10. - 1.e7 microns:  Ray, P., 1972:  Broadband Complex
c             Refractive Indices of Ice and Water, Appl. Opt. 11,
c             1836-1844

c      (There is a new reference, Wieliczka, D. et al., Appl. Opt.
c       28, 1714-1719, 1989, with some updated data for the IR)
c ---------------------------------------------------------------------

c            *** Specifications of local variables

c      FRAC   :  quantity between 0 and 1 used for interpolation
c      RRE    :  real part of refractive index
c      RIM    :  imag part of refractive index
c      T3IM   :  temporay imag part of refractive index at 300K
c      T3RE   :  temporary real part of refractive index at 300K
c      CORRE  :  correction to real index
c      CORIM  :  correction to imag. index
c      WD     :  weight factor for data points
c      WR     :  weight factor for Ray model
c      WLMIN  :  minimum tabulated wavelength
c      WLMAX  :  maximum tabulated wavelength
c ---------------------------------------------------------------------

c     .. Parameters ..

      INTEGER   NREF
      PARAMETER ( NREF = 1261)
c     ..
c     .. Scalar Arguments ..

      REAL      TEMP, WL
c     ..
c     .. Local Scalars ..

      INTEGER   I
      REAL      CORIM, CORRE, FRAC, RIM, RIMD, RRE, RRED, T3IM, T3RE,
     &          WLMAX, WLMIN
      COMPLEX   RAY3, RAYTEM
c     ..
c     .. External Functions ..

      COMPLEX   RAYWAT
      EXTERNAL  RAYWAT
c     ..
c     .. External Subroutines ..

      EXTERNAL  ERRMSG
c     ..
c     .. Intrinsic Functions ..

      INTRINSIC AIMAG, LOG, CMPLX, REAL
c     ..
c     .. Common blocks ..
c                               ** refractive index data

      COMMON    / SEGEL / WLTAB, REREF, IMREF

      REAL      IMREF( NREF), REREF( NREF), WLTAB( NREF)
c     ..
c     .. Data statements ..

      DATA      WLMIN, WLMAX / 0.01, 1.E+8 /
c     ..


      IF( WL.LT.WLMIN .OR. WL.GT.WLMAX )
     &    CALL ERRMSG( 'REFWAT--wavelength out of table range',.TRUE.)


      IF( WL.LT.10.0 ) THEN
c                                          ** use tabular interpolation
         DO 10 I = 2, NREF

            IF( WL.LE.WLTAB(I) ) GO TO 20

   10    CONTINUE


   20    CONTINUE
         FRAC   = LOG( WL / WLTAB(I - 1)) /
     &            LOG( WLTAB(I) / WLTAB(I - 1))
         RRED   = REREF( I - 1) + FRAC*( REREF(I) - REREF(I - 1))
         RIMD   = IMREF( I - 1)*( IMREF(I) / IMREF(I - 1))**FRAC
         REFWAT = CMPLX( RRED, RIMD)


      ELSE IF( WL.GE.10.0 .AND. WL.LE.1.E7) THEN

c                                            ** temperature correction
c                                            ** for data using Ray model
         RAY3   = RAYWAT( WL, 298.)
         T3RE   = REAL( RAY3)
         T3IM   = AIMAG( RAY3)
         RAYTEM = RAYWAT( WL, TEMP)
         CORRE  = REAL( RAYTEM) - T3RE
         CORIM  = AIMAG( RAYTEM) - T3IM

c                                          ** use tabular interpolation
         DO 30 I = 2, NREF

            IF( WL.LE.WLTAB(I)) GO TO 40

   30    CONTINUE


   40    CONTINUE
         FRAC   = LOG( WL / WLTAB(I-1) ) /
     &            LOG( WLTAB(I) / WLTAB(I-1) )
         RRED   = REREF(I-1) + FRAC*( REREF(I) - REREF(I-1) )
         RIMD   = IMREF(I-1) * ( IMREF(I) / IMREF(I-1) )**FRAC


         IF( WL.GE.10.0 .AND. WL.LT.20.0) THEN
c                                                    ** weighted average

            RRE    = ( (WL - 10) / 10.)*CORRE + RRED
            RIM    = ( (WL - 10) / 10.)*CORIM + RIMD

         ELSE IF( WL.GE.20.0 .AND. WL.LE.1.E7) THEN

            RRE    = RRED + CORRE
            RIM    = RIMD + CORIM

         END IF


         REFWAT = CMPLX( RRE, RIM)


      ELSE IF( WL.GT.1.E7) THEN
c                                                  ** use Ray model
         REFWAT = RAYWAT( WL, TEMP)

      END IF

      END

      COMPLEX FUNCTION RAYWAT( WL, TEMP)

c       Calculates complex refractive index of pure liquid water for
c       wavelengths between 2.0 microns and 10.0 m.

c    I N P U T :  WL   = wavelength ( 2 microns to 1.e7 microns )
c                 TEMP = temperature (K)

c    O U T P U T :  RAYWAT = complex refractive index
c                            ( with positive imag. part )

c    METHOD : Ray's analytic fits based on some theories of Debye

c    REFERENCE : Ray, P., 1972:  Broadband Complex Refractive Indices
c                  of Ice and Water, Appl. Opt. 11, 1836-1844

c ---------------------------------------------------------------------

c            *** Specifications of local variables

c      EPSILS :  static dielectric constant
c                   ( epsilon-sub-s, Ray Eq. 4 )
c      EPSINF :  high-frequency dielectric const
c                   ( epsilon-sub-s, Ray Eq. 7a )
c      EPSRE, :  real and imaginary parts of dielectric constant
c        EPSIM      ( Ray Eqs. 5,6 )
c      PI     :  3.14159...
c      AB     :  summation terms, Ray Eq. 8
c      ARRE   :  correction to real index for WL < 6 micron
c      BRRE   :  correction to imag. index
c      RRE    :  real part of refractive index
c      RIM    :  imag part of refractive index
c      TC     :  Centigrade temperature
c      TERM   :  summation terms, Ray Eq. 9
c      TBARP1 :  TBAR + 1, coefficient in Ray Eq. 9
c      WLCM   :  wavelength in cm
c      WLDEBY :  wavelength above which to apply Debye theory
c      WLMIN  :  minimum tabulated wavelength
c      WLMAX  :  maximum tabulated wavelength
c      WLS    :  relaxation wavelength in cm
c                   ( lambda-sub-s, Ray Eq. 7C )
c      ALPHA  :  temporary storage variables
c ---------------------------------------------------------------------

c     .. Scalar Arguments ..

      REAL      TEMP, WL
c     ..
c     .. Local Scalars ..

      REAL      ALPHA, BET, DEL, EPSILS, EPSINF, FREC, GAM, PI, RIM,
     &          RRE, TC, WLCEN, WLCM, WLMAX, WLMIN, WLS, WVL
      COMPLEX   CEPS, CI
c     ..
c     .. External Subroutines ..

      EXTERNAL  ERRMSG
c     ..
c     .. Intrinsic Functions ..

      INTRINSIC ABS, AIMAG, LOG10, ASIN, CMPLX, COS, EXP,
     &          REAL, SIN, SQRT
c     ..
c     .. Statement Functions ..

      REAL      AB, ARRE, BRRE, TBARP1, TERM
c     ..
c     .. Save statement ..

      SAVE      PI
c     ..
c     .. Data statements ..

      DATA      PI / 0.0 /, CI / ( 0.0, 1.0) /
      DATA      WLMIN, WLMAX / 1.0, 1.E+8 /
c     ..
c     .. Statement Function definitions ..

      AB( WVL, BET, WLCEN, DEL, GAM) = BET*
     &   EXP( - ABS(LOG10(WVL/WLCEN)/DEL)**GAM)

      TBARP1( TC, WVL) = 1.+ 1.E-4*( TC - 25.)*EXP( (WVL/4.)**0.25)

      TERM( WVL, FREC, BET, GAM) = 
     &    BET*( FREC**2 - (1.E4/WVL)**2) /
     &       ( (FREC**2 - (1.E4/WVL)**2)**2 + GAM*(1.E4/WVL)**2)

      BRRE( TC, WVL) = TBARP1( TC, WVL) *
     &       SQRT( 1.83899 + TERM(WVL, 1639.0,  52340.4,  10399.2)
     &                     + TERM(WVL, 588.24, 345005.0, 259913.0)
     &                     + TERM(WVL, 161.29,  43319.7,  27661.2) )

      ARRE( TC, WVL) = TBARP1( TC, WVL)*
     &       SQRT( 1.79907 + TERM(WVL, 3352.27, 999140.0,  151963.0) 
     &                     + TERM(WVL, 1639.00,  50483.5,    9246.27) 
     &                     + TERM(WVL,  588.24, 844697.0, 1076150.0) )
c     ..


      IF( WL.LT.WLMIN .OR. WL.GT.WLMAX )
     &    CALL ERRMSG( 'RAYWAT--wavelength out of table range',.TRUE.)

      IF( PI.EQ.0.0 ) PI = 2.*ASIN( 1.0)

c                       *** use Ray's fits to Debye theory expressions

      TC     = TEMP - 273.
      EPSINF = 5.27137 + TC*( 0.0216474 - 0.00131198*TC)
      EPSILS = 78.54*( 1. + (TC - 25.)*
     &         (- 4.579E-3 + (TC - 25.)*(1.19E-5 - 2.8E-8*(TC -
     &         25.))))

c                                               *** Ray Eq. 7B
      ALPHA  = - 16.8129 / TEMP + 0.0609265
      WLCM   = 1.E-4*WL
      WLS    = 3.3836E-4*EXP( 2514. / TEMP)
      CEPS   = EPSINF + ( EPSILS - EPSINF) /
     &         ( 1.0 + (COS(0.5*PI*(1. - ALPHA)) + CI*SIN(0.5*PI*(1. -
     &         ALPHA)))*(WLS/WLCM)**(1. - ALPHA)) - 0.00666667*CI*WLCM

c                             ** complex refractive index from Cole-
c                             ** Cole extension to Debye theory

      RAYWAT = SQRT( CEPS)
      RIM    = - AIMAG( RAYWAT)
      RRE    = REAL( RAYWAT)
c                                  ** corrections to imag. index to
c                                  ** account for absorption bands
c                                  ** ( Ray Eq. 8 + Table 2 )

      IF( WL.LT.3000. .AND. WL.GT.300.) THEN

         RIM    = RIM + AB( WL, 0.25, 300., 0.47, 3.0) +
     &            AB( WL, 0.41, 62., 0.35, 1.7) +
     &            AB( WL, 0.39, 17., 0.45, 1.3)

      ELSE IF( WL.LE.300. .AND. WL.GT.62.) THEN

         RIM    = RIM + AB( WL, 0.25, 300., 0.40, 2.0) +
     &            AB( WL, 0.41, 62., 0.35, 1.7) +
     &            AB( WL, 0.39, 17., 0.45, 1.3)

      ELSE IF( WL.LE.62. .AND. WL.GT.17.) THEN

         RIM    = RIM + AB( WL, 0.25, 300., 0.40, 2.0) +
     &            AB( WL, 0.41, 62., 0.22, 1.8) +
     &            AB( WL, 0.39, 17., 0.45, 1.3)

      ELSE IF( WL.LE.17. .AND. WL.GT.6.1) THEN

         RIM    = RIM + AB( WL, 0.12, 6.1, 0.042, 0.6) +
     &            AB( WL, 0.39, 17., 0.165, 2.4) +
     &            AB( WL, 0.41, 62., 0.22, 1.8)

      ELSE IF( WL.LE.6.1 .AND. WL.GT.4.95) THEN

         RIM    = RIM + AB( WL, 0.12, 6.1, 0.009, 2.0) +
     &            AB( WL, 0.01, 4.95, 0.05, 1.0)

      ELSE IF( WL.LE.4.95 .AND. WL.GT.2.95) THEN

         RIM    = RIM + AB( WL, 0.27, 2.97, 0.04, 2.0) +
     &            AB( WL, 0.01, 4.95, 0.06, 1.0)

      ELSE IF( WL.LE.2.95) THEN

         RIM    = RIM + AB( WL, 0.27, 2.97, 0.025, 2.0) +
     &            AB( WL, 0.01, 4.95, 0.06, 1.0)

      END IF
c                                  ** corrections to real index
c                                  ** ( Ray Eq. 9 + Table 3 )

      IF( WL.LT.1000. .AND. WL.GT.340) THEN

         RRE    = RRE*( (WL - 340.) / 660.) +
     &            BRRE( TC, WL)*( (1000. - WL) / 660.)

      ELSE IF( WL.LE.340. .AND. WL.GT.7.0) THEN

         RRE    = BRRE( TC, WL)

      ELSE IF( WL.LE.7. .AND. WL.GT.6.) THEN

         RRE    = ARRE( TC, WL)*( 7. - WL) + BRRE( TC, WL)*( WL - 6.)

      ELSE IF( WL.LE.6.0) THEN

         RRE    = ARRE( TC, WL)

      END IF

      RAYWAT = CMPLX( RRE, RIM)

      END

      BLOCK DATA  QDATA

c         Complex refractive indices of pure water

c     REFERENCE:  Segelstein, D., 1981:
c                "The Complex Refractive Index of Water", M.S. Thesis,
c                 University of Missouri--Kansas City

c     .. Parameters ..

      INTEGER   NREF
      PARAMETER ( NREF = 1261)
c     ..
c     .. Local Scalars ..

      INTEGER   I
c     ..
c     .. Common blocks ..

      COMMON    / SEGEL / WLTAB, REREF, IMREF

      REAL      IMREF( NREF), REREF( NREF), WLTAB( NREF)
c     ..
c     .. Data statements ..

c                                          ** wavelength in microns
      DATA (WLTAB(I), I = 1, 90 ) /
     &.10000E-01,.10990E-01,.11995E-01,.13002E-01,.13996E-01,.15996E-01,
     &.17989E-01,.19999E-01,.21979E-01,.23988E-01,.26002E-01,.27990E-01,
     &.29992E-01,.31989E-01,.33963E-01,.35975E-01,.38019E-01,.39994E-01,
     &.41976E-01,.43954E-01,.46026E-01,.47973E-01,.50003E-01,.52000E-01,
     &.53951E-01,.55976E-01,.57943E-01,.59979E-01,.61944E-01,.63973E-01,
     &.66069E-01,.68077E-01,.69984E-01,.71945E-01,.73961E-01,.76033E-01,
     &.77983E-01,.79983E-01,.82035E-01,.83946E-01,.85901E-01,.87902E-01,
     &.89950E-01,.92045E-01,.93972E-01,.95940E-01,.97949E-01,.10000E+00,
     &.10209E+00,.10399E+00,.10593E+00,.10789E+00,.10990E+00,.11194E+00,
     &.11402E+00,.11588E+00,.11803E+00,.11995E+00,.12190E+00,.12388E+00,
     &.12589E+00,.12794E+00,.13002E+00,.13213E+00,.13397E+00,.13614E+00,
     &.13804E+00,.13996E+00,.14191E+00,.14388E+00,.14588E+00,.14791E+00,
     &.14997E+00,.15205E+00,.15417E+00,.15596E+00,.15812E+00,.15996E+00,
     &.16218E+00,.16406E+00,.16596E+00,.16788E+00,.16982E+00,.17219E+00,
     &.17418E+00,.17498E+00,.17989E+00,.18493E+00,.19011E+00,.19498E+00/
      DATA (WLTAB(I), I = 91, 180 ) /
     &.19999E+00,.20512E+00,.20989E+00,.21478E+00,.21979E+00,.22491E+00,
     &.23014E+00,.23496E+00,.23988E+00,.24491E+00,.25003E+00,.25527E+00,
     &.26002E+00,.26485E+00,.26977E+00,.27479E+00,.27990E+00,.28510E+00,
     &.28973E+00,.29512E+00,.29992E+00,.30479E+00,.30974E+00,.31477E+00,
     &.31989E+00,.32509E+00,.33037E+00,.33497E+00,.33963E+00,.34514E+00,
     &.34995E+00,.35481E+00,.35975E+00,.36475E+00,.36983E+00,.37497E+00,
     &.38019E+00,.38459E+00,.38994E+00,.39537E+00,.39994E+00,.40458E+00,
     &.41020E+00,.41495E+00,.41976E+00,.42462E+00,.42954E+00,.43451E+00,
     &.43954E+00,.44463E+00,.44978E+00,.45499E+00,.46026E+00,.46452E+00,
     &.46989E+00,.47534E+00,.47973E+00,.48529E+00,.48978E+00,.49545E+00,
     &.50003E+00,.50466E+00,.51050E+00,.51523E+00,.52000E+00,.52481E+00,
     &.52966E+00,.53456E+00,.53951E+00,.54450E+00,.54954E+00,.55463E+00,
     &.55976E+00,.56494E+00,.57016E+00,.57544E+00,.57943E+00,.58479E+00,
     &.59020E+00,.59566E+00,.59979E+00,.60534E+00,.60954E+00,.61518E+00,
     &.61944E+00,.62517E+00,.62951E+00,.63533E+00,.63973E+00,.64565E+00/
      DATA (WLTAB(I), I = 181, 270 ) /
     &.65013E+00,.65464E+00,.66069E+00,.66527E+00,.66988E+00,.67453E+00,
     &.68077E+00,.68549E+00,.69024E+00,.69502E+00,.69984E+00,.70469E+00,
     &.70958E+00,.71450E+00,.71945E+00,.72444E+00,.72946E+00,.73451E+00,
     &.73961E+00,.74473E+00,.74989E+00,.75509E+00,.76033E+00,.76560E+00,
     &.76913E+00,.77446E+00,.77983E+00,.78524E+00,.79068E+00,.79433E+00,
     &.79983E+00,.80538E+00,.80910E+00,.81470E+00,.82035E+00,.82414E+00,
     &.82985E+00,.83560E+00,.83946E+00,.84528E+00,.84918E+00,.85507E+00,
     &.85901E+00,.86497E+00,.87096E+00,.87498E+00,.87902E+00,.88512E+00,
     &.88920E+00,.89536E+00,.89950E+00,.90573E+00,.90991E+00,.91411E+00,
     &.92045E+00,.92470E+00,.92897E+00,.93541E+00,.93972E+00,.94406E+00,
     &.95060E+00,.95499E+00,.95940E+00,.96605E+00,.97051E+00,.97499E+00,
     &.97949E+00,.98401E+00,.99083E+00,.99541E+00,.10000E+01,.10093E+01,
     &.10209E+01,.10304E+01,.10399E+01,.10495E+01,.10593E+01,.10691E+01,
     &.10789E+01,.10889E+01,.10990E+01,.11092E+01,.11194E+01,.11298E+01,
     &.11402E+01,.11508E+01,.11588E+01,.11695E+01,.11803E+01,.11912E+01/
      DATA (WLTAB(I), I = 271, 360 ) /
     &.11995E+01,.12106E+01,.12190E+01,.12303E+01,.12388E+01,.12503E+01,
     &.12589E+01,.12706E+01,.12794E+01,.12912E+01,.13002E+01,.13092E+01,
     &.13213E+01,.13305E+01,.13397E+01,.13490E+01,.13614E+01,.13709E+01,
     &.13804E+01,.13900E+01,.13996E+01,.14093E+01,.14191E+01,.14289E+01,
     &.14388E+01,.14488E+01,.14588E+01,.14689E+01,.14791E+01,.14894E+01,
     &.14997E+01,.15101E+01,.15205E+01,.15311E+01,.15417E+01,.15488E+01,
     &.15596E+01,.15704E+01,.15812E+01,.15885E+01,.15996E+01,.16106E+01,
     &.16218E+01,.16293E+01,.16406E+01,.16482E+01,.16596E+01,.16711E+01,
     &.16788E+01,.16904E+01,.16982E+01,.17100E+01,.17219E+01,.17298E+01,
     &.17418E+01,.17498E+01,.17620E+01,.17701E+01,.17783E+01,.17906E+01,
     &.17989E+01,.18113E+01,.18197E+01,.18281E+01,.18408E+01,.18493E+01,
     &.18621E+01,.18707E+01,.18793E+01,.18880E+01,.19011E+01,.19099E+01,
     &.19187E+01,.19320E+01,.19409E+01,.19498E+01,.19588E+01,.19679E+01,
     &.19815E+01,.19907E+01,.19999E+01,.20091E+01,.20184E+01,.20277E+01,
     &.20417E+01,.20512E+01,.20606E+01,.20701E+01,.20797E+01,.20893E+01/
      DATA (WLTAB(I), I = 361, 450 ) /
     &.20989E+01,.21086E+01,.21184E+01,.21281E+01,.21380E+01,.21478E+01,
     &.21577E+01,.21677E+01,.21777E+01,.21878E+01,.21979E+01,.22080E+01,
     &.22182E+01,.22284E+01,.22387E+01,.22491E+01,.22594E+01,.22699E+01,
     &.22803E+01,.22909E+01,.23014E+01,.23121E+01,.23174E+01,.23281E+01,
     &.23388E+01,.23496E+01,.23605E+01,.23714E+01,.23823E+01,.23878E+01,
     &.23988E+01,.24099E+01,.24210E+01,.24322E+01,.24378E+01,.24491E+01,
     &.24604E+01,.24717E+01,.24774E+01,.24889E+01,.25003E+01,.25119E+01,
     &.25177E+01,.25293E+01,.25410E+01,.25527E+01,.25645E+01,.25704E+01,
     &.25763E+01,.25823E+01,.25882E+01,.25942E+01,.26062E+01,.26122E+01,
     &.26182E+01,.26242E+01,.26303E+01,.26363E+01,.26485E+01,.26546E+01,
     &.26607E+01,.26669E+01,.26730E+01,.26792E+01,.26853E+01,.26977E+01,
     &.27040E+01,.27102E+01,.27164E+01,.27227E+01,.27290E+01,.27416E+01,
     &.27479E+01,.27542E+01,.27606E+01,.27669E+01,.27797E+01,.27861E+01,
     &.27925E+01,.27990E+01,.28119E+01,.28184E+01,.28249E+01,.28314E+01,
     &.28379E+01,.28510E+01,.28576E+01,.28642E+01,.28708E+01,.28840E+01/
      DATA (WLTAB(I), I = 451, 540 ) /
     &.28907E+01,.28973E+01,.29040E+01,.29174E+01,.29242E+01,.29309E+01,
     &.29376E+01,.29512E+01,.29580E+01,.29648E+01,.29785E+01,.29854E+01,
     &.29992E+01,.30479E+01,.30974E+01,.31477E+01,.31989E+01,.32509E+01,
     &.33037E+01,.33497E+01,.33963E+01,.34514E+01,.34995E+01,.35481E+01,
     &.35975E+01,.36475E+01,.36983E+01,.37497E+01,.38019E+01,.38459E+01,
     &.38994E+01,.39537E+01,.39994E+01,.40458E+01,.41020E+01,.41495E+01,
     &.41976E+01,.42462E+01,.42954E+01,.43451E+01,.43954E+01,.44463E+01,
     &.44978E+01,.45499E+01,.46026E+01,.46452E+01,.46989E+01,.47534E+01,
     &.47973E+01,.48529E+01,.48978E+01,.49545E+01,.50003E+01,.50466E+01,
     &.51050E+01,.51523E+01,.52000E+01,.52481E+01,.52966E+01,.53456E+01,
     &.53951E+01,.54450E+01,.54954E+01,.55463E+01,.55976E+01,.56494E+01,
     &.57016E+01,.57544E+01,.57943E+01,.58479E+01,.59020E+01,.59566E+01,
     &.59979E+01,.60534E+01,.60954E+01,.61518E+01,.61944E+01,.62517E+01,
     &.62951E+01,.63533E+01,.63973E+01,.64565E+01,.65013E+01,.65464E+01,
     &.66069E+01,.66527E+01,.66988E+01,.67453E+01,.68077E+01,.68549E+01/
      DATA (WLTAB(I), I = 541, 630 ) /
     &.69024E+01,.69502E+01,.69984E+01,.70469E+01,.70958E+01,.71450E+01,
     &.71945E+01,.72444E+01,.72946E+01,.73451E+01,.73961E+01,.74473E+01,
     &.74989E+01,.75509E+01,.76033E+01,.76560E+01,.76913E+01,.77446E+01,
     &.77983E+01,.78524E+01,.79068E+01,.79433E+01,.79983E+01,.80538E+01,
     &.80910E+01,.81470E+01,.82035E+01,.82414E+01,.82985E+01,.83560E+01,
     &.83946E+01,.84528E+01,.84918E+01,.85507E+01,.85901E+01,.86497E+01,
     &.87096E+01,.87498E+01,.87902E+01,.88512E+01,.88920E+01,.89536E+01,
     &.89950E+01,.90573E+01,.90991E+01,.91411E+01,.92045E+01,.92470E+01,
     &.92897E+01,.93541E+01,.93972E+01,.94406E+01,.95060E+01,.95499E+01,
     &.95940E+01,.96605E+01,.97051E+01,.97499E+01,.97949E+01,.98401E+01,
     &.99083E+01,.99541E+01,.10000E+02,.10046E+02,.10093E+02,.10139E+02,
     &.10209E+02,.10257E+02,.10304E+02,.10351E+02,.10399E+02,.10447E+02,
     &.10495E+02,.10544E+02,.10593E+02,.10641E+02,.10691E+02,.10740E+02,
     &.10789E+02,.10839E+02,.10889E+02,.10940E+02,.10990E+02,.11041E+02,
     &.11092E+02,.11143E+02,.11194E+02,.11246E+02,.11298E+02,.11350E+02/
      DATA (WLTAB(I), I = 631, 720 ) /
     &.11402E+02,.11455E+02,.11508E+02,.11561E+02,.11588E+02,.11641E+02,
     &.11695E+02,.11749E+02,.11803E+02,.11858E+02,.11912E+02,.11940E+02,
     &.11995E+02,.12050E+02,.12106E+02,.12162E+02,.12190E+02,.12246E+02,
     &.12303E+02,.12359E+02,.12388E+02,.12445E+02,.12503E+02,.12560E+02,
     &.12589E+02,.12647E+02,.12706E+02,.12764E+02,.12794E+02,.12853E+02,
     &.12912E+02,.12942E+02,.13002E+02,.13062E+02,.13092E+02,.13152E+02,
     &.13213E+02,.13243E+02,.13305E+02,.13335E+02,.13397E+02,.13459E+02,
     &.13490E+02,.13552E+02,.13614E+02,.13646E+02,.13709E+02,.13740E+02,
     &.13804E+02,.13836E+02,.13900E+02,.13964E+02,.13996E+02,.14060E+02,
     &.14093E+02,.14158E+02,.14191E+02,.14256E+02,.14289E+02,.14355E+02,
     &.14388E+02,.14454E+02,.14488E+02,.14555E+02,.14588E+02,.14655E+02,
     &.14689E+02,.14757E+02,.14791E+02,.14859E+02,.14894E+02,.14962E+02,
     &.14997E+02,.15066E+02,.15101E+02,.15136E+02,.15205E+02,.15241E+02,
     &.15311E+02,.15346E+02,.15417E+02,.15453E+02,.15488E+02,.15560E+02,
     &.15596E+02,.15668E+02,.15704E+02,.15740E+02,.15812E+02,.15849E+02/
      DATA (WLTAB(I), I = 721, 810 ) /
     &.15885E+02,.15959E+02,.15996E+02,.16032E+02,.16106E+02,.16144E+02,
     &.16218E+02,.16255E+02,.16293E+02,.16368E+02,.16406E+02,.16444E+02,
     &.16482E+02,.16558E+02,.16596E+02,.16634E+02,.16711E+02,.16749E+02,
     &.16788E+02,.16866E+02,.16904E+02,.16943E+02,.16982E+02,.17061E+02,
     &.17100E+02,.17140E+02,.17219E+02,.17258E+02,.17298E+02,.17338E+02,
     &.17418E+02,.17458E+02,.17498E+02,.17539E+02,.17620E+02,.17660E+02,
     &.17701E+02,.17742E+02,.17783E+02,.17865E+02,.17906E+02,.17947E+02,
     &.17989E+02,.18030E+02,.18113E+02,.18155E+02,.18197E+02,.18239E+02,
     &.18281E+02,.18365E+02,.18408E+02,.18450E+02,.18493E+02,.18535E+02,
     &.18621E+02,.18664E+02,.18707E+02,.18750E+02,.18793E+02,.18836E+02,
     &.18880E+02,.18967E+02,.19011E+02,.19055E+02,.19099E+02,.19143E+02,
     &.19187E+02,.19231E+02,.19320E+02,.19364E+02,.19409E+02,.19454E+02,
     &.19498E+02,.19543E+02,.19588E+02,.19634E+02,.19679E+02,.19770E+02,
     &.19815E+02,.19861E+02,.19907E+02,.19953E+02,.19999E+02,.20989E+02,
     &.21979E+02,.23014E+02,.23988E+02,.25003E+02,.26002E+02,.26977E+02/
      DATA (WLTAB(I), I = 811, 900 ) /
     &.27990E+02,.28973E+02,.29992E+02,.30974E+02,.31989E+02,.33037E+02,
     &.33963E+02,.34995E+02,.35975E+02,.36983E+02,.38019E+02,.38994E+02,
     &.39994E+02,.41020E+02,.41976E+02,.42954E+02,.43954E+02,.44978E+02,
     &.46026E+02,.46989E+02,.47973E+02,.48978E+02,.50003E+02,.52000E+02,
     &.53951E+02,.55976E+02,.57943E+02,.59979E+02,.61944E+02,.63973E+02,
     &.66069E+02,.68077E+02,.69984E+02,.71945E+02,.73961E+02,.76033E+02,
     &.77983E+02,.79983E+02,.82035E+02,.83946E+02,.85901E+02,.87902E+02,
     &.89950E+02,.92045E+02,.93972E+02,.95940E+02,.97949E+02,.10000E+03,
     &.10990E+03,.11995E+03,.13002E+03,.13996E+03,.14997E+03,.15996E+03,
     &.16982E+03,.17989E+03,.19011E+03,.19999E+03,.20989E+03,.21979E+03,
     &.23014E+03,.23988E+03,.25003E+03,.26002E+03,.26977E+03,.27990E+03,
     &.28973E+03,.29992E+03,.31989E+03,.33963E+03,.35975E+03,.38019E+03,
     &.39994E+03,.41976E+03,.43954E+03,.46026E+03,.47973E+03,.50003E+03,
     &.52000E+03,.53951E+03,.55976E+03,.57943E+03,.59979E+03,.65013E+03,
     &.69984E+03,.74989E+03,.79983E+03,.84918E+03,.89950E+03,.95060E+03/
      DATA (WLTAB(I), I = 901, 990 ) /
     &.10000E+04,.11000E+04,.12001E+04,.12999E+04,.13999E+04,.15000E+04,
     &.15999E+04,.16998E+04,.18001E+04,.19002E+04,.19999E+04,.20999E+04,
     &.21999E+04,.22999E+04,.23999E+04,.24998E+04,.26002E+04,.27002E+04,
     &.28003E+04,.29000E+04,.29999E+04,.31003E+04,.31996E+04,.32999E+04,
     &.34002E+04,.35003E+04,.36000E+04,.37000E+04,.38001E+04,.39003E+04,
     &.40004E+04,.41002E+04,.41995E+04,.43003E+04,.44005E+04,.44999E+04,
     &.46004E+04,.47000E+04,.47995E+04,.49000E+04,.50003E+04,.51004E+04,
     &.52000E+04,.53003E+04,.54001E+04,.55005E+04,.56002E+04,.57003E+04,
     &.57996E+04,.59007E+04,.60007E+04,.60996E+04,.62001E+04,.62994E+04,
     &.64003E+04,.64998E+04,.65993E+04,.67004E+04,.67999E+04,.68992E+04,
     &.70000E+04,.71007E+04,.71995E+04,.72996E+04,.73995E+04,.75007E+04,
     &.75998E+04,.77002E+04,.78001E+04,.78995E+04,.80000E+04,.81001E+04,
     &.81999E+04,.83000E+04,.84000E+04,.85000E+04,.86000E+04,.87000E+04,
     &.87999E+04,.89000E+04,.89999E+04,.91000E+04,.92000E+04,.92999E+04,
     &.94000E+04,.94999E+04,.96000E+04,.97000E+04,.98000E+04,.99000E+04/
      DATA (WLTAB(I), I = 991, 1080 ) /
     &.10000E+05,.11000E+05,.12000E+05,.12999E+05,.13999E+05,.15000E+05,
     &.15999E+05,.16998E+05,.18001E+05,.19002E+05,.19999E+05,.20999E+05,
     &.21999E+05,.22999E+05,.23999E+05,.24998E+05,.26002E+05,.27002E+05,
     &.28003E+05,.29000E+05,.29999E+05,.31003E+05,.31996E+05,.32999E+05,
     &.34002E+05,.35003E+05,.36000E+05,.37000E+05,.38001E+05,.39003E+05,
     &.40004E+05,.41002E+05,.41995E+05,.43003E+05,.44005E+05,.44999E+05,
     &.46004E+05,.47000E+05,.47995E+05,.49000E+05,.50003E+05,.51004E+05,
     &.52000E+05,.53003E+05,.54001E+05,.55005E+05,.56002E+05,.57003E+05,
     &.57996E+05,.59007E+05,.60007E+05,.60996E+05,.62001E+05,.62994E+05,
     &.64003E+05,.64998E+05,.65993E+05,.67004E+05,.67999E+05,.68992E+05,
     &.70000E+05,.71007E+05,.71995E+05,.72996E+05,.73995E+05,.75007E+05,
     &.75998E+05,.77002E+05,.78001E+05,.78995E+05,.80002E+05,.81003E+05,
     &.81997E+05,.83004E+05,.84004E+05,.84996E+05,.86000E+05,.86996E+05,
     &.88004E+05,.89002E+05,.89991E+05,.90991E+05,.92003E+05,.93004E+05,
     &.93994E+05,.94995E+05,.96006E+05,.97006E+05,.97994E+05,.98992E+05/
      DATA (WLTAB(I), I = 1081, 1170 ) /
     &.10000E+06,.10990E+06,.11995E+06,.13002E+06,.13996E+06,.14997E+06,
     &.15996E+06,.16982E+06,.17989E+06,.19011E+06,.19999E+06,.20989E+06,
     &.21979E+06,.23014E+06,.23988E+06,.25003E+06,.26002E+06,.26977E+06,
     &.27990E+06,.28973E+06,.29992E+06,.30974E+06,.31989E+06,.33037E+06,
     &.33963E+06,.34995E+06,.35975E+06,.36983E+06,.38019E+06,.38994E+06,
     &.39994E+06,.41020E+06,.41976E+06,.42954E+06,.43954E+06,.44978E+06,
     &.46026E+06,.46989E+06,.47973E+06,.48978E+06,.50003E+06,.51050E+06,
     &.52000E+06,.52966E+06,.53951E+06,.54954E+06,.55976E+06,.57016E+06,
     &.57943E+06,.59020E+06,.59979E+06,.60954E+06,.61944E+06,.62951E+06,
     &.63973E+06,.65013E+06,.66069E+06,.66988E+06,.68077E+06,.69024E+06,
     &.69984E+06,.70958E+06,.71945E+06,.72946E+06,.73961E+06,.74989E+06,
     &.76033E+06,.76913E+06,.77983E+06,.79068E+06,.79983E+06,.80910E+06,
     &.82035E+06,.82985E+06,.83946E+06,.84918E+06,.85901E+06,.87096E+06,
     &.87902E+06,.88920E+06,.89950E+06,.90991E+06,.92045E+06,.92897E+06,
     &.93972E+06,.95060E+06,.95940E+06,.97051E+06,.97949E+06,.99083E+06/
      DATA (WLTAB(I), I = 1171, 1261 ) /
     &.10000E+07,.10990E+07,.11995E+07,.13002E+07,.13996E+07,.14997E+07,
     &.15996E+07,.16982E+07,.17989E+07,.19011E+07,.19999E+07,.20989E+07,
     &.21979E+07,.23014E+07,.23988E+07,.25003E+07,.26002E+07,.26977E+07,
     &.27990E+07,.28973E+07,.29992E+07,.30974E+07,.31989E+07,.33037E+07,
     &.33963E+07,.34995E+07,.35975E+07,.36983E+07,.38019E+07,.38994E+07,
     &.39994E+07,.41020E+07,.41976E+07,.42954E+07,.43954E+07,.44978E+07,
     &.46026E+07,.46989E+07,.47973E+07,.48978E+07,.50003E+07,.51050E+07,
     &.52000E+07,.52966E+07,.53951E+07,.54954E+07,.55976E+07,.57016E+07,
     &.57943E+07,.59020E+07,.59979E+07,.60954E+07,.61944E+07,.62951E+07,
     &.63973E+07,.65013E+07,.66069E+07,.66988E+07,.68077E+07,.69024E+07,
     &.69984E+07,.70958E+07,.71945E+07,.72946E+07,.73961E+07,.74989E+07,
     &.76033E+07,.76913E+07,.77983E+07,.79068E+07,.79983E+07,.80910E+07,
     &.82035E+07,.82985E+07,.83946E+07,.84918E+07,.85901E+07,.87096E+07,
     &.87902E+07,.88920E+07,.89950E+07,.90991E+07,.92045E+07,.92897E+07,
     &.93972E+07,.95060E+07,.95940E+07,.97051E+07,.97949E+07,.99083E+07,
     &.10000E+08/

c                                        ** real refractive indices
      DATA (REREF(I), I = 1, 90 ) /
     &  0.968416,  0.964778,  0.960953,  0.956954,  0.952792,  0.944124,
     &  0.934744,  0.924583,  0.913973,  0.902694,  0.890837,  0.878766,
     &  0.866493,  0.854141,  0.842171,  0.830720,  0.819753,  0.809997,
     &  0.802291,  0.797737,  0.797007,  0.805579,  0.820742,  0.830957,
     &  0.835240,  0.835295,  0.831628,  0.830901,  0.840575,  0.866994,
     &  0.903527,  0.941801,  0.981692,  1.020921,  1.049744,  1.068724,
     &  1.087685,  1.111682,  1.140628,  1.173382,  1.214969,  1.259495,
     &  1.302663,  1.346760,  1.387639,  1.425425,  1.455868,  1.476628,
     &  1.493473,  1.506677,  1.516305,  1.523589,  1.528933,  1.535363,
     &  1.543211,  1.548070,  1.553435,  1.560870,  1.570304,  1.584638,
     &  1.606068,  1.626822,  1.633849,  1.619420,  1.586268,  1.536403,
     &  1.496271,  1.471129,  1.461485,  1.460977,  1.469275,  1.489551,
     &  1.521276,  1.559942,  1.596861,  1.620422,  1.641473,  1.650184,
     &  1.652917,  1.653100,  1.647245,  1.635062,  1.605555,  1.568183,
     &  1.549412,  1.543062,  1.513343,  1.491881,  1.475183,  1.462543/
      DATA (REREF(I), I = 91, 180 )/
     &  1.451724,  1.442296,  1.434685,  1.427828,  1.421603,  1.415921,
     &  1.410702,  1.406358,  1.402321,  1.398535,  1.394993,  1.391674,
     &  1.388881,  1.386239,  1.383726,  1.381341,  1.379072,  1.376902,
     &  1.375086,  1.373098,  1.371437,  1.369839,  1.368287,  1.366812,
     &  1.365376,  1.363990,  1.362616,  1.361513,  1.360441,  1.359231,
     &  1.358224,  1.357247,  1.356295,  1.355370,  1.354470,  1.353594,
     &  1.352740,  1.352046,  1.351231,  1.350438,  1.349793,  1.349159,
     &  1.348417,  1.347811,  1.347219,  1.346636,  1.346066,  1.345505,
     &  1.344956,  1.344418,  1.343889,  1.343368,  1.342858,  1.342455,
     &  1.341961,  1.341475,  1.341093,  1.340620,  1.340248,  1.339791,
     &  1.339430,  1.339073,  1.338635,  1.338288,  1.337944,  1.337607,
     &  1.337273,  1.336943,  1.336615,  1.336292,  1.335972,  1.335656,
     &  1.335344,  1.335035,  1.334729,  1.334425,  1.334200,  1.333902,
     &  1.333609,  1.333316,  1.333100,  1.332813,  1.332598,  1.332317,
     &  1.332106,  1.331826,  1.331619,  1.331345,  1.331144,  1.330877/
      DATA (REREF(I), I = 181, 270 )/
     &  1.330683,  1.330490,  1.330238,  1.330052,  1.329869,  1.329690,
     &  1.329452,  1.329278,  1.329106,  1.328938,  1.328769,  1.328603,
     &  1.328440,  1.328279,  1.328120,  1.327963,  1.327808,  1.327652,
     &  1.327502,  1.327350,  1.327201,  1.327055,  1.326909,  1.326764,
     &  1.326667,  1.326524,  1.326382,  1.326244,  1.326104,  1.326012,
     &  1.325874,  1.325739,  1.325648,  1.325512,  1.325379,  1.325290,
     &  1.325157,  1.325025,  1.324937,  1.324805,  1.324718,  1.324590,
     &  1.324502,  1.324373,  1.324244,  1.324159,  1.324074,  1.323946,
     &  1.323859,  1.323732,  1.323648,  1.323520,  1.323434,  1.323351,
     &  1.323222,  1.323138,  1.323054,  1.322926,  1.322842,  1.322757,
     &  1.322630,  1.322546,  1.322462,  1.322333,  1.322249,  1.322165,
     &  1.322080,  1.321994,  1.321866,  1.321780,  1.321695,  1.321521,
     &  1.321303,  1.321128,  1.320952,  1.320775,  1.320596,  1.320416,
     &  1.320233,  1.320051,  1.319865,  1.319678,  1.319488,  1.319296,
     &  1.319103,  1.318909,  1.318763,  1.318566,  1.318366,  1.318162/
      DATA (REREF(I), I = 271, 360 )/
     &  1.318008,  1.317799,  1.317641,  1.317427,  1.317263,  1.317042,
     &  1.316873,  1.316645,  1.316470,  1.316233,  1.316052,  1.315868,
     &  1.315618,  1.315425,  1.315228,  1.315031,  1.314760,  1.314547,
     &  1.314329,  1.314104,  1.313871,  1.313671,  1.313518,  1.313373,
     &  1.313220,  1.313055,  1.312888,  1.312715,  1.312525,  1.312318,
     &  1.312093,  1.311852,  1.311604,  1.311352,  1.311097,  1.310923,
     &  1.310659,  1.310387,  1.310114,  1.309928,  1.309642,  1.309352,
     &  1.309055,  1.308855,  1.308548,  1.308341,  1.308021,  1.307672,
     &  1.307435,  1.307073,  1.306829,  1.306453,  1.306070,  1.305809,
     &  1.305413,  1.305142,  1.304727,  1.304442,  1.304155,  1.303718,
     &  1.303418,  1.302947,  1.302616,  1.302269,  1.301709,  1.301291,
     &  1.300633,  1.300214,  1.299860,  1.299545,  1.298998,  1.298791,
     &  1.298793,  1.298681,  1.298590,  1.298472,  1.298308,  1.298051,
     &  1.297607,  1.297292,  1.296913,  1.296499,  1.296066,  1.295606,
     &  1.294919,  1.294457,  1.293973,  1.293476,  1.292966,  1.292438/
      DATA (REREF(I), I = 361, 450 )/
     &  1.291899,  1.291353,  1.290795,  1.290221,  1.289634,  1.289033,
     &  1.288418,  1.287787,  1.287139,  1.286474,  1.285790,  1.285087,
     &  1.284365,  1.283619,  1.282852,  1.282064,  1.281256,  1.280421,
     &  1.279561,  1.278675,  1.277755,  1.276797,  1.276305,  1.275295,
     &  1.274257,  1.273184,  1.272062,  1.270902,  1.269682,  1.269059,
     &  1.267787,  1.266450,  1.265059,  1.263577,  1.262824,  1.261297,
     &  1.259683,  1.257969,  1.257102,  1.255347,  1.253465,  1.251445,
     &  1.250383,  1.248127,  1.245672,  1.243014,  1.240167,  1.238670,
     &  1.237089,  1.235426,  1.233679,  1.231834,  1.227769,  1.225483,
     &  1.223082,  1.220535,  1.218078,  1.215699,  1.209954,  1.206519,
     &  1.202951,  1.199289,  1.195340,  1.191390,  1.188087,  1.179962,
     &  1.174582,  1.168874,  1.160993,  1.152876,  1.149520,  1.142068,
     &  1.136183,  1.132860,  1.131711,  1.132778,  1.130913,  1.127959,
     &  1.127558,  1.129478,  1.128413,  1.125532,  1.125351,  1.127523,
     &  1.133346,  1.142386,  1.145545,  1.152284,  1.162372,  1.178446/
      DATA (REREF(I), I = 451, 540 )/
     &  1.185419,  1.195889,  1.208002,  1.229654,  1.240033,  1.252073,
     &  1.263935,  1.285942,  1.297762,  1.307891,  1.326310,  1.334533,
     &  1.352917,  1.411876,  1.452013,  1.466753,  1.461522,  1.449409,
     &  1.432585,  1.417064,  1.404875,  1.393260,  1.384213,  1.376092,
     &  1.368863,  1.362546,  1.356937,  1.351891,  1.347393,  1.343958,
     &  1.340174,  1.336658,  1.333929,  1.331403,  1.328504,  1.326183,
     &  1.323997,  1.321906,  1.319948,  1.318113,  1.316398,  1.314920,
     &  1.313587,  1.312483,  1.311785,  1.311588,  1.311451,  1.311148,
     &  1.310657,  1.309721,  1.308720,  1.307228,  1.305885,  1.304258,
     &  1.301965,  1.299910,  1.297550,  1.294933,  1.292117,  1.289015,
     &  1.285729,  1.282194,  1.278291,  1.273883,  1.268802,  1.262994,
     &  1.256584,  1.248370,  1.242239,  1.234896,  1.229289,  1.231892,
     &  1.242862,  1.268459,  1.295314,  1.330121,  1.341605,  1.339863,
     &  1.335754,  1.329242,  1.325038,  1.320468,  1.317726,  1.314837,
     &  1.311404,  1.309021,  1.306716,  1.304521,  1.301901,  1.300125/
      DATA (REREF(I), I = 541, 630 )/
     &  1.298382,  1.296751,  1.295193,  1.293609,  1.292093,  1.290696,
     &  1.289296,  1.287944,  1.286624,  1.285242,  1.283912,  1.282606,
     &  1.281248,  1.279895,  1.278508,  1.277123,  1.276220,  1.274794,
     &  1.273363,  1.271952,  1.270543,  1.269613,  1.268163,  1.266657,
     &  1.265652,  1.264125,  1.262564,  1.261488,  1.259903,  1.258240,
     &  1.257072,  1.255384,  1.254220,  1.252405,  1.251193,  1.249353,
     &  1.247433,  1.246095,  1.244791,  1.242789,  1.241424,  1.239322,
     &  1.237862,  1.235657,  1.234142,  1.232659,  1.230259,  1.228589,
     &  1.226967,  1.224439,  1.222699,  1.220909,  1.218113,  1.216115,
     &  1.214136,  1.211068,  1.208909,  1.206729,  1.204471,  1.202228,
     &  1.198600,  1.195932,  1.193164,  1.190334,  1.187365,  1.183900,
     &  1.180893,  1.178360,  1.174182,  1.170827,  1.167354,  1.163960,
     &  1.160584,  1.157248,  1.153843,  1.150368,  1.146959,  1.143601,
     &  1.140345,  1.137372,  1.134419,  1.131445,  1.128640,  1.125466,
     &  1.122010,  1.118841,  1.116059,  1.113289,  1.110334,  1.107674/
      DATA (REREF(I), I = 631, 720 )/
     &  1.105361,  1.103057,  1.100705,  1.097503,  1.096584,  1.096068,
     &  1.094339,  1.092339,  1.090622,  1.089062,  1.086474,  1.086163,
     &  1.087480,  1.087926,  1.087993,  1.086723,  1.087212,  1.089721,
     &  1.090913,  1.091270,  1.092375,  1.095643,  1.098011,  1.099603,
     &  1.100816,  1.104624,  1.107403,  1.108999,  1.110319,  1.114243,
     &  1.116753,  1.118262,  1.122067,  1.124841,  1.126485,  1.130583,
     &  1.133825,  1.135773,  1.139515,  1.141428,  1.145850,  1.149628,
     &  1.151643,  1.156338,  1.160150,  1.161869,  1.165763,  1.167947,
     &  1.172049,  1.174089,  1.178513,  1.182458,  1.184740,  1.189086,
     &  1.191399,  1.195603,  1.197623,  1.201594,  1.203552,  1.207465,
     &  1.209428,  1.213645,  1.215328,  1.218762,  1.220973,  1.225566,
     &  1.227627,  1.231631,  1.233597,  1.237500,  1.239445,  1.243348,
     &  1.245318,  1.249380,  1.251704,  1.253631,  1.256977,  1.258880,
     &  1.263173,  1.265082,  1.268440,  1.270391,  1.272559,  1.276473,
     &  1.278233,  1.282639,  1.284709,  1.286576,  1.290576,  1.292723/
      DATA (REREF(I), I = 721, 810 )/
     &  1.294706,  1.298872,  1.301310,  1.303145,  1.306556,  1.308540,
     &  1.313112,  1.315327,  1.317122,  1.320901,  1.322675,  1.324631,
     &  1.326773,  1.330870,  1.333056,  1.334869,  1.338869,  1.341074,
     &  1.342949,  1.347481,  1.349696,  1.351233,  1.352834,  1.356772,
     &  1.359301,  1.361083,  1.364655,  1.367219,  1.369211,  1.370751,
     &  1.374519,  1.376758,  1.378598,  1.380029,  1.383660,  1.385875,
     &  1.387734,  1.389417,  1.390838,  1.394313,  1.396377,  1.398169,
     &  1.399826,  1.401123,  1.404604,  1.406786,  1.408657,  1.410419,
     &  1.412092,  1.415276,  1.417548,  1.419809,  1.421557,  1.422820,
     &  1.426178,  1.428308,  1.429982,  1.431240,  1.432797,  1.434643,
     &  1.435881,  1.439563,  1.441618,  1.442846,  1.444197,  1.445486,
     &  1.446666,  1.447502,  1.450255,  1.452188,  1.453825,  1.455604,
     &  1.456898,  1.457713,  1.458719,  1.459690,  1.460391,  1.463349,
     &  1.465400,  1.466543,  1.467000,  1.467249,  1.467642,  1.483693,
     &  1.499422,  1.516402,  1.529309,  1.537967,  1.544080,  1.546670/
      DATA (REREF(I), I = 811, 900 )/
     &  1.546272,  1.542658,  1.535500,  1.527225,  1.519076,  1.511879,
     &  1.505906,  1.498932,  1.492960,  1.486740,  1.481006,  1.478232,
     &  1.476571,  1.475642,  1.477194,  1.480747,  1.485266,  1.491543,
     &  1.499424,  1.508821,  1.520272,  1.531473,  1.542270,  1.567492,
     &  1.594131,  1.619157,  1.643739,  1.669053,  1.690223,  1.709762,
     &  1.729441,  1.747333,  1.762824,  1.777162,  1.790800,  1.805539,
     &  1.819110,  1.830882,  1.842330,  1.851943,  1.859854,  1.867327,
     &  1.874242,  1.880545,  1.886330,  1.891384,  1.895435,  1.899131,
     &  1.907505,  1.911671,  1.919973,  1.927412,  1.934154,  1.941655,
     &  1.948419,  1.955736,  1.965156,  1.974559,  1.983438,  1.992287,
     &  2.001418,  2.010446,  2.020318,  2.029224,  2.037243,  2.045135,
     &  2.052476,  2.059773,  2.073976,  2.086956,  2.099543,  2.112811,
     &  2.125742,  2.139507,  2.153213,  2.166254,  2.177335,  2.188736,
     &  2.200349,  2.210869,  2.220374,  2.228339,  2.236685,  2.254575,
     &  2.270109,  2.290196,  2.312599,  2.337241,  2.363856,  2.385313/
      DATA (REREF(I), I = 901, 990 )/
     &  2.399111,  2.436760,  2.481153,  2.527536,  2.577344,  2.629097,
     &  2.679108,  2.729264,  2.781861,  2.831974,  2.881863,  2.933900,
     &  2.983258,  3.032401,  3.084049,  3.133464,  3.179887,  3.228984,
     &  3.279470,  3.326631,  3.374610,  3.422465,  3.468221,  3.516889,
     &  3.563346,  3.607096,  3.650102,  3.695213,  3.741930,  3.785136,
     &  3.829496,  3.873564,  3.917021,  3.960586,  4.003601,  4.045111,
     &  4.084851,  4.125763,  4.166540,  4.207585,  4.248425,  4.288766,
     &  4.328263,  4.367201,  4.403706,  4.442443,  4.482558,  4.518750,
     &  4.555811,  4.593558,  4.631138,  4.667698,  4.704528,  4.740422,
     &  4.776470,  4.811400,  4.844068,  4.881015,  4.915201,  4.946351,
     &  4.979800,  5.013994,  5.047771,  5.083439,  5.116001,  5.146161,
     &  5.177179,  5.209531,  5.241539,  5.273172,  5.304929,  5.336323,
     &  5.367389,  5.398286,  5.428878,  5.459208,  5.489262,  5.519027,
     &  5.548489,  5.577699,  5.606586,  5.635201,  5.663535,  5.691521,
     &  5.719272,  5.746661,  5.773802,  5.800631,  5.827179,  5.853423/
      DATA (REREF(I), I = 991, 1080 )/
     &  5.879378,  6.131865,  6.346035,  6.538143,  6.711149,  6.867192,
     &  7.007965,  7.135674,  7.252419,  7.358822,  7.455943,  7.544423,
     &  7.625553,  7.701126,  7.768902,  7.831158,  7.889643,  7.941322,
     &  7.989355,  8.033791,  8.074469,  8.112180,  8.147128,  8.179843,
     &  8.209818,  8.238281,  8.264599,  8.288448,  8.311297,  8.332788,
     &  8.352700,  8.371979,  8.389151,  8.405156,  8.420858,  8.435660,
     &  8.449912,  8.462521,  8.474673,  8.486470,  8.497290,  8.507814,
     &  8.517490,  8.526995,  8.535947,  8.545560,  8.554652,  8.563363,
     &  8.571605,  8.579613,  8.587191,  8.594362,  8.601342,  8.607948,
     &  8.614382,  8.620473,  8.626323,  8.632030,  8.637431,  8.642623,
     &  8.647693,  8.652568,  8.657181,  8.661691,  8.666029,  8.670273,
     &  8.674287,  8.678217,  8.681999,  8.685638,  8.689204,  8.692637,
     &  8.695941,  8.699181,  8.702404,  8.705507,  8.708553,  8.711486,
     &  8.714366,  8.717139,  8.719808,  8.722432,  8.725011,  8.727494,
     &  8.729884,  8.732236,  8.734549,  8.736776,  8.738919,  8.741029/
      DATA (REREF(I), I = 1081, 1170 )/
     &  8.743107,  8.761392,  8.776133,  8.788091,  8.797836,  8.804337,
     &  8.809677,  8.814082,  8.817867,  8.821129,  8.823834,  8.826184,
     &  8.828230,  8.830103,  8.831656,  8.833091,  8.834348,  8.835449,
     &  8.836475,  8.837375,  8.838141,  8.838809,  8.839434,  8.840020,
     &  8.840491,  8.840974,  8.841394,  8.841790,  8.842166,  8.842491,
     &  8.842801,  8.843095,  8.843349,  8.843591,  8.843823,  8.844044,
     &  8.844254,  8.844436,  8.844609,  8.844776,  8.844936,  8.845089,
     &  8.845220,  8.845345,  8.845467,  8.845583,  8.845696,  8.845804,
     &  8.845896,  8.845997,  8.846082,  8.846164,  8.846244,  8.846321,
     &  8.846396,  8.846468,  8.846538,  8.846596,  8.846662,  8.846716,
     &  8.846769,  8.846821,  8.846871,  8.846920,  8.846967,  8.847013,
     &  8.847058,  8.847094,  8.847137,  8.847178,  8.847212,  8.847244,
     &  8.847283,  8.847314,  8.847344,  8.847374,  8.847403,  8.847437,
     &  8.847459,  8.847485,  8.847512,  8.847537,  8.847563,  8.847582,
     &  8.847606,  8.847630,  8.847648,  8.847671,  8.847688,  8.847710/
      DATA (REREF(I), I = 1171, 1261 )/
     &  8.847727,  8.847889,  8.848013,  8.848109,  8.848183,  8.848243,
     &  8.848291,  8.848330,  8.848364,  8.848392,  8.848415,  8.848435,
     &  8.848452,  8.848467,  8.848480,  8.848491,  8.848501,  8.848510,
     &  8.848518,  8.848524,  8.848531,  8.848536,  8.848541,  8.848546,
     &  8.848550,  8.848554,  8.848557,  8.848560,  8.848563,  8.848566,
     &  8.848568,  8.848570,  8.848572,  8.848574,  8.848576,  8.848577,
     &  8.848579,  8.848580,  8.848581,  8.848582,  8.848584,  8.848585,
     &  8.848585,  8.848586,  8.848587,  8.848588,  8.848589,  8.848589,
     &  8.848590,  8.848591,  8.848591,  8.848592,  8.848592,  8.848593,
     &  8.848593,  8.848594,  8.848594,  8.848594,  8.848595,  8.848595,
     &  8.848595,  8.848596,  8.848596,  8.848596,  8.848596,  8.848597,
     &  8.848597,  8.848597,  8.848597,  8.848598,  8.848598,  8.848598,
     &  8.848598,  8.848598,  8.848598,  8.848598,  8.848599,  8.848599,
     &  8.848599,  8.848599,  8.848599,  8.848599,  8.848599,  8.848599,
     &  8.848600,  8.848600,  8.848600,  8.848600,  8.848600,  8.848600,
     &  8.848600/

c                                    ** imaginary refractive indices
      DATA (IMREF(I), I = 1, 90 ) /
     &.17450E-02,.23702E-02,.31462E-02,.40718E-02,.51736E-02,.79577E-02,
     &.11636E-01,.16360E-01,.22274E-01,.29498E-01,.38176E-01,.48505E-01,
     &.60644E-01,.74609E-01,.90738E-01,.10934E+00,.13025E+00,.15339E+00,
     &.17980E+00,.20883E+00,.24143E+00,.27656E+00,.29977E+00,.31535E+00,
     &.33097E+00,.34978E+00,.37393E+00,.41190E+00,.45582E+00,.50326E+00,
     &.53554E+00,.56337E+00,.57915E+00,.58586E+00,.58048E+00,.58586E+00,
     &.59812E+00,.61347E+00,.62920E+00,.64534E+00,.65734E+00,.65734E+00,
     &.65282E+00,.64386E+00,.62920E+00,.60505E+00,.57516E+00,.54299E+00,
     &.51855E+00,.49294E+00,.47075E+00,.44853E+00,.43032E+00,.41475E+00,
     &.39883E+00,.38264E+00,.37050E+00,.35958E+00,.34897E+00,.33868E+00,
     &.32195E+00,.28760E+00,.23922E+00,.18698E+00,.14886E+00,.13329E+00,
     &.14216E+00,.16780E+00,.19266E+00,.21667E+00,.24087E+00,.26411E+00,
     &.27720E+00,.27720E+00,.25810E+00,.23377E+00,.20220E+00,.16703E+00,
     &.13514E+00,.10394E+00,.72409E-01,.39975E-01,.39975E-02,.20035E-02,
     &.11825E-02,.83905E-03,.59950E-04,.12495E-05,.36223E-06,.18497E-06/
      DATA (IMREF(I), I = 91, 180 ) /
     &.11010E-06,.67110E-07,.38441E-07,.19989E-07,.12700E-07,.11582E-07,
     &.11010E-07,.10710E-07,.10490E-07,.99035E-08,.93066E-08,.86058E-08,
     &.79945E-08,.74437E-08,.68516E-08,.62920E-08,.57915E-08,.54049E-08,
     &.47950E-08,.44034E-08,.41475E-08,.38264E-08,.35464E-08,.33250E-08,
     &.31900E-08,.30817E-08,.29839E-08,.28826E-08,.27656E-08,.26533E-08,
     &.25281E-08,.24199E-08,.23163E-08,.22171E-08,.21173E-08,.20314E-08,
     &.19399E-08,.18399E-08,.17611E-08,.16626E-08,.15805E-08,.14886E-08,
     &.14216E-08,.13390E-08,.12583E-08,.11689E-08,.10884E-08,.10181E-08,
     &.93927E-09,.86854E-09,.80871E-09,.77945E-09,.75996E-09,.74953E-09,
     &.72911E-09,.70112E-09,.70923E-09,.71580E-09,.73416E-09,.78486E-09,
     &.92425E-09,.10784E-08,.12670E-08,.14615E-08,.15696E-08,.16398E-08,
     &.17571E-08,.18871E-08,.20979E-08,.22688E-08,.24423E-08,.26594E-08,
     &.28694E-08,.31318E-08,.34339E-08,.38441E-08,.44339E-08,.52214E-08,
     &.63649E-08,.77231E-08,.96336E-08,.11319E-07,.12382E-07,.13298E-07,
     &.13989E-07,.14716E-07,.15024E-07,.15516E-07,.15696E-07,.16062E-07/
      DATA (IMREF(I), I = 181, 270 ) /
     &.16741E-07,.17774E-07,.19399E-07,.20314E-07,.20979E-07,.21767E-07,
     &.23003E-07,.24705E-07,.26533E-07,.29634E-07,.33480E-07,.41001E-07,
     &.49979E-07,.59950E-07,.72911E-07,.91367E-07,.11502E-06,.13483E-06,
     &.14581E-06,.15303E-06,.15588E-06,.15805E-06,.15805E-06,.15696E-06,
     &.15268E-06,.14784E-06,.14086E-06,.13390E-06,.12817E-06,.12583E-06,
     &.12497E-06,.12700E-06,.13298E-06,.14481E-06,.16210E-06,.18188E-06,
     &.20408E-06,.22428E-06,.24592E-06,.26902E-06,.29295E-06,.31535E-06,
     &.33480E-06,.35464E-06,.37479E-06,.39065E-06,.40531E-06,.42344E-06,
     &.44034E-06,.46216E-06,.48617E-06,.51498E-06,.56989E-06,.66956E-06,
     &.83041E-06,.10597E-05,.13683E-05,.17709E-05,.21692E-05,.25573E-05,
     &.29322E-05,.31900E-05,.33581E-05,.34641E-05,.35018E-05,.34801E-05,
     &.34181E-05,.33365E-05,.32530E-05,.31311E-05,.29998E-05,.26878E-05,
     &.23523E-05,.20007E-05,.16900E-05,.14194E-05,.12989E-05,.12589E-05,
     &.13286E-05,.14986E-05,.17084E-05,.20384E-05,.26284E-05,.38689E-05,
     &.59510E-05,.93065E-05,.10689E-04,.11195E-04,.11603E-04,.11812E-04/
      DATA (IMREF(I), I = 271, 360 ) /
     &.11995E-04,.11906E-04,.11790E-04,.11602E-04,.11389E-04,.11002E-04,
     &.10791E-04,.10905E-04,.11394E-04,.12212E-04,.14002E-04,.16390E-04,
     &.19120E-04,.22506E-04,.28490E-04,.40466E-04,.45050E-04,.58035E-04,
     &.78017E-04,.10600E-03,.15303E-03,.25397E-03,.31973E-03,.35383E-03,
     &.36290E-03,.36374E-03,.36040E-03,.33868E-03,.30185E-03,.26594E-03,
     &.22480E-03,.19579E-03,.17410E-03,.16025E-03,.14414E-03,.13483E-03,
     &.12396E-03,.11402E-03,.10708E-03,.99401E-04,.93474E-04,.88042E-04,
     &.83098E-04,.80964E-04,.79030E-04,.75908E-04,.73976E-04,.74044E-04,
     &.74953E-04,.76013E-04,.77427E-04,.80499E-04,.84099E-04,.89000E-04,
     &.95102E-04,.99998E-04,.10512E-03,.11202E-03,.12187E-03,.13304E-03,
     &.13592E-03,.13712E-03,.13797E-03,.14184E-03,.15516E-03,.18612E-03,
     &.32047E-03,.52094E-03,.72242E-03,.92213E-03,.11609E-02,.16780E-02,
     &.18272E-02,.19222E-02,.19089E-02,.18484E-02,.17171E-02,.15481E-02,
     &.14021E-02,.12497E-02,.11010E-02,.99035E-03,.88877E-03,.80499E-03,
     &.73925E-03,.67420E-03,.62057E-03,.57252E-03,.52941E-03,.48842E-03/
      DATA (IMREF(I), I = 361, 450 ) /
     &.46429E-03,.44034E-03,.41763E-03,.39700E-03,.38264E-03,.37050E-03,
     &.35875E-03,.35058E-03,.34339E-03,.33946E-03,.33790E-03,.33868E-03,
     &.34103E-03,.34577E-03,.35710E-03,.37393E-03,.38975E-03,.40812E-03,
     &.42933E-03,.45060E-03,.46859E-03,.49180E-03,.51143E-03,.54299E-03,
     &.59950E-03,.63649E-03,.68516E-03,.74266E-03,.79212E-03,.84877E-03,
     &.90947E-03,.99035E-03,.10710E-02,.11502E-02,.12497E-02,.13483E-02,
     &.14716E-02,.15805E-02,.17092E-02,.18105E-02,.19002E-02,.19534E-02,
     &.19897E-02,.20174E-02,.20691E-02,.21419E-02,.22688E-02,.23109E-02,
     &.23377E-02,.23867E-02,.24254E-02,.24762E-02,.25751E-02,.27027E-02,
     &.29771E-02,.33021E-02,.40160E-02,.43630E-02,.48283E-02,.53677E-02,
     &.62775E-02,.73247E-02,.85465E-02,.10490E-01,.12700E-01,.14514E-01,
     &.16398E-01,.18612E-01,.20502E-01,.28170E-01,.38001E-01,.46216E-01,
     &.54801E-01,.64832E-01,.74437E-01,.83520E-01,.92852E-01,.10204E+00,
     &.11189E+00,.12100E+00,.13116E+00,.14216E+00,.15410E+00,.16703E+00,
     &.17980E+00,.19399E+00,.20596E+00,.21817E+00,.22898E+00,.23922E+00/
      DATA (IMREF(I), I = 451, 540 ) /
     &.24934E+00,.25810E+00,.26472E+00,.27151E+00,.27592E+00,.27976E+00,
     &.28041E+00,.28235E+00,.28170E+00,.27848E+00,.27592E+00,.27214E+00,
     &.27214E+00,.23977E+00,.19177E+00,.13483E+00,.92425E-01,.61065E-01,
     &.36880E-01,.26109E-01,.19489E-01,.13207E-01,.93927E-02,.67888E-02,
     &.51498E-02,.42344E-02,.35958E-02,.34024E-02,.34024E-02,.35301E-02,
     &.38001E-02,.41571E-02,.46003E-02,.50675E-02,.56207E-02,.62200E-02,
     &.68832E-02,.75996E-02,.84487E-02,.93066E-02,.10299E-01,.11397E-01,
     &.12382E-01,.13608E-01,.14716E-01,.15481E-01,.15696E-01,.15516E-01,
     &.14990E-01,.14414E-01,.13702E-01,.13116E-01,.12411E-01,.11798E-01,
     &.11112E-01,.10612E-01,.10111E-01,.99035E-02,.97902E-02,.98808E-02,
     &.10299E-01,.10784E-01,.11582E-01,.12583E-01,.14184E-01,.16588E-01,
     &.20314E-01,.24819E-01,.32945E-01,.43230E-01,.62200E-01,.86455E-01,
     &.10685E+00,.12497E+00,.13085E+00,.11716E+00,.87860E-01,.69469E-01,
     &.56989E-01,.49521E-01,.44853E-01,.41763E-01,.39246E-01,.37307E-01,
     &.35628E-01,.34498E-01,.33712E-01,.33097E-01,.32718E-01,.32418E-01/
      DATA (IMREF(I), I = 541, 630 ) /
     &.32195E-01,.32121E-01,.31973E-01,.31900E-01,.31973E-01,.32047E-01,
     &.32047E-01,.32195E-01,.32195E-01,.32269E-01,.32418E-01,.32493E-01,
     &.32568E-01,.32718E-01,.32794E-01,.33021E-01,.33097E-01,.33250E-01,
     &.33480E-01,.33712E-01,.33946E-01,.34103E-01,.34260E-01,.34498E-01,
     &.34657E-01,.34897E-01,.35139E-01,.35301E-01,.35628E-01,.35792E-01,
     &.36040E-01,.36374E-01,.36542E-01,.36880E-01,.37136E-01,.37479E-01,
     &.37826E-01,.38088E-01,.38441E-01,.38796E-01,.39155E-01,.39518E-01,
     &.39883E-01,.40345E-01,.40718E-01,.41095E-01,.41475E-01,.41956E-01,
     &.42344E-01,.42933E-01,.43330E-01,.43731E-01,.44339E-01,.44750E-01,
     &.45372E-01,.46003E-01,.46643E-01,.47183E-01,.47840E-01,.48505E-01,
     &.49294E-01,.49979E-01,.50791E-01,.51736E-01,.52698E-01,.53801E-01,
     &.58048E-01,.56337E-01,.58451E-01,.59950E-01,.61914E-01,.63943E-01,
     &.66190E-01,.68516E-01,.70923E-01,.73585E-01,.76523E-01,.79577E-01,
     &.82945E-01,.86455E-01,.89700E-01,.93280E-01,.96781E-01,.99952E-01,
     &.10394E+00,.10834E+00,.11293E+00,.11716E+00,.12184E+00,.12700E+00/
      DATA (IMREF(I), I = 631, 720 ) /
     &.13207E+00,.13702E+00,.14216E+00,.14716E+00,.15198E+00,.15696E+00,
     &.16210E+00,.16780E+00,.17410E+00,.18021E+00,.18655E+00,.19266E+00,
     &.19897E+00,.20549E+00,.21125E+00,.21767E+00,.22376E+00,.22950E+00,
     &.23593E+00,.24199E+00,.24762E+00,.25281E+00,.25929E+00,.26411E+00,
     &.26902E+00,.27402E+00,.27912E+00,.28365E+00,.28826E+00,.29295E+00,
     &.29771E+00,.30116E+00,.30605E+00,.31031E+00,.31390E+00,.31827E+00,
     &.32269E+00,.32568E+00,.32945E+00,.33250E+00,.33635E+00,.34024E+00,
     &.34260E+00,.34657E+00,.34897E+00,.35139E+00,.35464E+00,.35710E+00,
     &.35958E+00,.36207E+00,.36458E+00,.36795E+00,.36965E+00,.37221E+00,
     &.37393E+00,.37566E+00,.37739E+00,.37913E+00,.38088E+00,.38264E+00,
     &.38441E+00,.38618E+00,.38707E+00,.38975E+00,.39155E+00,.39336E+00,
     &.39427E+00,.39609E+00,.39700E+00,.39883E+00,.39975E+00,.40160E+00,
     &.40252E+00,.40438E+00,.40531E+00,.40531E+00,.40718E+00,.40812E+00,
     &.41001E+00,.41001E+00,.41190E+00,.41285E+00,.41380E+00,.41475E+00,
     &.41571E+00,.41763E+00,.41763E+00,.41859E+00,.41956E+00,.42052E+00/
      DATA (IMREF(I), I = 721, 810 ) /
     &.42052E+00,.42246E+00,.42246E+00,.42246E+00,.42344E+00,.42441E+00,
     &.42539E+00,.42539E+00,.42539E+00,.42637E+00,.42637E+00,.42736E+00,
     &.42736E+00,.42834E+00,.42834E+00,.42834E+00,.42933E+00,.42933E+00,
     &.42933E+00,.43032E+00,.42933E+00,.42933E+00,.42933E+00,.43032E+00,
     &.43032E+00,.42933E+00,.43032E+00,.43032E+00,.42933E+00,.42933E+00,
     &.42933E+00,.42933E+00,.42834E+00,.42834E+00,.42834E+00,.42834E+00,
     &.42736E+00,.42736E+00,.42637E+00,.42736E+00,.42637E+00,.42637E+00,
     &.42539E+00,.42539E+00,.42539E+00,.42539E+00,.42441E+00,.42441E+00,
     &.42344E+00,.42344E+00,.42344E+00,.42246E+00,.42149E+00,.42052E+00,
     &.42052E+00,.41956E+00,.41859E+00,.41763E+00,.41763E+00,.41667E+00,
     &.41571E+00,.41571E+00,.41380E+00,.41285E+00,.41190E+00,.41095E+00,
     &.41001E+00,.40906E+00,.40906E+00,.40812E+00,.40718E+00,.40625E+00,
     &.40438E+00,.40345E+00,.40252E+00,.40160E+00,.40067E+00,.40067E+00,
     &.39883E+00,.39700E+00,.39518E+00,.39427E+00,.39336E+00,.38176E+00,
     &.37221E+00,.36290E+00,.34817E+00,.33558E+00,.32269E+00,.31031E+00/
      DATA (IMREF(I), I = 811, 900 ) /
     &.29908E+00,.28893E+00,.28170E+00,.27912E+00,.27976E+00,.28300E+00,
     &.28628E+00,.29160E+00,.29908E+00,.30675E+00,.31900E+00,.33173E+00,
     &.34418E+00,.35875E+00,.37393E+00,.38796E+00,.40252E+00,.41763E+00,
     &.43230E+00,.44647E+00,.45792E+00,.46751E+00,.47730E+00,.49750E+00,
     &.50791E+00,.51617E+00,.52335E+00,.52576E+00,.52455E+00,.52455E+00,
     &.52335E+00,.52094E+00,.51736E+00,.51380E+00,.51026E+00,.50791E+00,
     &.50210E+00,.49635E+00,.49067E+00,.48394E+00,.47730E+00,.47183E+00,
     &.46536E+00,.46003E+00,.45477E+00,.44853E+00,.44339E+00,.43832E+00,
     &.41763E+00,.41667E+00,.41859E+00,.42052E+00,.42637E+00,.43330E+00,
     &.44034E+00,.45060E+00,.46109E+00,.46967E+00,.47840E+00,.48729E+00,
     &.49635E+00,.50558E+00,.51380E+00,.52094E+00,.52819E+00,.53554E+00,
     &.54299E+00,.55054E+00,.56597E+00,.58048E+00,.59675E+00,.61347E+00,
     &.62920E+00,.64534E+00,.65886E+00,.67265E+00,.68516E+00,.69950E+00,
     &.71251E+00,.72409E+00,.73585E+00,.74781E+00,.76171E+00,.79212E+00,
     &.83136E+00,.87255E+00,.91157E+00,.95015E+00,.98354E+00,.10111E+01/
      DATA (IMREF(I), I = 901, 990 ) /
     &.10418E+01,.11199E+01,.11912E+01,.12609E+01,.13301E+01,.13928E+01,
     &.14517E+01,.15107E+01,.15671E+01,.16166E+01,.16703E+01,.17179E+01,
     &.17627E+01,.18092E+01,.18535E+01,.18910E+01,.19310E+01,.19733E+01,
     &.20090E+01,.20426E+01,.20792E+01,.21096E+01,.21423E+01,.21742E+01,
     &.21993E+01,.22279E+01,.22547E+01,.22855E+01,.23099E+01,.23329E+01,
     &.23599E+01,.23801E+01,.24043E+01,.24226E+01,.24451E+01,.24603E+01,
     &.24808E+01,.24997E+01,.25176E+01,.25351E+01,.25515E+01,.25668E+01,
     &.25810E+01,.25947E+01,.26073E+01,.26254E+01,.26363E+01,.26466E+01,
     &.26619E+01,.26711E+01,.26853E+01,.26921E+01,.27051E+01,.27108E+01,
     &.27226E+01,.27270E+01,.27371E+01,.27472E+01,.27497E+01,.27580E+01,
     &.27662E+01,.27739E+01,.27803E+01,.27867E+01,.27861E+01,.27918E+01,
     &.27963E+01,.28009E+01,.28047E+01,.28080E+01,.28111E+01,.28137E+01,
     &.28158E+01,.28240E+01,.28253E+01,.28262E+01,.28268E+01,.28269E+01,
     &.28332E+01,.28326E+01,.28316E+01,.28303E+01,.28352E+01,.28332E+01,
     &.28309E+01,.28347E+01,.28318E+01,.28350E+01,.28315E+01,.28276E+01/
      DATA (IMREF(I), I = 991, 1080 ) /
     &.28300E+01,.28066E+01,.27731E+01,.27333E+01,.26847E+01,.26296E+01,
     &.25757E+01,.25188E+01,.24609E+01,.23966E+01,.23377E+01,.22803E+01,
     &.22243E+01,.21702E+01,.21134E+01,.20639E+01,.20109E+01,.19611E+01,
     &.19142E+01,.18681E+01,.18239E+01,.17811E+01,.17410E+01,.17009E+01,
     &.16630E+01,.16270E+01,.15911E+01,.15570E+01,.15251E+01,.14931E+01,
     &.14632E+01,.14341E+01,.14047E+01,.13781E+01,.13520E+01,.13271E+01,
     &.13022E+01,.12779E+01,.12560E+01,.12331E+01,.12120E+01,.11912E+01,
     &.11711E+01,.11521E+01,.11329E+01,.11150E+01,.10970E+01,.10802E+01,
     &.10629E+01,.10471E+01,.10311E+01,.10160E+01,.10011E+01,.98671E+00,
     &.97273E+00,.95894E+00,.94556E+00,.93280E+00,.92022E+00,.90780E+00,
     &.89596E+00,.88448E+00,.87275E+00,.86177E+00,.85092E+00,.84060E+00,
     &.83021E+00,.82014E+00,.81038E+00,.80074E+00,.79143E+00,.78222E+00,
     &.77327E+00,.76454E+00,.75603E+00,.74757E+00,.73940E+00,.73138E+00,
     &.72364E+00,.71592E+00,.70833E+00,.70094E+00,.69383E+00,.68672E+00,
     &.67975E+00,.67296E+00,.66635E+00,.65984E+00,.65336E+00,.64705E+00/
      DATA (IMREF(I), I = 1081, 1170 ) /
     &.64090E+00,.58451E+00,.53801E+00,.49830E+00,.46365E+00,.43370E+00,
     &.40728E+00,.38370E+00,.36299E+00,.34482E+00,.32771E+00,.31231E+00,
     &.29833E+00,.28595E+00,.27409E+00,.26345E+00,.25351E+00,.24411E+00,
     &.23561E+00,.22740E+00,.22019E+00,.21320E+00,.20644E+00,.20081E+00,
     &.19444E+00,.18914E+00,.18399E+00,.17897E+00,.17450E+00,.17013E+00,
     &.16588E+00,.16210E+00,.15805E+00,.15445E+00,.15094E+00,.14784E+00,
     &.14481E+00,.14151E+00,.13861E+00,.13577E+00,.13329E+00,.13085E+00,
     &.12817E+00,.12583E+00,.12325E+00,.12128E+00,.11907E+00,.11716E+00,
     &.11502E+00,.11319E+00,.11138E+00,.10934E+00,.10759E+00,.10612E+00,
     &.10442E+00,.10299E+00,.10134E+00,.99952E-01,.98580E-01,.97004E-01,
     &.95673E-01,.94360E-01,.92852E-01,.91789E-01,.90530E-01,.89287E-01,
     &.88265E-01,.87054E-01,.85860E-01,.84877E-01,.83905E-01,.82754E-01,
     &.81807E-01,.80871E-01,.79761E-01,.78848E-01,.77945E-01,.77231E-01,
     &.76171E-01,.75473E-01,.74609E-01,.73755E-01,.73079E-01,.72242E-01,
     &.71415E-01,.70760E-01,.69950E-01,.69309E-01,.68516E-01,.68044E-01/
      DATA (IMREF(I), I = 1171, 1261 ) /
     &.67265E-01,.61205E-01,.56207E-01,.51974E-01,.48283E-01,.45060E-01,
     &.42344E-01,.39791E-01,.37652E-01,.35710E-01,.33946E-01,.32344E-01,
     &.30888E-01,.29566E-01,.28365E-01,.27214E-01,.26230E-01,.25223E-01,
     &.24366E-01,.23485E-01,.22740E-01,.22019E-01,.21320E-01,.20739E-01,
     &.20081E-01,.19534E-01,.19002E-01,.18484E-01,.18021E-01,.17571E-01,
     &.17131E-01,.16703E-01,.16323E-01,.15914E-01,.15588E-01,.15233E-01,
     &.14921E-01,.14581E-01,.14282E-01,.13989E-01,.13734E-01,.13483E-01,
     &.13207E-01,.12966E-01,.12729E-01,.12497E-01,.12268E-01,.12072E-01,
     &.11852E-01,.11662E-01,.11476E-01,.11293E-01,.11112E-01,.10909E-01,
     &.10759E-01,.10612E-01,.10442E-01,.10299E-01,.10158E-01,.99952E-02,
     &.98580E-02,.97004E-02,.95673E-02,.94578E-02,.93280E-02,.92000E-02,
     &.90947E-02,.89493E-02,.88469E-02,.87456E-02,.86256E-02,.85269E-02,
     &.84293E-02,.83328E-02,.82185E-02,.81244E-02,.80314E-02,.79577E-02,
     &.78486E-02,.77587E-02,.76876E-02,.75996E-02,.75299E-02,.74437E-02,
     &.73585E-02,.72911E-02,.72076E-02,.71415E-02,.70598E-02,.70112E-02,
     &.69309E-02/
c     ..

      END

