function [uth,tb12]=hirs_tb2uth(tb12,viewangle,satellite, icewater)
% AMSUB_TB2UTH calculates UTH from AMSU-B channel 18 Tb
%
% PURPOSE   The function  is based on the scaling method presented in Soden
% and Bretherton (1993)
%           et., al. where ln(UTH)= a +b*Tb
%
%  IN
%         tb12                      Brightness temperature for HIRS channel
%                                   12 (6.5 micron)
%         viewangle                 viewing angle (off nadir) under which the
%                                   instruments measures (0 = nadir).
%                                   Scalar, or vector the same length as
%                                   size(tb12,1) or size(tb12)==size(viewangle) 
%
%         satellite                 'n18';
%
%         icewater (optional flag)  Set 'w' for UTH with respece to water,
%                                   'i' for UTH with respect to ice. If
%                                   nothing is given, provide UTH with
%                                   respect to water.
%
%  OUT    uth                       calculated by the scaling method
%                                   presented by Soden and Bretherton (1993):
%         tb12                      filtered version of tb12 (values that
%                                   were below valid min are flagged as NaN) 
%
%
% USAGE
%         uth=hirs_tb2uth(tb12,viewangle,icewater)
%
% $Id: hirs_tb2uth.m 7954 2012-10-28 20:57:41Z seliasson $
% Salomon Eliasson

errId = ['atmlab:' mfilename 'badInput'];
assert(nargin>=2,errId,'At least 2 argumnents required!')
viewangle = viewangle(:); % make sure it's always a rowvector
dimtest =  ismember(size(tb12),size(viewangle,1));
if ~dimtest(1) && dimtest(2)
    tb12 = tb12'; %then flip it
    flp = true;
else flp=false;
end
assert(isequal(size(tb12),size(viewangle)) | ...
    isequal(size(tb12,1),size(viewangle,1)) | ...
    isscalar(viewangle),errId,'sizes of ''tb12'' and ''viewangle'' don''t match')
if nargin < 4
    icewater='i';
end
if isscalar(viewangle)
    viewangle = viewangle*ones(size(tb12));
end

% From Ajil; regression coefficients for NOAA18 HIRS using Chevallier profiles (tropics)

switch satellite
    case {'noaa18','n18'}
        % ICE
        aw=[30.6670652066303          30.4941535481248          30.3491511617953  ...
            30.2249532391769          30.1178684495338          30.0242376100305  ...
            29.9415866583268          29.868121667893           29.802467028195   ...
            29.7435479334073          29.6905114959292          29.6426719111223  ...
            29.5994714548024          29.5604521816226          29.5252350258211  ...
            29.4935041295309          29.4649949321133          29.4394850116124  ...
            29.4167869717815          29.3967428720245          29.3792198376325  ...
            29.364106585575           29.3513106706515          29.340756307      ...
            29.3323826568268          29.3261425057273          29.3220012649637  ...
            29.3199362574556];
        
        bw=[-0.117945574768725        -0.116916476646109        -0.116044448589813  ...
            -0.115291444098681        -0.114635820335841        -0.114058082210159  ...
            -0.113544656112139        -0.11308553588133         -0.112673000894565  ...
            -0.112300983606722        -0.11196464225324         -0.111660062612903  ...
            -0.111384045047004        -0.111133949380345        -0.110907579932633  ...
            -0.110703098999673        -0.110518960870733        -0.110353860922478  ...
            -0.110206695955494        -0.110076533038635        -0.10996258488327   ...
            -0.109864190299876        -0.10978079866719         -0.109711957617433  ...
            -0.109657303342444        -0.109616553076046        -0.109589499423281  ...
            -0.109576006297241];
        
        validMinW = [224.230939351794           224.70770021556          225.131959520495    ...
            225.514026172193          225.86043114089           226.176318756936    ...
            226.465618592359          226.731396362438          226.969182292803    ...
            227.177083226468          227.36861207396           227.54502501285    ...
            227.701232342825          227.842754256693          227.972137825972    ...
            228.090063287101          228.197111881001          228.293781304778    ...
            228.380497958684          228.457626720557          228.525478786376    ...
            228.584317977423          228.634365814508          228.675805585945    ...
            228.708785580931          228.733421617924          228.749798965027    ...
            228.757973723394];

        
        
        % WATER
        ai=[31.2164076863505          31.0433433005347          30.8983581689751    ...
            30.7743013041315          30.6674531209836          30.5741338977246    ...
            30.4918522814653          30.4188008621987          30.3535925980219    ...
            30.2951428189602          30.2425900034245          30.1952407225327    ...
            30.1525304842479          30.1139953187109          30.0792507914453    ...
            30.047976261242          30.0199029112708           29.994804541538     ...
            29.9724904142409          29.9527996480935          29.9355967981028    ...
            29.9207683553834           29.908219971298          29.8978742605019    ...
            29.8896690744189          29.8835561642443          29.8795001736228    ...
            29.8774779175842];
        
        bi=[-0.11830019995808       -0.117290095211986        -0.116435410595745    ...
          -0.115698362110187        -0.115057515309168        -0.114493532234087    ...
          -0.113992951830069        -0.113545855656916        -0.113144589717254    ...
          -0.11278313753744         -0.112456696739265        -0.112161383655798    ...
          -0.111894022484969        -0.111651991746319        -0.111433110493521    ...
          -0.111235552686084        -0.111057781882035        -0.110898500846861    ...
          -0.110756612284897        -0.110631187987917        -0.110521444444766    ...
          -0.110426723480616        -0.110346476868051        -0.110280254122412    ...
          -0.110227692892863        -0.110188511509451        -0.110162503360374    ...
          -0.110149532862807];
      
      validMinI = [224.230939351794           224.70770021556          225.131959520495 ...
          225.514026172193           225.86043114089          226.176318756936  ...
          226.465618592359          226.731396362438          226.969182292803  ...
          227.177083226468           227.36861207396           227.54502501285  ...
          227.701232342825          227.842754256693          227.972137825972  ...
          228.090063287101          228.197111881001          228.293781304778  ...
          228.380497958684          228.457626720557          228.525478786376  ...
          228.584317977423          228.634365814508          228.675805585945  ...
          228.708785580931          228.733421617924          228.749798965027  ...
          228.757973723394];
        
    otherwise
        error(['atmlab:' mfilename ':NotCofigured'],'Nothing is configured for satellite="%s"',satellite)
end

% corresponding angles
ang=[                 49.5                      47.7                      45.9  ...
                      44.1                      42.3                      40.5  ...
                      38.7                      36.9                      35.1  ...
                      33.3                      31.5                      29.7  ...
                      27.9                      26.1                      24.3  ...
                      22.5                      20.7                      18.9  ...
                      17.1                      15.3                      13.5  ...
                      11.7                       9.9                       8.1  ...
                       6.3                       4.5                       2.7  ...
                       0.9];


%  Conversion function
TB2UTH = @(a,b,Tb) exp( a + b*Tb);

if strcmp(icewater,'i')
    [A,B,V]    = deal(ai,bi,validMinI);
elseif strcmp(icewater,'w')
    [A,B,V]    = deal(ai,bi,validMinW);
else
    error(['atmlab:' mfilename ':BadInput'],'unknown water phase "%s"',icewater)
end


uth = zeros(size(tb12));
% LOOP over the viewing angles
for k = 1:length(viewangle)
    % Find the CLOSET ANGLE in the lookup table to the angles given as input.
    % Pick closest_angle(1) since sometimes there are two answers. 1e-3 for
    % precision error
    ind=find(abs(ang-viewangle(k)) - min(abs(ang-viewangle(k))) < 1e-3);
    [a,b,Tb]    = deal(A(ind(1)),B(ind(1)),tb12(k,:));
    uth(k,:)    = TB2UTH(a,b,Tb);
    
    if isvector(uth)
        test = tb12(k)<V(ind);
        if test
            uth(k) = -9999; % flag stuff outside the range the regression was made
        end
    else
        uth(k,tb12(k,:)<V(ind)) = -9999; % flag stuff outside the range the regression was made
    end
end

if flp, uth=uth'; end % put back in original orientation